<?php

namespace LightrCommerce\Plugin;

use LightrCommerce\BigCommerce\Helper as BigCommerceHelper;

class Wizard {

    static function testPermanentUrl() {
        check_admin_referer('test_bigcommerce_permanent_url');
        check_ajax_referer('test_bigcommerce_permanent_url');

        $url = filter_input(INPUT_POST, 'store_permanent_url', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        $url = rtrim($url, '/');
        $url = strtolower($url);

        if (BigCommerceHelper::isValidPermanentUrl($url)) {
            $graphqlUrl = $url . '/graphql';

            if (BigCommerceHelper::isValidGraphQlUrl($graphqlUrl)) {
                $response = [
                    'ok' => true,
                    'permanentUrl' => $url
                ];
            } else {
                $response = [
                    'ok' => false,
                    'error' => esc_html__('We have not found a BigCommerce store in the provided URL.', 'lightrcommerce')
                ];
            }
        } else {
            $response = [
                'ok' => false,
                'error' => esc_html__('The provided permanent store address seems invalid.', 'lightrcommerce')
            ];
        }

        wp_send_json($response);
        wp_die();
    }

    static function testAndSaveSetup() {
        check_admin_referer('lightrcommerce_setup');
        check_ajax_referer('lightrcommerce_setup');

        $permanentUrl = filter_input(INPUT_POST, 'lightrcommerce_store_permanent_url', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        $clientId = filter_input(INPUT_POST, 'lightrcommerce_client_id', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        $accessToken = filter_input(INPUT_POST, 'lightrcommerce_access_token', FILTER_SANITIZE_FULL_SPECIAL_CHARS);

        try {
            // check credentials validity
            $authToken = self::_getAuthTokenFromSetup($permanentUrl, $clientId, $accessToken);

            // save setup options
            $middleware = self::_createMiddleware($permanentUrl, $clientId, $accessToken);

            update_option('lightrcommerce_bc_endpoint_id', $middleware->id);
            update_option('lightrcommerce_bc_endpoint_name', $middleware->name);
            update_option('lightrcommerce_bc_endpoint_url', $middleware->url);
            update_option('lightrcommerce_bc_permanent_url', $permanentUrl);

            // get store data
//            $storeDetails = self::getStoreDetails($permanentUrl, $authToken);

            $nextStepUrl = esc_url_raw(add_query_arg([
                'page' => 'lightrcommerce_wizard',
                'tab' => 'step' . 2
                            ], admin_url('admin.php')
                    )
            );

            $response = [
                'ok' => true,
                'nextUrl' => $nextStepUrl
            ];
        } catch (\Exception $e) {
            $response = [
                'ok' => false,
                'error' => $e->getMessage()
            ];
        }

        wp_send_json($response);
        wp_die();
    }

    private static function _getAuthTokenFromSetup(string $permanentUrl, string $clientId, string $accessToken): string {
        $storeUrlData = BigCommerceHelper::parsePermanentStoreUrl($permanentUrl);

        if (!$storeUrlData) {
            throw new \Exception(esc_html__('The provided permanent store address seems invalid.', 'lightrcommerce'));
        }

        $headers = [
            'X-Auth-Token' => $accessToken,
            'X-Auth-Client' => $clientId,
            'Content-Type' => 'application/json',
        ];

        $expires = gmdate('U') + 100000000;

        $origin = wp_parse_url(get_site_url());
        $origin = sprintf('%s://%s', $origin['scheme'], $origin['host']);

        $query = sprintf('
            {
              "channel_id": %d,
              "expires_at": %d,
              "allowed_cors_origins": [
                "%s"
              ]
            }
        ', $storeUrlData->channelId, $expires, $origin);

        $endpoint = $storeUrlData->apiUrl . 'storefront/api-token';

        $request = wp_safe_remote_post($endpoint, [
            'headers' => $headers,
            'body' => $query,
        ]);

        $status = wp_remote_retrieve_response_code($request);
        $body = json_decode(wp_remote_retrieve_body($request));

        if ($status != \WP_Http::OK) {
            // translators: the response to the remote API call
            $errorMsg = __('Failed to connect. Please, check your credentials. BigCommerce responded "%s".', 'lightrcommerce');
            throw new \Exception(sprintf($errorMsg, $body->title)); // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped
        }

        return $body->data->token;
    }

//    public static function getStoreDetails(string $permanentUrl, string $authToken) {
//        $storeUrlData = BigCommerceHelper::parsePermanentStoreUrl($permanentUrl);
//
//        $headers = [
//            'Content-Type' => 'application/json',
//            'Authorization' => 'Bearer ' . $authToken
//        ];
//
//        $query = '
//            query {
//                site {
//                    settings {
//                        storeName
//                        storeHash
//                        status
//                        channelId
//                        logoV2 {
//                            ...StoreImageLogoFields
//                        }
//                    }
//                }
//            }
//            fragment StoreImageLogoFields on StoreImageLogo {
//                image {
//                    urlOriginal
//                    altText
//                }
//            }
//        ';
//
//        $endpoint = $storeUrlData->graphqlUrl;
//
//        $response = wp_safe_remote_post($endpoint, [
//            'headers' => $headers,
//            'body' => wp_json_encode(['query' => $query])
//        ]);
//
//        $bc_data = json_decode(wp_remote_retrieve_body($response));
//    }

    private static function _createMiddleware(string $permanentUrl, string $clientId, string $accessToken) {
        /* @var $user \WP_User */
        $user = wp_get_current_user();

        $endpoint = 'https://api.lightrcommerce.com/api/account/create';

        $response = wp_safe_remote_post($endpoint, [
            'body' => [
                'permanentUrl' => $permanentUrl,
                'clientId' => $clientId,
                'accessToken' => $accessToken,
                'frontend' => get_site_url()
            ]
        ]);

        $status = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($status != \WP_Http::OK) {
            throw new \Exception(__('Failed to connect. Please, check your BigCommerce credentials.', 'lightrcommerce')); // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped
        }

        return json_decode($body);
    }

    public static function uninstall() {
        if (!isset($_REQUEST["_wpnonce"]) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST["_wpnonce"])), "lightrcommerce_uninstall")) {
            wp_safe_redirect(wp_get_referer());
            exit;
        }

        $endpoint = 'https://api.lightrcommerce.com/api/account/destroy';

        $response = wp_safe_remote_post($endpoint, [
            'body' => [
                'id' => get_option('lightrcommerce_bc_endpoint_id'),
            ]
        ]);

        $status = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($status != \WP_Http::NO_CONTENT) {
            lightrcommerce_addFlashNotice(__('Failed to delete your data. Please, contact our support team.', 'lightrcommerce'), 'error');
            wp_safe_redirect(wp_get_referer());
            exit;
        }

        \LightrCommerce\Plugin\Helper::deleteOptions();

        $seeYouSoonPage = add_query_arg([
            'page' => 'lightrcommerce_see_you_soon',
                ], admin_url('admin.php')
        );

        wp_safe_redirect($seeYouSoonPage);
        exit;
    }
}
