<?php

namespace LightrCommerce\Plugin;

class Helper {

    /**
     * Checks if the plugin has completed the setup
     * @return bool
     */
    public static function isSetupCompleted(): bool {
        // return false;
        $required = [
            'lightrcommerce_bc_permanent_url',
            'lightrcommerce_bc_endpoint_url',
        ];

        foreach ($required as $option) {
            if (!get_option($option)) {
                return false;
            }
        }

        return true;
    }

    /**
     * 
     * @return bool
     */
    public static function isFrontendEnabled(): bool {
        return (bool) get_option('lightrcommerce_bc_frontend_enabled');
    }

    /**
     * Checks if Elementor is installed and enabled
     * @return bool
     */
    public static function isElementorActive(): bool {
        $isElementorInstalled = defined('ELEMENTOR_VERSION')
            || class_exists('Elementor\App\App')
            || (function_exists('is_plugin_active') && is_plugin_active('elementor/elementor.php'))
            || in_array('elementor/elementor.php', get_option('active_plugins'));

        return $isElementorInstalled;
    }

    /**
     * 
     * @return array
     */
    public static function getWidgetsList(): array {
        try {
            $config = \LightrCommerce\BigCommerce\Core::getConfig();
            $widgets = $config->widgets ?? [];
        } catch (LightrCommerce\BigCommerce\MiddlewareNotFoundException) {
            $widgets = [];
        } catch (\Exception $ex) {
            $widgets = [];
        }
        
        $widgets = array_filter($widgets, fn($widget) => $widget->enabled);

        foreach ($widgets as $widget) {
            $widget->className = '\LightrCommerce\Plugin\Widget\\' . $widget->code;
            $widget->icon = sprintf('%s/icons/%s.svg', LIGHTRCOMMERCE_ASSETS_URL, $widget->code);
            $widget->premium = false;
            $widget->wordpress = self::isWordPressWidget($widget);
            $widget->elementor = false;

            if (self::isElementorActive()) {
                $elementorClassName = '\LightrCommerce\Plugin\Widget\Elementor\\' . $widget->code;

                if (class_exists($elementorClassName)) {
                    $widget->elementor = true;
                    $widget->elementorClassName = $elementorClassName;
                }
            }
        }

        return $widgets;
    }
    
    /**
     * 
     * @param string $partialName
     * @param array $args
     * @return string
     */
    static function loadPartial(string $partialName, array $args = []): string {
        extract($args);
        $partialPath = sprintf('%s/partials/%s.php', LIGHTRCOMMERCE_TEMPLATE_PATH, $partialName);
        ob_start();
        require $partialPath;
        $html = ob_get_contents();
        ob_end_clean();
        return $html;
    }

    /**
     * 
     * @return string
     */
    public static function getWidgetsListHtml(): string {
        return self::loadPartial('widgets_list');
    }

    /**
     * 
     * @param type $widget
     * @return bool
     */
    private static function isWordPressWidget($widget): bool {
        $widgetPath = sprintf('%s/src/Plugin/Widget/%s.php', LIGHTRCOMMERCE_PATH, $widget->code);
        return !file_exists($widgetPath);
    }
    
    /**
     * 
     * @return bool
     */
    public static function deleteOptions(): bool {
        $options = [
            'lightrcommerce_bc_catalog_page',
            'lightrcommerce_bc_endpoint_id',
            'lightrcommerce_bc_endpoint_name',
            'lightrcommerce_bc_endpoint_url',
            'lightrcommerce_bc_permanent_url',
            'lightrcommerce_bc_frontend_enabled',
            'lightrcommerce_config',
        ];

        return array_walk($options, 'delete_option');
    }
    
    /**
     * Print internal content (not user input) without escaping.
     */
    public static function printUnescapedInternalString($string) {
        echo $string; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    }
}
