<?php

namespace LightrCommerce\Plugin;

use LightrCommerce\BigCommerce\Core as Api;
use LightrCommerce\Plugin\Helper;

// this requires are needed for the cli plugin check
require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-base.php';
require_once ABSPATH . 'wp-admin/includes/class-wp-filesystem-direct.php';

class Controller {

    const CATALOG_QUERY_VAR = 'lightrcommerce-catalog-path';
    const API_QUERY_VAR = 'lightrcommerce-api';

    static $currentCatalogItem;

    /**
     * Register the routes for the objects of the controller.
     */
    static function addRewriteRule() {
        $pageId = get_option('lightrcommerce_bc_catalog_page');

        if (!$pageId) {
            // no page set to use as catalog entry point
            return null;
        }

        // @var $page \WP_Post
        $post = get_post($pageId);

        // see list of possible status: https://wordpress.org/documentation/article/post-status/
        $isValidPostStatus = $post->post_status == 'publish' || $post->post_status == 'private';

        if (!$post || !$isValidPostStatus) {
            // page does not exist
            return null;
        }

        // if plain permalink schema
        if (get_option('permalink_structure') == '') {
            add_rewrite_rule(
                    '^p=' . $post->ID . '\b',
                    'index.php?page_id=' . $pageId . '&' . self::CATALOG_QUERY_VAR . '=$matches[1]',
                    'top'
            );
            self::addHtaccessRewriteRules();
        } else {
            add_rewrite_rule(
                    '^lightrcommerce/(.*)/?$',
                    'index.php?page_id=' . $pageId . '&' . self::CATALOG_QUERY_VAR . '=$matches[1]',
                    'top'
            );
            add_rewrite_rule(
                    '^api/(.*)/?$',
                    'index.php?' . self::API_QUERY_VAR . '=$matches[1]',
                    'top'
            );
            self::removeHtaccessRewriteRules();
        }

        flush_rewrite_rules();
    }

    /**
     * Add the plugin's query vars
     */
    static function addQueryVars($vars) {
        $vars[] = self::CATALOG_QUERY_VAR;
        $vars[] = self::API_QUERY_VAR;
        return $vars;
    }

    /**
     * Add the template redirect.
     */
    static function catalogRedirect($query) {
        //if the query var lightrcommerce-catalog-path is set load the template
        if (self::isCatalogPage($query)) {
            wp_enqueue_style('lightrcommerce_catalog', "$assetsUrl/assets/css/lightrcommerce.css", [], LIGHTRCOMMERCE_VERSION);

            add_filter('the_content', function ($content) use ($query) {
                try {
                    $catalogItem = self::loadCatalogItem($query);
                } catch (\Exception $ex) {
                    $args = compact('ex');

                    if ($ex->getCode() == 404) {
                        $catalogItem['html'] = Helper::loadPartial('not_found', $args);
                    } else {
                        $catalogItem['html'] = Helper::loadPartial('middleware_error', $args);
                    }
                }

                $assetsUrl = rtrim(LIGHTRCOMMERCE_URL, '/');

                return '<div class="lightrcommerce-wrapper">' . $catalogItem['html'] . '</div>';
            });
        }

        return $query;
    }

    static function apiRedirect($query) {
        if ($query->get(self::API_QUERY_VAR) && $query->is_main_query()) {
            $api_path = $query->get(self::API_QUERY_VAR);

            $requestUri = filter_input(INPUT_SERVER, 'REQUEST_URI', FILTER_SANITIZE_URL);

            $endpoint = str_replace('/api/', '', $requestUri);

            switch ($endpoint) {
                case 'search':
                    $response = \LightrCommerce\BigCommerce\Core::callApi($endpoint, [
                                'method' => 'POST',
                                'headers' => [
                                    'accept' => 'application/json',
                                    'content-type' => 'application/json',
                                ],
                                'body' => file_get_contents('php://input'),
                    ]);

                    header('Content-Type: application/json');
                    
                    $acceptEncoding = filter_input(INPUT_SERVER, 'HTTP_ACCEPT_ENCODING', FILTER_SANITIZE_FULL_SPECIAL_CHARS);

                    if (stripos($acceptEncoding, 'gzip') !== false) {
                        ob_start('ob_gzhandler');
                        echo wp_json_encode($response);
                        ob_end_flush();
                    } else {
                        echo wp_json_encode($response);
                    }
                    die();

                default:
                    $headers = array_filter([
                        'accept' => filter_input(INPUT_SERVER, 'HTTP_ACCEPT', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
                        'content-type' => filter_input(INPUT_SERVER, 'CONTENT_TYPE', FILTER_SANITIZE_FULL_SPECIAL_CHARS)
                    ]);
                    $response = \LightrCommerce\BigCommerce\Core::callApi($endpoint, [
                                'method' => filter_input(INPUT_SERVER, 'REQUEST_METHOD', FILTER_SANITIZE_FULL_SPECIAL_CHARS),
                                'headers' => $headers,
                                'body' => file_get_contents('php://input'),
                                    ], false);
                    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
                    echo $response;
                    die();
            }
        }
    }

    static function parseResponse(string $response) {
        $html = $response;

        $meta = [];

        $html = preg_replace('#<meta charset=".*?" */>#iu', '', $html);

        $total = preg_match_all(
                '#<meta (property|name)="(.*?)" content="(.*?)" ?/?>#msiu',
                $html,
                $matches
        );

        for ($i = 0; $i < $total; ++$i) {
            $meta[] = [$matches[1][$i], $matches[2][$i], $matches[3][$i]];
            $html = str_replace($matches[0][$i], '', $html);
        }

        if (preg_match('#<title>(.*?)</title>#', $html, $match)) {
            $title = $match[1];
            $html = str_replace($match[0], '', $html);
        } else {
            $title = null;
        }

        return [
            'title' => $title,
            'meta' => $meta,
            'html' => $html,
        ];
    }

    static function loadCatalogItem(\WP_Query $query) {
        if (!self::$currentCatalogItem) {
            $response = Api::getOutput('/lightrcommerce/' . $query->get(self::CATALOG_QUERY_VAR));
            $parsed = self::parseResponse($response);

            self::$currentCatalogItem = $parsed;
        }

        return self::$currentCatalogItem;
    }

    static function bufferStart($html) {
        ob_start();
    }

    static function addGeneratorTag(): void {
        printf('<meta name="generator" content="LightR Commerce headless integration for WordPress, by YoungDogs." />');
    }

    static function addHeadCountTag(): void {
        printf('<meta name="next-head-count" content="1" />');
    }

    static function isCatalogPage(\WP_Query $query): bool {
        return $query->get(self::CATALOG_QUERY_VAR);
    }

    /**
     * @global \WP_Query $wp_query
     */
    static function headBufferEnd() {
        global $wp_query;

        if (self::isCatalogPage($wp_query)) {
            $head = ob_get_contents();
            ob_end_clean();

            try {
                self::loadCatalogItem($wp_query);

                $newTitle = sprintf('<title>%s</title>', self::$currentCatalogItem['title']);
                $head = preg_replace('#<title>.*?</title>#', $newTitle, $head);

                // replace existing meta tags with those from the middleware
                foreach (self::$currentCatalogItem['meta'] as $meta) {
                    $pattern = sprintf(
                            '#<meta %s="%s" content=".*?".*?/?>#msiu',
                            preg_quote($meta[0]),
                            preg_quote($meta[1])
                    );
                    $newMeta = sprintf('<meta %s="%s" content="%s">', $meta[0], $meta[1], $meta[2]);

                    if (preg_match($pattern, $head)) {
                        $head = preg_replace($pattern, $newMeta, $head);
                    } else {
                        $head .= $newMeta;
                    }
                }
            } catch (\Exception $ex) {
                
            } finally {
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
                echo $head;
            }
        } else {
            ob_end_flush();
        }
    }

    static function addHtaccessRewriteRules() {
        $htaccessPath = ABSPATH . '.htaccess';

        $fs = new \WP_Filesystem_Direct(true);

        if ($fs->exists($htaccessPath) && !$fs->is_writable($htaccessPath)) {
            return;
        }

        $content = $fs->get_contents($htaccessPath);

        $apiQueryVar = self::API_QUERY_VAR;
        $catalogQueryVar = self::CATALOG_QUERY_VAR;
        $catalogPageId = get_option('lightrcommerce_bc_catalog_page');

        self::removeHtaccessRewriteRules();
        
        // remove existing rewrite snippet if exists
        $newContent = preg_replace('~# BEGIN LightR Commerce plugin.*?# END LightR Commerce plugin~ms', '', $content);

        $newContent .= PHP_EOL . "
# BEGIN LightR Commerce plugin
<IfModule mod_rewrite.c>
  RewriteCond %{REQUEST_FILENAME} !-f
  RewriteCond %{REQUEST_FILENAME} !-d
  RewriteRule ^api/(.*)/?$ /index.php?$apiQueryVar=$1

  RewriteCond %{REQUEST_FILENAME} !-f
  RewriteCond %{REQUEST_FILENAME} !-d
  RewriteRule ^lightrcommerce/(.*)/?$ /index.php?page_id=$catalogPageId&$catalogQueryVar=$1
</IfModule>
# END LightR Commerce plugin
";

        if ($content != $newContent) {
            $fs->put_contents($htaccessPath, $newContent);
        }
    }

    static function removeHtaccessRewriteRules() {
        $htaccessPath = ABSPATH . '.htaccess';

        $fs = new \WP_Filesystem_Direct(true);

        if ($fs->exists($htaccessPath) && !$fs->is_writable($htaccessPath)) {
            return;
        }

        $content = $fs->get_contents($htaccessPath);

        $newContent = preg_replace('~# BEGIN LightR Commerce plugin.*?# END LightR Commerce plugin~ms', '', $content);

        if ($content != $newContent) {
            $fs->put_contents($htaccessPath, $newContent);
        }
    }
}
