<?php

namespace LightrCommerce\Plugin;

use LightrCommerce\Plugin\Helper;

class Admin {

    public function __construct() {
        add_action('admin_menu', [$this, 'addMenuPages']);
        add_action('admin_init', [$this, 'registerSettingFields']);

        $isLightrCommercePage = strpos(filter_input(INPUT_GET, 'page', FILTER_SANITIZE_FULL_SPECIAL_CHARS) ?? '', 'lightrcommerce') === 0;

        if ($isLightrCommercePage) {
            add_action('admin_enqueue_scripts', function (string $hook) {
                wp_enqueue_style('lightrcommerce_wizard', LIGHTRCOMMERCE_ASSETS_URL . '/wizard.css', null, LIGHTRCOMMERCE_VERSION);

                wp_enqueue_script('tailwind-js', LIGHTRCOMMERCE_ASSETS_URL . '/js/tailwind.min.js', [], '3.4.5', false, ['in_footer' => true]); // phpcs:ignore WordPress.WP.EnqueuedResourceParameters.NoExplicitVersion

                
                wp_enqueue_script('lightrcommerce_wizard_script', LIGHTRCOMMERCE_ASSETS_URL . '/js/wizard.js', ['jquery'], LIGHTRCOMMERCE_VERSION, [
                    'in_footer' => true
                ]);

                wp_localize_script('lightrcommerce_wizard_script', 'lightrcommerce', [
                    'admin_ajax_url' => admin_url( 'admin-ajax.php' )
                ]);

                if ($hook == 'toplevel_page_lightrcommerce_wizard') {
                    wp_enqueue_script('vimeo', 'https://player.vimeo.com/api/player.js', [], false, ['in_footer' => true]); // phpcs:ignore WordPress.WP.EnqueuedResourceParameters.NoExplicitVersion
                }
            });
        }

        $this->addLinksToPluginsList();
    }

    public function registerSettingFields() {
        add_settings_section(
                'lightrcommerce_general_section',
                'General Settings',
                null,
                'lightrcommerce'
        );
        add_settings_field(
                'lightrcommerce_bc_endpoint_url',
                'Endpoint URL',
                [$this, 'optionOutput'],
                'lightrcommerce',
                'lightrcommerce_general_section',
                ['code' => 'lightrcommerce_bc_endpoint_url']
        );
        add_settings_field(
                'lightrcommerce_api_key',
                'api_key',
                [$this, 'optionOutput'],
                'lightrcommerce',
                'lightrcommerce_general_section',
                ['code' => 'lightrcommerce_api_key']
        );
        register_setting('lightrcommerce_general_group', 'lightrcommerce_bc_endpoint_url');
        register_setting('lightrcommerce_general_group', 'lightrcommerce_api_key');

        add_settings_section(
                'lightrcommerce_catalog_section',
                'Catalog Settings',
                null,
                'lightrcommerce'
        );
        add_settings_field(
                'lightrcommerce_bc_catalog_page',
                'Catalog Page',
                [$this, 'optionOutput'],
                'lightrcommerce',
                'lightrcommerce_catalog_section',
                ['code' => 'lightrcommerce_bc_catalog_page']
        );
        register_setting('lightrcommerce_catalog_group', 'lightrcommerce_bc_catalog_page');
    }

    public function addMenuPages() {
        if (Helper::isSetupCompleted()) {
            add_menu_page(
                    'LightR Commerce',
                    'LightR Commerce',
                    'manage_options',
                    'lightrcommerce',
                    [$this, 'loadOverviewPage'],
                    'dashicons-cart',
                    6
            );
        } else {
            add_menu_page(
                    'LightR Commerce',
                    'LightR Commerce',
                    'manage_options',
                    'lightrcommerce_wizard',
                    [$this, 'loadWizardPage'],
                    'dashicons-cart',
                    6
            );
        }

        if (\LightrCommerce\Plugin\Helper::isSetupCompleted()) {
            add_submenu_page('lightrcommerce', __('Overview', 'lightrcommerce') . ' - LightrCommerce', 'Overview', 'manage_options', 'lightrcommerce', [$this, 'loadOverviewPage']);
            add_submenu_page('lightrcommerce', 'BigCommerce - LightrCommerce', 'BigCommerce', 'manage_options', 'lightrcommerce_bigcommerce', [$this, 'loadBigCommercePage']);
            add_submenu_page('lightrcommerce', __('Setup Wizard', 'lightrcommerce') . ' - LightrCommerce', 'Setup Wizard', 'manage_options', 'lightrcommerce_wizard', [$this, 'loadWizardPage']);
            add_submenu_page('lightrcommerce', __('Support', 'lightrcommerce') . ' - LightrCommerce', 'Support', 'manage_options', 'lightrcommerce_support', [$this, 'loadSupportPage']);
        }

        add_submenu_page('lightrcommerce - hidden', 'Support - LightrCommerce', 'Support', 'manage_options', 'lightrcommerce_unsupported', [$this, 'loadUnsupportedPlatformPage']);
        add_submenu_page('lightrcommerce - hidden', 'See you soon - LightrCommerce', 'See you soon', 'manage_options', 'lightrcommerce_see_you_soon', [$this, 'loadSeeYouSoonPage']);
    }

    private function _loadPage(string $pageName) {
        $path = sprintf('%ssrc/templates/%s.php', LIGHTRCOMMERCE_PATH, $pageName);

        echo Helper::loadPartial('_wrapper_before'); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        include $path;
        echo Helper::loadPartial('_wrapper_after'); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    }

    public function loadOverviewPage() {
        return $this->_loadPage('dashboard');
    }

    public function loadBigCommercePage() {
        return $this->_loadPage('bigcommerce');
    }

    public function loadWizardPage() {
        return $this->_loadPage('wizard');
    }

    public function loadSupportPage() {
        return $this->_loadPage('support');
    }

    public function loadUnsupportedPlatformPage() {
        return $this->_loadPage('unsupported_platform');
    }

    public function loadSeeYouSoonPage() {
        return $this->_loadPage('see_you_soon');
    }

    public function optionOutput($args) {
        $code = $args['code'];
        $value = get_option($code);
        ?><input type="text" name="<?php esc_attr($code) ?>" value="<?php esc_attr($value) ?>" /><?php
    }

    /**
     * Adds links to the list of plugins available at Backend > Plugins > Installed Plugins
     */
    public function addLinksToPluginsList() {
        add_filter('plugin_action_links_' . LIGHTRCOMMERCE_PLUGIN_BASE, function ($links) {
            $url = esc_url(add_query_arg('page', 'lightrcommerce_support', get_admin_url() . 'admin.php'));

            $settings_link = sprintf('<a href="%s">%s</a>', $url, __('Support', 'lightrcommerce'));

            array_unshift(
                    $links,
                    $settings_link
            );

            if (Helper::isSetupCompleted()) {
                $url = esc_url(add_query_arg(
                                'page',
                                'lightrcommerce',
                                get_admin_url() . 'admin.php'
                ));

                $settings_link = sprintf('<a href="%s">%s</a>', $url, __('Configuration', 'lightrcommerce'));
            } else {
                $url = esc_url(add_query_arg(
                                'page',
                                'lightrcommerce_wizard',
                                get_admin_url() . 'admin.php'
                ));

                $settings_link = sprintf('<a href="%s">%s</a>', $url, __('Complete configuration wizard', 'lightrcommerce'));
            }

            array_unshift(
                    $links,
                    $settings_link
            );

            $response = wp_remote_get('https://api.lightrcommerce.com/api/plugin/wordpress', [
                'timeout' => 2
            ]);

            if (!is_wp_error($response)) {
                $body = @json_decode(wp_remote_retrieve_body($response));

                if (isset($body->version) && version_compare(LIGHTRCOMMERCE_VERSION, $body->version, '<')) {
                    $links[] = sprintf('<span style="display: inline-flex; gap: 5px"><svg style="width: 1em; color: #d63638" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24" stroke-width="2.5" stroke="currentColor" class="size-6">
  <path stroke-linecap="round" stroke-linejoin="round" d="M16.023 9.348h4.992v-.001M2.985 19.644v-4.992m0 0h4.992m-4.993 0 3.181 3.183a8.25 8.25 0 0 0 13.803-3.7M4.031 9.865a8.25 8.25 0 0 1 13.803-3.7l3.181 3.182m0-4.991v4.99" />
</svg>
<a href="%s" download style="font-weight: bold; white-space: nowrap">Update to version %s</a></span>', $body->url, $body->version);
                }
            }

            return $links;
        });
    }
}
