<?php

namespace LightrCommerce;

use LightrCommerce\Plugin\Admin;
use LightrCommerce\Plugin\Helper;
use LightrCommerce\Plugin\Wizard;
use LightrCommerce\BigCommerce\Core as Api;
use LightrCommerce\BigCommerce\MiddlewareNotFoundException;

require __DIR__ . '/FlashNotices.php';

class Plugin {

    const CONFIG_OPTION_KEY = 'lightrcommerce_config';

    public function __construct() {
        spl_autoload_register([$this, 'autoload']);

        if (is_admin()) {
            new Admin();
        }

        add_action('plugins_loaded', 'LightrCommerce\\Plugin::load_textdomain');

        if (Helper::isSetupCompleted()) {
            // widgets need to be registered both in admin and frontend
            add_action('widgets_init', [$this, 'registerWidgets']);

            if (is_admin()) {
                $options = get_option(self::CONFIG_OPTION_KEY, null);
                if (!$options || (time() - $options->timestamp > 86400)) {
                    $this->updateConfig();
                }

                // elementor widgets
                if (Helper::isElementorActive()) {
                    add_action('elementor/elements/categories_registered', [
                        $this,
                        'registerElementorCategory',
                    ]);
                    add_action('elementor/widgets/widgets_registered', [$this, 'addElementorModules']);
                }

                add_action('admin_post_lightrcommerce_save_catalog_page', [$this, 'catalogPageFormHandler']);
                add_action('admin_post_lightrcommerce_create_catalog_page', [$this, 'catalogPageCreateFormHandler']);
                add_action('admin_post_lightrcommerce_save_settings', [$this, 'saveSettings']);
                add_action('admin_post_lightrcommerce_clear_widgets_cache', [$this, 'clearWidgetsCache']);
                add_action('admin_post_lightrcommerce_uninstall', [Wizard::class, 'uninstall']);
                add_action('wp_head', 'LightrCommerce\\Plugin\\Controller::addHeadCountTag', 0);
            } else {
                if (Helper::isFrontendEnabled()) {
                    add_action('init', 'LightrCommerce\\Plugin\\Controller::addRewriteRule');
                    add_filter('query_vars', 'LightrCommerce\\Plugin\\Controller::addQueryVars');
                    add_action('pre_get_posts', 'LightrCommerce\\Plugin\\Controller::catalogRedirect');
                    add_action('pre_get_posts', 'LightrCommerce\\Plugin\\Controller::apiRedirect');
                    add_action('wp_head', 'LightrCommerce\\Plugin\\Controller::bufferStart', -99);
                    add_action('wp_head', 'LightrCommerce\\Plugin\\Controller::headBufferEnd', 99);
                    add_action('wp_head', 'LightrCommerce\\Plugin\\Controller::addGeneratorTag', 0);
                    add_action('wp_head', 'LightrCommerce\\Plugin\\Controller::addHeadCountTag', 0);

                    add_action('wp_enqueue_scripts', function () {
                        wp_enqueue_script('lightrcommerce-script', LIGHTRCOMMERCE_URL . 'assets/js/lightrcommerce.js', [], LIGHTRCOMMERCE_VERSION, [
                            'in_footer' => true
                        ]);
                    });
                    
                    if (Helper::isElementorActive()) {
                        add_action('elementor/widgets/widgets_registered', [$this, 'addElementorModules']);
                    }
                }
            }
            
        } else {
            add_action('admin_post_lightrcommerce_setup', [$this, 'setupFormHandler']);
        }


        add_action('wp_ajax_test_bigcommerce_permanent_url', [Wizard::class, 'testPermanentUrl']);
        add_action('wp_ajax_lightrcommerce_setup', [Wizard::class, 'testAndSaveSetup']);
    }

    private function updateConfig() {
        try {
            $config = Api::getConfig();
            $config->timestamp = time();
            update_option(self::CONFIG_OPTION_KEY, $config);
        } catch (MiddlewareNotFoundException $ex) {
            
        } catch (\Exception $e) {
//            echo $e;
        }
    }

    /**
     * Get config for a widget
     */
    static function getWidgetConfig(string $widget_code) {
        $config = get_option(self::CONFIG_OPTION_KEY, null);
        if ($config !== null) {
            foreach ($config->widgets as $widget) {
                if ($widget->code == $widget_code) {
                    return $widget;
                }
            }
        }
        return null;
    }

    public function autoload($class) {
        if (0 !== strpos($class, __NAMESPACE__)) {
            return;
        }

        $class = str_replace(__NAMESPACE__ . '\\', '', $class);
        $class = str_replace('\\', '/', $class);
        $file = LIGHTRCOMMERCE_PATH . 'src/' . $class . '.php';
        if (file_exists($file)) {
            require_once $file;
        }
    }

    public function registerWidgets() {
        try {
            $config = Api::getConfig();

            foreach ($config->widgets ?? [] as $widget) {
                $classname = 'LightrCommerce\Plugin\Widget\\' . $widget->code;
                if (class_exists($classname)) {
                    register_widget($classname);
                }
            }
        } catch (MiddlewareNotFoundException $e) {
            
        } catch (\Exception $e) {
//            echo $e;
        }
    }

    public function registerElementorCategory() {
        \Elementor\Plugin::instance()->elements_manager->add_category(
                'lightrcommerce',
                [
                    'title' => 'LightRCommerce Node',
                    'icon' => 'fa fa-plug',
                ],
                1
        );
    }

    public function addElementorModules() {
        try {
            $widgets = Helper::getWidgetsList();

            $elementorWidgets = array_filter($widgets, fn($widget) => $widget->elementor);

            foreach ($elementorWidgets as $widget) {
                \Elementor\Plugin::instance()->widgets_manager->register_widget_type(
                        new $widget->elementorClassName
                );
            }
        } catch (MiddlewareNotFoundException $e) {
            
        } catch (\Exception $e) {
//            echo $e;
        }
    }

    static function load_textdomain() {
        $plugin_rel_path = 'lightrcommerce/' .
                basename(dirname(__FILE__)) .
                '/languages'; /* Relative to WP_PLUGIN_DIR */
        load_plugin_textdomain('lightrcommerce', false, $plugin_rel_path);
    }

    public function setupFormHandler() {
        $parameters = [
            'lightrcommerce_bc_api_url' => 'BIGCOMMERCE_STORE_API_URL',
            'lightrcommerce_bc_graphql_url' => 'BIGCOMMERCE_STOREFRONT_API_URL',
            'lightrcommerce_bc_client_id' => 'BIGCOMMERCE_STORE_API_CLIENT_ID',
            'lightrcommerce_bc_client_secret' => 'BIGCOMMERCE_STORE_API_CLIENT_SECRET',
        ];

        $body = null;

        foreach ($parameters as $src => $dest) {
            $value = filter_input(INPUT_POST, $src, FILTER_SANITIZE_FULL_SPECIAL_CHARS);

            if (empty($value)) {
                throw new \Exception(esc_html('Missing ' . $src));
            }

            $body[$dest] = sanitize_text_field($value);
        }

        try {
            /**
             * If post is ok, we get back an array. However, if the request fails,
             * we receive an object of type WP_Error. Thank you, WordPress.
             */
            $endpoint = 'https://api.lightrcommerce.com/api/account/create';

            $response = wp_safe_remote_post($endpoint, [
                'body' => wp_json_encode($body),
                'headers' => [
                    'Content-Type' => 'application/json',
                ],
            ]);

            if (is_a($response, 'WP_Error')) {
                throw new \Exception(
                                sprintf(
                                        'An error occurred: %s',
                                        implode(PHP_EOL, $response->get_error_messages())
                                )
                );
            }

            if ($response['response']['code'] != 200) {
                if ($response['body']) {
                    try {
                        $body = json_decode($response['body']);

                        if ($body && isset($body->error->code) && isset($body->error->message)) {
                            throw new \Exception(
                                            sprintf(
                                                    'An error occurred: (%s) %s',
                                                    $body->error->code,
                                                    $body->error->message
                                            )
                            );
                        }
                    } catch (Exception $ex) {
                        throw new \Exception($ex);
                    }
                }

                throw new \Exception(
                                sprintf(
                                        'Unexpected response from remote API: %d %s',
                                        $response['response']['code'],
                                        $response['response']['message']
                                )
                );
            }

            // save just-created account to db
            $account = json_decode($response['body']);

            update_option('lightrcommerce_bc_endpoint_id', $account->id);
            update_option('lightrcommerce_bc_endpoint_url', $account->url);
            update_option('lightrcommerce_bc_endpoint_name', $account->name);

            // save parameters used to create the account
            foreach ($parameters as $key => $value) {
                $value = filter_input(INPUT_POST, $key, FILTER_SANITIZE_FULL_SPECIAL_CHARS);
                $value = sanitize_text_field($value);

                update_option($key, $value);
            }

            lightrcommerce_addFlashNotice('You have been successfully connected!', 'success', false);

            wp_redirect($this->getSettingsUrl(2));
        } catch (\Exception $e) {
            lightrcommerce_addFlashException($e);

            wp_redirect($this->getSettingsUrl(1));
        }
    }

    public function catalogPageFormHandler() {
        check_admin_referer('lightrcommerce_save_catalog_page');
        check_ajax_referer('lightrcommerce_save_catalog_page');

        $pageId = (int) filter_input(INPUT_POST, 'lightrcommerce_bc_catalog_page', FILTER_VALIDATE_INT);

        if ($pageId > 0) {
            update_option('lightrcommerce_bc_catalog_page', $pageId, '', 'no');
            $nextStep = 3;
        } else {
            lightrcommerce_addFlashNotice('The selected page is not correct', 'error');
            $nextStep = 2;
        }

        wp_redirect($this->getSettingsUrl($nextStep));
    }

    public function catalogPageCreateFormHandler() {
        check_admin_referer('lightrcommerce_create_catalog_page');
        check_ajax_referer('lightrcommerce_create_catalog_page');

        $newPageName = trim(filter_input(INPUT_POST, 'lightrcommerce_bc_catalog_page_new', FILTER_SANITIZE_FULL_SPECIAL_CHARS));

        $suffix = '';

        // check if we can use wp_unique_post_slug()
        // https://developer.wordpress.org/reference/functions/wp_unique_post_slug/
        do {
            $pageSlug = sanitize_title($newPageName . $suffix++);
        } while (get_page_by_path($pageSlug, OBJECT, 'page'));

        $newPage = array(
            'post_type' => 'page',
            'post_title' => $newPageName,
            'post_content' => __('LightrCommerce page for catalog', 'lightrcommerce'),
            'post_status' => 'publish',
            'post_author' => wp_get_current_user()->ID,
            'post_name' => $pageSlug,
            'post_parent' => get_option('page_on_front')
        );

        $newPageId = wp_insert_post($newPage);

        update_option('lightrcommerce_bc_catalog_page', $newPageId, '', 'no');

        wp_redirect($this->getSettingsUrl(3));
    }

    private function getSettingsUrl($step = null): string {
        if ($step) {
            return esc_url_raw(
                    add_query_arg(
                            [
                                'page' => 'lightrcommerce_wizard',
                                'tab' => 'step' . $step,
                            ],
                            admin_url('admin.php')
                    )
            );
        }

        return esc_url_raw(
                add_query_arg(
                        [
                            'page' => 'lightrcommerce_wizard',
                        ],
                        admin_url('admin.php')
                )
        );
    }

    public function saveSettings() {
        if (!isset($_REQUEST["_wpnonce"]) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST["_wpnonce"])), "lightrcommerce_save_settings")) {
            wp_safe_redirect(wp_get_referer());
            exit;
        }

        $bcCatalogPage = (int) filter_input(INPUT_POST, 'lightrcommerce_bc_catalog_page', FILTER_VALIDATE_INT);
        update_option('lightrcommerce_bc_catalog_page', $bcCatalogPage);

        $bcFrontendEnabled = !empty($_POST['lightrcommerce_bc_frontend_enabled']);
        update_option('lightrcommerce_bc_frontend_enabled', $bcFrontendEnabled);

        wp_safe_redirect(wp_get_referer());
        exit;
    }

    public function clearWidgetsCache() {
        if (!isset($_REQUEST["_wpnonce"]) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST["_wpnonce"])), "lightrcommerce_clear_widgets_cache")) {
            wp_safe_redirect(wp_get_referer());
            exit;
        }

        delete_option('lightrcommerce_config');

        lightrcommerce_addFlashNotice('Widgets configuration is cleared.');

        wp_safe_redirect(wp_get_referer());
        exit;
    }
}
