<?php

if (!defined('ABSPATH')) exit; // exit if accessed directly

// https://www.webprogramo.com/admin-notices-after-a-page-refresh-on-wordpress/1183/
/**
 * Add a flash notice to {prefix}options table until a full page refresh is done
 *
 * @param string $notice our notice message
 * @param string $type This can be "info", "warning", "error" or "success", "warning" as default
 * @param boolean $dismissible set this to TRUE to add is-dismissible functionality to your notice
 * @return void
 */
function lightrcommerce_addFlashNotice(string $notice = '', string $type = 'warning', bool $dismissible = true) {
    // Here we return the notices saved on our option, if there are not notices, then an empty array is returned
    $notices = get_option('lightrcommerce_flash_notices', []);

    $dismissible_text = $dismissible ? 'is-dismissible' : '';

    // We add our new notice.
    array_push($notices, [
        'notice' => $notice,
        'type' => $type,
        'dismissible' => $dismissible_text,
    ]);

    // Then we update the option with our notices array
    update_option('lightrcommerce_flash_notices', $notices);
}

function lightrcommerce_addFlashException(\Exception $e) {
    lightrcommerce_addFlashNotice($e->getMessage(), 'error', true);
}

/**
 * Function executed when the 'admin_notices' action is called, here we check if there are notices on
 * our database and display them, after that, we remove the option to prevent notices being displayed forever.
 * @return void
 */
// We add our display_flash_notices function to the admin_notices
add_action(
    'admin_notices',
    function () {
        $notices = get_option('lightrcommerce_flash_notices', []);

        // Iterate through our notices to be displayed and print them.
        foreach ($notices as $notice) {
            printf(
                '<div class="notice notice-%1$s %2$s"><h3>%4$s</h3><p>%3$s</p></div>',
                esc_attr($notice['type']),
                esc_attr($notice['dismissible']),
                esc_html($notice['notice']),
                esc_html($notice['title'] ?? 'LightrCommerce for BigCommerce by Young-dogs.com')
            );
        }

        // Now we reset our options to prevent notices being displayed forever.
        if (!empty($notices)) {
            delete_option('lightrcommerce_flash_notices', []);
        }
    },
    12
);
