<?php

namespace LightrCommerce\BigCommerce;

use LightrCommerce\BigCommerce\MiddlewareNotFoundException;

class Core {

    static function getEndpointURL() {
        return get_option('lightrcommerce_bc_endpoint_url');
    }

    static function getApiUrl() {
        return rtrim(self::getEndpointURL(), '/') . '/api/';
    }

    static function getOutput(string $endpoint): string {
        if (strpos($endpoint, self::getEndpointURL()) !== 0) {
            $endpoint = rtrim(self::getEndpointURL(), '/') . '/' . $endpoint;
        }

        // https://developer.wordpress.org/plugins/http-api/
        // https://developer.wordpress.org/plugins/http-api/#wordpress-transients
        $response = wp_safe_remote_get($endpoint);
        $status = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        if ($status >= 400) {
            error_log("Call to '$endpoint' failed with status $status. Response is: " . print_r($response, true));

            if (WP_DEBUG && WP_DEBUG_DISPLAY) {
                return "<details><summary>There was an error rendering this widget</summary><strong>Status</strong>: <code>$status</code><br><strong>Endpoint</strong>: <code>$endpoint</code><br><strong>Response</strong>:<div>" . htmlentities($body) . "</div></details>";
            }

            return "There was an error rendering this widget";
        }

        $body = preg_replace('#<html lang="\w+">#', '<html>', $body);
        $body = preg_replace('#<title>.*?</title>#', '', $body);

        $body = str_replace(
                ['<!DOCTYPE html><html><head>', '</head>', '</html>'],
                '',
                $body
        );
        $body = str_replace(
                'src="/_next',
                'src="' . rtrim(self::getEndpointURL(), '/') . '/_next',
                $body
        );
        $body = str_replace(
                'href="/_next',
                'href="' . rtrim(self::getEndpointURL(), '/') . '/_next',
                $body
        );

        return $body;
    }

    static function getConfig() {
        return self::callApi('config');
    }

    static function getProducts() {
        return self::callApi('products');
    }

    static function getCategories() {
        return self::callApi('categories');
    }

    static function getApiResponse(string $endpoint, string $method = 'get', array $options = []) {
        // call to any of the functions of wp_remote_*
        // see https://developer.wordpress.org/plugins/http-api/
        $function = sprintf('wp_remote_%s', strtolower($method));

        if (strpos($endpoint, self::getApiUrl()) !== 0) {
            $endpoint = rtrim(self::getApiUrl(), '/') . '/' . $endpoint;
        }

        return $function($endpoint, $options);
    }

    static function callApi(
            string $endpoint,
            array $options = [],
            bool $returnJsonResponse = true
    ) {
        // https://developer.wordpress.org/plugins/http-api/
        // https://developer.wordpress.org/plugins/http-api/#wordpress-transients
        $response = self::getApiResponse(self::getApiUrl() . $endpoint, 'get', $options);
        $status = (int) wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        switch ($status) {
            case 404:
                throw new MiddlewareNotFoundException();
                
            case $status >= 400 && $status <= 599:
                // translators: 1: endpoint, 2: status code, 3: full response body
                $msg = sprintf(__('Call to "%1$s" failed with status %2$d. Response was: %3$s', 'lightrcommerce'), $endpoint, $status, $body);
                throw new \Exception($msg, $status); // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped

            default:
                return $returnJsonResponse ? json_decode($body) : $body;
        }
    }

    function callBigCommerce(string $endpoint, $data = [], $method = 'GET') {
        $options = [
            'method' => $method,
            'headers' => [
                'accept' => 'application/json',
                'cache-control' => 'no-cache',
                'content-type' => 'application/json',
                'x-auth-token' => self::ACCESS_TOKEN,
            ],
            'body' => $data
        ];

        $response = wp_remote_request($endpoint, $options);
        $status = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);

        return [
            'status' => $status,
            'data' => json_decode($body),
        ];
    }
}
