<?php
/**
 * LenoChat Login View
 *
 * @package LenoChat
 */

if (!defined("WPINC")) {
    die();
}

$status = get_option("lenochat_status");
$license = get_option("lenochat_license");
$lang = get_option("lenochat_language");
$encodedToken = "";
$lenochat_needs_redirect = false;

require_once realpath(dirname(__FILE__) . "/..") . "/lenochat-env-functions.php";

try {
    $env = lenochat_load_env_func();
    $website_url = isset($env["LENOCHAT_URL"]) ? esc_url($env["LENOCHAT_URL"]) : "";
    $forgot_password_url = isset($env["LENOCHAT_APP_URL"]) ? esc_url(rtrim($env["LENOCHAT_APP_URL"], "/") . "/auth/forgot-password") : "";
    $about_us_url = isset($env["LENOCHAT_URL"]) ? esc_url(rtrim($env["LENOCHAT_URL"], "/") . "/about-us") : "";
    $help_url = isset($env["LENOCHAT_URL"]) ? esc_url(rtrim($env["LENOCHAT_URL"], "/") . "/help-center") : "";
    $privacy_url = isset($env["LENOCHAT_URL"]) ? esc_url(rtrim($env["LENOCHAT_URL"], "/") . "/privacy-policy") : "";
    $banner_url = isset($env["LENOCHAT_LOGO_URL"]) ? esc_url($env["LENOCHAT_LOGO_URL"]) : "";
} catch (Exception $e) {
    $website_url = "https://www.lenochat.com";
    $forgot_password_url = "https://www.lenochat.com/auth/forgot-password";
    $about_us_url = "https://www.lenochat.com/about-us";
    $help_url = "https://www.lenochat.com/help-center";
    $privacy_url = "https://www.lenochat.com/privacy-policy";
    $banner_url = "";
}

if ($license === false) {
    add_option("lenochat_license", "");
    $license = "";
}

if ($lang == "tr") {
    require_once realpath(dirname(__FILE__) . "/..") . "/language/tr.php";
} else {
    require_once realpath(dirname(__FILE__) . "/..") . "/language/en.php";
}

if (isset($_POST["lenochat_login_nonce"])) {
    $nonce = sanitize_text_field(wp_unslash($_POST["lenochat_login_nonce"]));
    if (wp_verify_nonce($nonce, "lenochat_login_action")) {
        if (isset($_POST["email"]) && isset($_POST["password"])) {
            try {
                $env = lenochat_load_env_func();
                $url = $env["AUTH_URL"] ?? null;

                if (!$url) {
                    throw new Exception("Auth URL not found");
                }

                $email = sanitize_email(wp_unslash($_POST["email"]));
                $password = sanitize_text_field(wp_unslash($_POST["password"]));

                $fields = ["email" => $email, "password" => $password];
                $jsonstring = json_encode($fields);

                $response = wp_remote_post($url, [
                    "body" => $jsonstring,
                    "headers" => ["Content-Type" => "application/json"],
                    "timeout" => 30,
                ]);

                if (is_wp_error($response)) {
                    throw new Exception($response->get_error_message());
                }

                $result = wp_remote_retrieve_body($response);
                $arr = json_decode($result, true);

                if (isset($arr["success"]) && $arr["success"]) {
                    update_option("lenochat_license", $arr["data"][0]["license"]);
                    if (isset($arr["data"][0]["company_id"])) {
                        update_option("lenochat_company_id", $arr["data"][0]["company_id"]);
                    }
                    $encodedToken = json_encode($arr["data"]);
                    $license = get_option("lenochat_license");
                } else {
                    echo '<div class="notice notice-error"><p>' . esc_html__("Invalid credentials. Please try again.", "lenochat") . "</p></div>";
                }
            } catch (Exception $e) {
                echo '<div class="notice notice-error"><p>' . esc_html__("Error: ", "lenochat") . esc_html($e->getMessage()) . "</p></div>";
            }
        }

        if (isset($_POST["switchLanguageLogin"]) && isset($_POST["languageLogin"])) {
            $language = sanitize_text_field(wp_unslash($_POST["languageLogin"]));
            if (get_option("lenochat_language") == null) {
                add_option("lenochat_language", $language);
            } else {
                update_option("lenochat_language", $language);
            }
            $lenochat_needs_redirect = true;
        }
    }
}

if (isset($_POST["lenochat_settings_nonce"])) {
    $settings_nonce = sanitize_text_field(wp_unslash($_POST["lenochat_settings_nonce"]));
    if (wp_verify_nonce($settings_nonce, "lenochat_settings_action")) {
        if (isset($_POST["toggleSubmit"])) {
            $widgetStatus = isset($_POST["widgetStatus"]) ? sanitize_text_field(wp_unslash($_POST["widgetStatus"])) : "0";
            update_option("lenochat_status", $widgetStatus);
            $status = $widgetStatus;
            
            // If widget is being activated (status = "1"), call activation endpoint
            if ($widgetStatus == "1") {
                try {
                    $env = lenochat_load_env_func();
                    $activate_url = isset($env["LENOCHAT_ACTIVATE_WORDPRESS_INTEGRATION"]) ? $env["LENOCHAT_ACTIVATE_WORDPRESS_INTEGRATION"] : null;
                    
                    if ($activate_url) {
                        $license = get_option("lenochat_license");
                        $company_id = get_option("lenochat_company_id");
                        
                        if (!empty($license) && !empty($company_id)) {
                            $request_body = [
                                "license" => intval($license),
                                "platform" => 20,
                                "companyId" => intval($company_id)
                            ];
                            
                            $response = wp_remote_post($activate_url, [
                                "body" => json_encode($request_body),
                                "headers" => ["Content-Type" => "application/json"],
                                "timeout" => 30,
                            ]);
                            
                            if (is_wp_error($response)) {
                                // Log error but don't block the toggle
                                error_log("LenoChat activation error: " . $response->get_error_message());
                            }
                        }
                    }
                } catch (Exception $e) {
                    // Log error but don't block the toggle
                    error_log("LenoChat activation exception: " . $e->getMessage());
                }
            }
            
            $lenochat_needs_redirect = true;
        }

        if (isset($_POST["switchLanguageSettings"]) && isset($_POST["languageSettings"])) {
            $language = sanitize_text_field(wp_unslash($_POST["languageSettings"]));
            if (get_option("lenochat_language") === false) {
                add_option("lenochat_language", $language);
            } else {
                update_option("lenochat_language", $language);
            }
            $lenochat_needs_redirect = true;
        }
    }
}

if ($lenochat_needs_redirect) {
    wp_register_script("lenochat-redirect", "", [], LENOCHAT_VERSION, true);
    wp_enqueue_script("lenochat-redirect");
    wp_add_inline_script("lenochat-redirect", "window.location.href = window.location.pathname + window.location.search;");
    return;
}

wp_enqueue_style("lenochat-fonts", plugins_url("admin/assets/css/font.css", dirname(__FILE__, 2)), [], LENOCHAT_VERSION);
wp_enqueue_style("lenochat-main-style", plugins_url("admin/assets/css/main.css", dirname(__FILE__, 2)), ["lenochat-fonts"], LENOCHAT_VERSION);

wp_register_script("lenochat-login-js", "", ["jquery"], LENOCHAT_VERSION, true);
wp_enqueue_script("lenochat-login-js");
wp_add_inline_script(
    "lenochat-login-js",
    "jQuery(document).ready(function($) {
    function checkExtensionStatus() {
        var status = '" .
        esc_js($status) .
        "';
        $('#widgetVisibilityToggle').prop('checked', status === '1');
    }

    function checkLoginStatus() {
        var js_variable = '" .
        esc_js($encodedToken) .
        "';
        var localData = localStorage.getItem('token');
        if (js_variable) {
            localStorage.setItem('token', js_variable);
            localData = js_variable;
        }

        if (localData) {
            $('#loginContainer').hide();
            $('#licenseContainer').show();
        } else {
            $('#licenseContainer').hide();
            $('#loginContainer').show();
        }
    }

    function setLanguageSelection() {
        var lang = '" .
        esc_js($lang) .
        "';
        if (lang === 'tr') {
            $('#lenochat-login-language-selector').val('tr');
            $('#lenochat-settings-language-selector').val('tr');
        } else {
            $('#lenochat-login-language-selector').val('en');
            $('#lenochat-settings-language-selector').val('en');
        }
    }

    checkExtensionStatus();
    checkLoginStatus();
    setLanguageSelection();

    $('#lenochat-login-language-selector').on('change', function() {
        $('#languageLogin').val($(this).val());
        $('#switchLanguageLogin').click();
    });

    $('#lenochat-settings-language-selector').on('change', function() {
        $('#languageSettings').val($(this).val());
        $('#switchLanguageSettings').click();
    });

    $('#widgetVisibilityToggle').on('change', function() {
        $('#widgetStatus').val($(this).prop('checked') ? '1' : '0');
        $('#toggleSubmit').click();
    });

    window.lenochat_logout = function () {
        localStorage.removeItem('token');
        window.location.reload();
    };
});"
);
?>

<!-- HTML layout starts below -->
<div class="wrap lenochat-admin">
    <meta charset="UTF-8">
    <meta http-equiv="X-UA-Compatible" content="ie=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">

    <main class="main" id="loginContainer">
        <div class="wrapper">
            <form method="POST" class="form" name="loginform" id="loginform">
                <?php wp_nonce_field("lenochat_login_action", "lenochat_login_nonce"); ?>
                <div class="banner">
                    <a href="<?php echo esc_url($website_url); ?>" target="_blank">
                        <img class="lenochatBanner" 
                             src="<?php echo esc_url(plugins_url("admin/assets/images/lenochatbanner.svg", dirname(__FILE__, 2))); ?>"
                             alt="LenoChat">
                    </a>
                </div>
                <div class="loginForm">
                    <div class="fieldContainer">
                        <div>
                            <div class="inputLabel"><?php echo esc_html($messages["Email"] ?? "Email"); ?></div>
                            <input class="inputField" type="email" name="email" id="email" placeholder="<?php echo esc_attr($messages["EmailPlaceholder"] ?? "Enter your email"); ?>">
                        </div>
                        <div>
                            <div class="passwordContainer">
                                <div class="inputLabel"><?php echo esc_html($messages["Password"] ?? "Password"); ?></div>
                                <a class="link" href="<?php echo esc_url($forgot_password_url); ?>" target="_blank">
                                    <?php echo esc_html($messages["ForgotPassword"] ?? "Forgot Password?"); ?>
                                </a>
                            </div>
                            <input class="inputField" type="password" name="password" id="password" placeholder="<?php echo esc_attr($messages["PasswordPlaceholder"] ?? "Enter your password"); ?>">
                        </div>
                    </div>
                    <input class="inputSubmit" type="submit" name="submit" value="<?php echo esc_attr($messages["Login"] ?? "Login"); ?>">
                </div>

                <div class="languageMenu">
                    <select name="language" id="lenochat-login-language-selector">
                        <option value="en"><?php echo esc_html($messages["English"] ?? "English"); ?></option>
                        <option value="tr"><?php echo esc_html($messages["Turkish"] ?? "Turkish"); ?></option>
                    </select>
                    <input type="hidden" id="languageLogin" name="languageLogin" value="<?php echo esc_attr($lang); ?>">
                    <button type="submit" id="switchLanguageLogin" name="switchLanguageLogin" style="display:none;">Switch</button>
                </div>
            </form>
            <div class="linkContainer">
                <a class="link" href="<?php echo esc_url($about_us_url); ?>" target="_blank">
                    <?php echo esc_html($messages["About"] ?? "About"); ?>
                </a>
                <a class="link" href="<?php echo esc_url($privacy_url); ?>" target="_blank">
                    <?php echo esc_html($messages["PrivacyPolicy"] ?? "Help"); ?>
                </a>
                <a class="link" href="<?php echo esc_url($help_url); ?>" target="_blank">
                    <?php echo esc_html($messages["Help"] ?? "Help"); ?>
                </a>
            </div>
            <div class="linkContainer">
                <a class="link" href="<?php echo esc_url($website_url); ?>" target="_blank">
                    LenoChat © 2025
                </a>
            </div>
        </div>
    </main>

    <main class="main" id="licenseContainer" style="display:none;">
        <div class="wrapper" style="max-width: 750px">
            <div class="banner">
                <a href="<?php echo esc_url($website_url); ?>" target="_blank">
                    <img class="lenochatBanner" 
                         src="<?php echo esc_url(plugins_url("admin/assets/images/lenochatbanner.svg", dirname(__FILE__, 2))); ?>"
                         alt="LenoChat">
                </a>
            </div>
            <form method="POST" class="form" name="settingsform" id="settingsform">
                <?php wp_nonce_field("lenochat_settings_action", "lenochat_settings_nonce"); ?>
                <div class="description">
                    <?php echo esc_html($messages["YourLicense"] ?? "Your License"); ?> <span class="licenseNumber"><?php echo esc_html($license); ?></span>.
                </div>
                <div class="description">
                    <?php echo esc_html($messages["DescriptionMessage"] ?? ""); ?>
                </div>
                <div class="widgetToggleContainer">
                    <div class="toggleLabel"><?php echo esc_html($messages["WidgetVisibility"] ?? "Widget Visibility"); ?></div>
                    <label class="switch">
                        <input type="checkbox" id="widgetVisibilityToggle" <?php echo $status == "1" ? "checked" : ""; ?>>
                        <span class="slider round"></span>
                    </label>
                    <input type="hidden" name="widgetStatus" id="widgetStatus" value="<?php echo esc_attr($status); ?>">
                    <button type="submit" id="toggleSubmit" name="toggleSubmit" style="display:none;">Save</button>
                </div>
                <div class="languageMenu">
                    <select name="language" id="lenochat-settings-language-selector">
                        <option value="en"><?php echo esc_html($messages["English"] ?? "English"); ?></option>
                        <option value="tr"><?php echo esc_html($messages["Turkish"] ?? "Turkish"); ?></option>
                    </select>
                    <input type="hidden" id="languageSettings" name="languageSettings" value="<?php echo esc_attr($lang); ?>">
                    <button type="submit" id="switchLanguageSettings" name="switchLanguageSettings" style="display:none;">Switch</button>
                </div>
            </form>
            <div class="linkContainer">
                <a class="link" href="<?php echo esc_url($about_us_url); ?>" target="_blank">
                    <?php echo esc_html($messages["About"] ?? "About"); ?>
                </a>
                <a class="link" href="<?php echo esc_url($privacy_url); ?>" target="_blank">
                    <?php echo esc_html($messages["PrivacyPolicy"] ?? "Help"); ?>
                </a>
                <a class="link" href="<?php echo esc_url($help_url); ?>" target="_blank">
                    <?php echo esc_html($messages["Help"] ?? "Help"); ?>
                </a>
            </div>
            <div class="linkContainer">
                <a class="link" href="<?php echo esc_url($website_url); ?>" target="_blank">
                    LenoChat © 2025
                </a>
            </div>
        </div>
        <div class="logoutButtonContainer">
            <button class="logoutButton" type="button" onclick="lenochat_logout()"><?php echo esc_html($messages["Logout"] ?? "Logout"); ?></button>
        </div>
    </main>
</div>
