<?php /**
 * Creates the top-level menu item for the plugin.
 *
 * Registers a new top-level menu item in the WordPress admin sidebar and uses
 * the dependency passed into the constructor in order to display the page
 * corresponding to this menu item.
 */
class LenoChat_Submenu
{
    /**
     * A reference the class responsible for rendering the menu page.
     *
     * @var    Submenu_Page
     * @access private
     */
    private $submenu_page;

    /**
     * Initializes all of the partial classes.
     *
     * @param Submenu_Page $submenu_page A reference to the class that renders the
     * page for the plugin.
     */
    public function __construct($submenu_page)
    {
        $this->submenu_page = $submenu_page;
    }

    /**
     * Adds a top-level menu for this plugin to the WordPress admin sidebar.
     */
    public function init()
    {
        add_action("admin_menu", [$this, "add_menu_page"]);
        add_action("admin_head", [$this, "add_menu_icon_styles"]);
    }

    /**
     * Creates the top-level menu item and calls on the Submenu Page object to render
     * the actual contents of the page.
     */
    public function add_menu_page()
    {
        // Get menu icon URL from environment config
        $menu_icon = "dashicons-format-chat"; // Default fallback
        try {
            require_once plugin_dir_path(__FILE__) . "lenochat-env-functions.php";
            $env = lenochat_load_env_func();
            if (isset($env["LENOCHAT_MENU_ICON_URL"]) && !empty($env["LENOCHAT_MENU_ICON_URL"])) {
                $menu_icon = esc_url($env["LENOCHAT_MENU_ICON_URL"]);
            }
        } catch (Exception $e) {
            // Use default icon if env loading fails
        }

        add_menu_page(
            "LenoChat",
            "LenoChat",
            "manage_options",
            "lenochat",
            [$this->submenu_page, "render"],
            $menu_icon,
            30
        );
    }

    /**
     * Adds custom CSS styles for the menu icon.
     */
    public function add_menu_icon_styles()
    {
        echo '<style>
            #toplevel_page_lenochat .wp-menu-image img {
                width: 20px;
                height: 20px;
                padding: 0px;
                margin: 7px 0;
                border-radius: 3px;
            }
        </style>';
    }
}
