<?php
/**
 * Shortcodes handler class
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class LVNAP_Shortcodes {

    /**
     * Instance of this class
     */
    private static $instance = null;

    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        add_action('init', array($this, 'register_shortcodes'));
    }

    /**
     * Register shortcodes
     */
    public function register_shortcodes() {
        add_shortcode('lean_video_player', array($this, 'video_player_shortcode'));
        add_shortcode('lean_audio_player', array($this, 'audio_player_shortcode'));
    }

    /**
     * Video player shortcode
     */
    public function video_player_shortcode($atts) {
        // Default attributes
        $atts = shortcode_atts(array(
            'type'     => 'youtube',
            'id'       => '',
            'autoplay' => 'false',
            'muted'    => 'false',
        ), $atts, 'lean_video_player');

        // Sanitize attributes
        $type = sanitize_text_field($atts['type']);
        $id = sanitize_text_field($atts['id']);
        $autoplay = $atts['autoplay'] === 'true' ? 'true' : 'false';
        $muted = $atts['muted'] === 'true' ? 'true' : 'false';

        // Validate required fields
        if (empty($id)) {
            return '<div class="lvnap-error">' . esc_html__('Video ID is required.', 'lean-video-and-audio-player') . '</div>';
        }

        // Validate type
        if (!in_array($type, array('youtube', 'vimeo', 'html5'))) {
            return '<div class="lvnap-error">' . esc_html__('Invalid video type. Supported types: youtube, vimeo, html5.', 'lean-video-and-audio-player') . '</div>';
        }

        // Generate unique ID for this player
        $player_id = 'lvnap-video-' . uniqid();

        // Build data settings
        $data_settings = array(
            'muted' => $muted,
            'debug_mode' => 'false'
        );

        ob_start();

        if ($type === 'youtube') {
            ?>
            <div class="lvnap-player lvnap-video plyr__video-embed" 
                 id="<?php echo esc_attr($player_id); ?>"
                 data-settings='<?php echo wp_json_encode($data_settings); ?>'>
                <iframe
                    src="https://www.youtube.com/embed/<?php echo esc_attr($id); ?>?autoplay=<?php echo esc_attr($autoplay === 'true' ? '1' : '0'); ?>&amp;origin=<?php echo esc_url(home_url()); ?>&amp;iv_load_policy=3&amp;modestbranding=1&amp;playsinline=1&amp;showinfo=0&amp;rel=0&amp;enablejsapi=1"
                    allowfullscreen
                    allowtransparency
                    allow="autoplay">
                </iframe>
            </div>
            <?php
        } elseif ($type === 'vimeo') {
            ?>
            <div class="lvnap-player lvnap-video plyr__video-embed" 
                 id="<?php echo esc_attr($player_id); ?>"
                 data-settings='<?php echo wp_json_encode($data_settings); ?>'>
                <iframe
                    src="https://player.vimeo.com/video/<?php echo esc_attr($id); ?>?autoplay=<?php echo esc_attr($autoplay === 'true' ? '1' : '0'); ?>&amp;byline=false&amp;portrait=false&amp;title=false&amp;speed=true&amp;transparent=0&amp;gesture=media"
                    allowfullscreen
                    allowtransparency
                    allow="autoplay">
                </iframe>
            </div>
            <?php
        } elseif ($type === 'html5') {
            ?>
            <video
                class="lvnap-player lvnap-video"
                id="<?php echo esc_attr($player_id); ?>"
                data-settings='<?php echo wp_json_encode($data_settings); ?>'
                <?php echo $autoplay === 'true' ? 'autoplay' : ''; ?>
                <?php echo $muted === 'true' ? 'muted' : ''; ?>
                controls>
                <source src="<?php echo esc_url($id); ?>" type="video/mp4">
                <p><?php echo esc_html__('Your browser doesn\'t support HTML5 video.', 'lean-video-and-audio-player'); ?></p>
            </video>
            <?php
        }

        return ob_get_clean();
    }

    /**
     * Audio player shortcode
     */
    public function audio_player_shortcode($atts) {
        // Default attributes
        $atts = shortcode_atts(array(
            'src'      => '',
            'autoplay' => 'false',
            'muted'    => 'false',
        ), $atts, 'lean_audio_player');

        // Sanitize attributes
        $src = esc_url($atts['src']);
        $autoplay = $atts['autoplay'] === 'true' ? 'true' : 'false';
        $muted = $atts['muted'] === 'true' ? 'true' : 'false';

        // Validate required fields
        if (empty($src)) {
            return '<div class="lvnap-error">' . esc_html__('Audio source URL is required.', 'lean-video-and-audio-player') . '</div>';
        }

        // Generate unique ID for this player
        $player_id = 'lvnap-audio-' . uniqid();

        // Get file extension
        $file_ext = pathinfo($src, PATHINFO_EXTENSION);
        if (empty($file_ext)) {
            $file_ext = 'mp3'; // Default fallback
        }

        // Build data settings
        $data_settings = array(
            'muted' => $muted,
            'debug_mode' => 'false'
        );

        ob_start();
        ?>
        <audio
            class="lvnap-player lvnap-audio" 
            id="<?php echo esc_attr($player_id); ?>"
            data-settings='<?php echo wp_json_encode($data_settings); ?>' 
            <?php echo $autoplay === 'true' ? 'autoplay' : ''; ?>
            <?php echo $muted === 'true' ? 'muted' : ''; ?>
            controls>
            <source src="<?php echo esc_url($src); ?>" type="audio/<?php echo esc_attr($file_ext); ?>">
            <p><?php echo esc_html__('Your browser doesn\'t support the audio element.', 'lean-video-and-audio-player'); ?></p>
        </audio>
        <?php

        return ob_get_clean();
    }
}

LVNAP_Shortcodes::get_instance();