(function($) {
    'use strict';

    /**
     * Initialize Lean Video Players
     */
    function initLeanVideoPlayers() {
        var videoPlayers = document.querySelectorAll('.lvnap-player.lvnap-video');
        
        if (videoPlayers.length === 0) {
            return;
        }

        videoPlayers.forEach(function(playerElement) {
            // Skip if already initialized
            if (playerElement.classList.contains('lvnap-initialized')) {
                return;
            }

            try {
                // Get settings from data attribute
                var settingsData = playerElement.getAttribute('data-settings');
                var settings = settingsData ? JSON.parse(settingsData) : {};

                // Default settings for video
                var playerConfig = {
                    debug: settings.debug_mode === 'true' ? true : false,
                    controls: ['play-large', 'play', 'progress', 'current-time', 'mute', 'volume', 'settings', 'fullscreen'],
                    muted: settings.muted === 'true' ? true : false,
                    clickToPlay: true,
                    keyboard: { 
                        focused: true, 
                        global: false 
                    },
                    tooltips: { 
                        controls: false, 
                        seek: true 
                    },
                    fullscreen: { 
                        enabled: true, 
                        fallback: true, 
                        iosNative: false 
                    }
                };

                // Initialize Plyr
                var player = new Plyr(playerElement, playerConfig);

                // Mark as initialized
                playerElement.classList.add('lvnap-initialized');

                // Log if debug mode
                if (playerConfig.debug) {
                    console.log('Lean Video Player initialized:', playerElement.id, player);
                }

            } catch (error) {
                console.error('Error initializing Lean Video Player:', error);
            }
        });
    }

    /**
     * Initialize Lean Audio Players
     */
    function initLeanAudioPlayers() {
        var audioPlayers = document.querySelectorAll('.lvnap-player.lvnap-audio');
        
        if (audioPlayers.length === 0) {
            return;
        }

        audioPlayers.forEach(function(playerElement) {
            // Skip if already initialized
            if (playerElement.classList.contains('lvnap-initialized')) {
                return;
            }

            try {
                // Get settings from data attribute
                var settingsData = playerElement.getAttribute('data-settings');
                var settings = settingsData ? JSON.parse(settingsData) : {};

                // Default settings for audio
                var playerConfig = {
                    debug: settings.debug_mode === 'true' ? true : false,
                    controls: ['play', 'progress', 'current-time', 'mute', 'volume', 'settings'],
                    muted: settings.muted === 'true' ? true : false,
                    keyboard: { 
                        focused: true, 
                        global: false 
                    },
                    tooltips: { 
                        controls: false, 
                        seek: true 
                    }
                };

                // Initialize Plyr
                var player = new Plyr(playerElement, playerConfig);

                // Mark as initialized
                playerElement.classList.add('lvnap-initialized');

                // Log if debug mode
                if (playerConfig.debug) {
                    console.log('Lean Audio Player initialized:', playerElement.id, player);
                }

            } catch (error) {
                console.error('Error initializing Lean Audio Player:', error);
            }
        });
    }

    /**
     * Initialize all players
     */
    function initAllPlayers() {
        initLeanVideoPlayers();
        initLeanAudioPlayers();
    }

    /**
     * Setup MutationObserver safely
     */
    function setupMutationObserver() {
        if (!window.MutationObserver) {
            return;
        }

        var observer = new MutationObserver(function(mutations) {
            var shouldReinit = false;
            
            mutations.forEach(function(mutation) {
                if (mutation.type === 'childList') {
                    mutation.addedNodes.forEach(function(node) {
                        if (node.nodeType === 1) { // Element node
                            if (node.classList && (
                                node.classList.contains('lvnap-player') ||
                                node.querySelector('.lvnap-player')
                            )) {
                                shouldReinit = true;
                            }
                        }
                    });
                }
            });

            if (shouldReinit) {
                // Small delay to ensure content is fully loaded
                setTimeout(initAllPlayers, 100);
            }
        });

        // Function to start observing with proper error handling
        function startObserving() {
            var targetNode = document.body || document.documentElement;
            
            if (targetNode && typeof targetNode.nodeType !== 'undefined' && targetNode.nodeType === 1) {
                try {
                    observer.observe(targetNode, {
                        childList: true,
                        subtree: true
                    });
                } catch (error) {
                    console.error('Error setting up MutationObserver:', error);
                }
            } else {
                // If body is not available yet, wait a bit and try again
                setTimeout(startObserving, 50);
            }
        }

        // Start observing
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', startObserving);
        } else {
            startObserving();
        }
    }

    /**
     * Document ready
     */
    $(document).ready(function() {
        // Initialize players when DOM is ready
        initAllPlayers();

        // Re-initialize when new content is loaded (for AJAX, page builders, etc.)
        $(document).on('DOMContentLoaded', initAllPlayers);
        
        // For page builders and dynamic content
        if (typeof window.elementorFrontend !== 'undefined') {
            window.elementorFrontend.hooks.addAction('frontend/element_ready/widget', initAllPlayers);
        }

        // Setup MutationObserver for dynamic content
        setupMutationObserver();
    });

    // Expose functions globally for manual initialization if needed
    window.LVNAP = {
        initVideoPlayers: initLeanVideoPlayers,
        initAudioPlayers: initLeanAudioPlayers,
        initAllPlayers: initAllPlayers
    };

})(jQuery);