<?php
/* ======================================================
   Scripts and Styles for Lead Management Hub
====================================================== */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Register and enqueue admin assets
 * 
 * @param string $hook The current admin page hook suffix
 */
function leadmahu_enqueue_admin_assets($hook)
{
    $allowed_hooks = array(
        'toplevel_page_leadmahu_dashboard',
        'leadmahu_page_leadmahu_general',
        'leadmahu_page_leadmahu_view_leads',
        'leadmahu_page_leadmahu_edit_lead',
        'leadmahu_page_leadmahu_form_generator',
        'leadmahu_page_leadmahu_form_list',
        'leadmahu_page_leadmahu_edit_form',
        'lead-hub_page_leadmahu_dashboard',
        'lead-hub_page_leadmahu_general',
        'lead-hub_page_leadmahu_view_leads',
        'lead-hub_page_leadmahu_edit_lead',
        'lead-hub_page_leadmahu_premium',
        'lead-hub_page_leadmahu_form_generator',
        'lead-hub_page_leadmahu_form_list',
        'lead-hub_page_leadmahu_edit_form'
    );
    
    // Load our assets on all plugin pages or any page that starts with our prefix
    if (!in_array($hook, $allowed_hooks) && strpos($hook, 'leadmahu_') === false && strpos($hook, 'lead-hub_page_') === false) {
        return;
    }
      // Register and enqueue admin styles
    wp_register_style(
        'leadmahu_admin_css', 
        plugin_dir_url(dirname(__FILE__)) . 'assets/css/admin-styles.css', 
        array(), 
        '1.0.3'
    );
    wp_enqueue_style('leadmahu_admin_css');
    
    // Register and enqueue our dashboard fix CSS (always load this to ensure dashboard styling)
    wp_register_style(
        'leadmahu_dashboard_fix_css', 
        plugin_dir_url(dirname(__FILE__)) . 'assets/css/dashboard-fix.css', 
        array('leadmahu_admin_css'), 
        '1.0.0'
    );
    wp_enqueue_style('leadmahu_dashboard_fix_css');
    
    // Add dynamic CSS for status colors
    $status_colors = array(
        'nuevo' => '#00a32a',
        'contactado' => '#ff9800',
        'convertido' => '#0073aa',
        'perdido' => '#d63638'
    );
    
    // Create dynamic CSS for status colors
    $dynamic_css = '';
    foreach ($status_colors as $status => $color) {
        $dynamic_css .= ".lead-info-value.status.$status { border-left-color: $color; }\n";
        $dynamic_css .= ".leadmahu-kpi-card.$status { border-left-color: $color; }\n";
    }
    
    // Add dynamic CSS for lead detail views
    wp_add_inline_style('leadmahu_admin_css', $dynamic_css);
    
    // Register and enqueue admin scripts
    wp_register_script(
        'leadmahu_admin_js', 
        plugin_dir_url(dirname(__FILE__)) . 'assets/js/admin-scripts.js', 
        array('jquery'), 
        '1.0.3', 
        true
    );
    
    // Add nonce and other data for AJAX requests
    wp_localize_script('leadmahu_admin_js', 'leadmahu_params', array(
        'filter_nonce' => wp_create_nonce('leadmahu_filter_nonce'),
        'ajaxurl' => admin_url('admin-ajax.php'),
        'plugin_url' => plugin_dir_url(dirname(__FILE__)),
        'status_colors' => $status_colors,
        'hook' => $hook
    ));
    
    wp_enqueue_script('leadmahu_admin_js');
    
    // Specific script for KPIs page
    if (strpos($hook, 'leadmahu_kpis') !== false) {
        wp_register_script(
            'chartjs', 
            plugin_dir_url(dirname(__FILE__)) . 'assets/js/chart.js', 
            array(), 
            '3.9.1', 
            true
        );
        wp_enqueue_script('chartjs');
    }
    
    // For form generator and edit form pages, add the custom fields handling script
    if (strpos($hook, 'form_generator') !== false || strpos($hook, 'edit_form') !== false) {
        // Add specific form generator scripts
        wp_add_inline_script('leadmahu_admin_js', '
            jQuery(document).ready(function($) {
                // Custom field management for form generator
                $("#add-custom-field").on("click", function() {
                    var newField = \'<div class="custom-field"><label>Nombre del Campo:</label>\' +
                        \'<input type="text" name="custom_field_names[]" required>\' +
                        \'<label>Tipo de Campo:</label>\' +
                        \'<select name="custom_field_types[]">\' +
                        \'<option value="text">Texto</option>\' +
                        \'<option value="email">Email</option>\' +
                        \'<option value="textarea">Area de Texto</option>\' +
                        \'<option value="number">Número</option>\' +
                        \'</select>\' +
                        \'<label><input type="checkbox" name="custom_field_required[]" value="1">Requerido</label>\' +
                        \'<div class="number-limits" style="display: none;">\' +
                        \'<label>Mínimo:</label><input type="number" name="custom_field_min[]" min="0">\' +
                        \'<label>Máximo:</label><input type="number" name="custom_field_max[]" min="0">\' +
                        \'</div>\' +
                        \'<button type="button" class="remove-custom-field">Eliminar</button></div>\';
                    $("#custom-fields-container").append(newField);
                });
                
                // Remove custom field button handler
                $("#custom-fields-container").on("click", ".remove-custom-field", function() {
                    $(this).parent(".custom-field").remove();
                });
                
                // Show/hide number limits based on field type
                $("#custom-fields-container").on("change", "select[name=\'custom_field_types[]\']", function() {
                    var selectedValue = $(this).val();
                    var numberLimits = $(this).siblings(".number-limits");
                    if (selectedValue === "number") {
                        numberLimits.show();
                    } else {
                        numberLimits.hide();
                    }
                });
                
                // Initialize field settings on page load
                $("select[name=\'custom_field_types[]\']").each(function() {
                    var selectedValue = $(this).val();
                    var numberLimits = $(this).siblings(".number-limits");
                    if (selectedValue === "number") {
                        numberLimits.show();
                    }
                });
            });
        ');
    }
    
    // For dashboard and leads management, add specific script
    if (strpos($hook, 'dashboard') !== false) {
        wp_add_inline_script('leadmahu_admin_js', '
            jQuery(document).ready(function($) {
                // Filter leads functionality
                $("#leadmahu_filter_button").on("click", function() {
                    var status = $("#leadmahu_filter_status").val();
                    var search = $("#leadmahu_filter_search").val();
                    var data = {
                        "action": "leadmahu_filter_leads",
                        "status": status,
                        "search": search,
                        "_wpnonce": leadmahu_params.filter_nonce
                    };
                    $.post(ajaxurl, data, function(response) {
                        $("#leadmahu_leads_table tbody").html(response);
                    });
                });
            });
        ');
    }
}
add_action('admin_enqueue_scripts', 'leadmahu_enqueue_admin_assets');

/**
 * Register and enqueue frontend assets
 */
function leadmahu_enqueue_frontend_assets() {
    wp_register_style(
        'leadmahu_frontend_css', 
        plugin_dir_url(dirname(__FILE__)) . 'assets/css/frontend.css', 
        array(), 
        '1.0.3'
    );
    wp_enqueue_style('leadmahu_frontend_css');    // Only load SweetAlert when needed
    
    // Add nonce verification but keep backward compatibility
    // This is a UI-only feature, but we're adding nonce to comply with standards
    $nonce_verified = isset($_GET['_wpnonce']) ? wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'leadmahu_form_submission') : true;
    
    if (isset($_GET['leadmahu_form_submitted']) && sanitize_text_field(wp_unslash($_GET['leadmahu_form_submitted'])) === '1' && $nonce_verified) {
        wp_enqueue_style(
            'sweetalert2', 
            plugin_dir_url(dirname(__FILE__)) . 'assets/css/sweetalert2.min.css', 
            array(), 
            '11.4.8'
        );
        
        wp_enqueue_script(
            'sweetalert2', 
            plugin_dir_url(dirname(__FILE__)) . 'assets/js/sweetalert2.min.js', 
            array(), 
            '11.4.8', 
            true
        );
        
        wp_add_inline_script('sweetalert2', '
            document.addEventListener("DOMContentLoaded", function() {
                Swal.fire({
                    title: "¡Gracias!",
                    text: "Tu mensaje ha sido enviado correctamente. Nos pondremos en contacto contigo pronto.",
                    icon: "success",
                    confirmButtonText: "OK",
                    customClass: {
                        container: "leadmahu-swal-container",
                        popup: "leadmahu-swal-popup"
                    }
                });
            });
        ');
    }
}
add_action('wp_enqueue_scripts', 'leadmahu_enqueue_frontend_assets');
