<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://codeberg.org/Klasse-Methode/wp-lauti
 * @since      1.0.0
 *
 * @package    Lauti_Calendar
 * @subpackage Lauti_Calendar/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Lauti_Calendar
 * @subpackage Lauti_Calendar/public
 * @author     Klasse & Methode <member@klasse-methode.it>
 */
class Lauti_Calendar_Public {


	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string $plugin_name       The name of the plugin.
	 * @param      string $version    The version of this plugin.
	 */
	public function __construct( $plugin_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->version     = $version;
	}

	/**
	 * Register the stylesheets for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Lauti_Calendar_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Lauti_Calendar_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/lauti-calendar-public.css', array(), $this->version, 'all' );
	}

	/**
	 * Register the JavaScript for the public-facing side of the site.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts() {

		/**
		 * This function is provided for demonstration purposes only.
		 *
		 * An instance of this class should be passed to the run() function
		 * defined in Lauti_Calendar_Loader as all of the hooks are defined
		 * in that particular class.
		 *
		 * The Lauti_Calendar_Loader will then create the relationship
		 * between the defined hooks and the functions defined in this
		 * class.
		 */

		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/lauti-calendar-public.js', array( 'jquery' ), $this->version, false );
	}

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @param      string $info    The info to check.
	 * @param      array  $infos    Array of infos to show.
	 */
	private function show_info( $info, $infos ) {
		return in_array( $info, $infos, true );
	}

	/**
	 * Register full shortcode
	 *
	 * @since    1.0.0
	 *
	 * @param   object $atts    The shortcode attributes.
	 * @example [lauti-calendar-list groupID=my-id title=my-title infos=time,location,organizers,involved,description link=true]
	 */
	public function lauti_calendar_list_shortcode( $atts ) {
		$options = get_option( 'lauti-calendar' );

		$place_id = $atts['placeid'] ?? null;
		$group_id = $atts['groupid'] ?? null;
		$title    = $atts['title'] ?? '';
		$infos    = $atts['infos'] ? explode( ',', $atts['infos'] ) : array( 'time', 'location', 'organizers, involved', 'description' );

		$start     = new DateTime();
		$start_str = str_replace( '+', 'Z', $start->format( \DateTime::RFC3339 ) );
		$start_str = str_replace( '00:00', '', $start_str );
		$params    = array(
			'placeIDs'    => $place_id,
			'groupIDs'    => $group_id,
			'sort'        => 'start',
			'startMin'    => $start_str,
			'multidayMax' => 0,
		);
		$events    = $this->load_events( $params );
		if ( null === $events ) {
			return;
		}
		foreach ( $events as $key => $event ) {
			$events[ $key ] = $this->convert_event( $event );
			if ( array_key_exists( 'link', $atts ) && 'true' === $atts['link'] ) {
				$events[ $key ]->link = $options['instance-url'] . '/event/' . $events[ $key ]->id;
			}
		}

		ob_start();

		require_once plugin_dir_path( __FILE__ ) . 'partials/lauti-calendar-list.php';

		return ob_get_clean();
	}

	/**
	 * Register full shortcode
	 *
	 * @since    1.0.0
	 *
	 * @param   object $atts    The shortcode attributes.
	 * @example [lauti-calendar-timetable id=my-id places="foo,bar" startHour=10 endHour=23]
	 */
	public function lauti_calendar_timetable_shortcode( $atts ) {
		$options = get_option( 'lauti-calendar' );

		$start_hour = intval( $atts['starthour'] );
		$end_hour   = intval( $atts['endhour'] );

		$places = explode( ',', $atts['places'] );
		$events = $this->load_events(
			array(
				'id'          => $atts['id'],
				'multidayMax' => 0,
			)
		);

		if ( null === $events ) {
			return;
		}
		if ( count( $events ) === 0 ) {
			return;
		}
		if ( property_exists( $events[0], 'children' ) ) {
			$events = $events[0]->children;
		}

		$days = array();

		$current_day       = new DateTime();
		$current_day       = $current_day->format( 'Y-m-d' );
		$found_current_day = false;

		foreach ( $events as $raw_event ) {
			$start = new DateTime( $raw_event->start );
			$end   = new DateTime( $raw_event->end );

			// Create a new day, if it doesn't exit yet.
			$day = $start->format( 'Y-m-d' );
			if ( ! array_key_exists( $day, $days ) ) {
				$start_day = new DateTime( $day );
				$start_day->setTime( $start_hour, 00 );

				$end_day = new DateTime( $day );
				$end_day->setTime( $end_hour, 00 );

				// Check if the $day is the current day.
				$is_current_day = $current_day === $day;
				if ( $is_current_day ) {
					$found_current_day = true;
				}

				$days[ $day ] = array(
					'title'  => IntlDateFormatter::formatObject( $start, 'eeee dd.MM', get_locale() ),
					'height' => $this->timetable_y_offset( $end_day, $start_hour ) - $this->timetable_y_offset( $start_day, $start_hour ),
					'start'  => $start_day,
					'end'    => $end_day,
					'active' => $is_current_day,
					'events' => array(),
				);
			}

			$event = $this->convert_event( $raw_event );
			// col is set to "all", if the place was not found in the $places array.
			$col = false;
			if ( property_exists( $event, 'location' ) ) {
				$col = array_search( $event->location, $places, true );
			}
			$event->col    = false === $col ? 'all' : $col + 1;
			$event->top    = $this->timetable_y_offset( $start, $start_hour );
			$event->height = $this->timetable_y_offset( $end, $start_hour ) - $event->top;

			$days[ $day ]['events'][] = $event;
		}

		// Set the first day as active if none of the days is the current one.
		if ( ! $found_current_day ) {
			$days[ array_key_first( $days ) ]['active'] = true;
		}

		ob_start();

		require_once plugin_dir_path( __FILE__ ) . 'partials/lauti-calendar-timetable.php';

		return ob_get_clean();
	}

	/**
	 * Converts a raw event from the API to a more usable format.
	 * See https://codeberg.org/Klasse-Methode/lauti/src/branch/main/service/event/event.go for reference.
	 *
	 * @param object $raw_event The raw event from the API.
	 *
	 * @return object The converted event.
	 */
	private function convert_event( $raw_event ) {
		$event           = new stdClass();
		$event->id       = $raw_event->id;
		$event->canceled = $raw_event->canceled;
		$event->name     = $raw_event->name;
		if ( strlen( $raw_event->location2 ) > 0 ) {
			$event->location = $raw_event->location2;
		}
		$event->organizers = array();

		foreach ( $raw_event->organizers as $organizer ) {
			if ( property_exists( $organizer, 'group' ) ) {
				$event->organizers[] = $organizer->group->name;
				continue;
			}

			if ( strlen( $organizer->name ) > 0 ) {
				// add organizer to array.
				$event->organizers[] = $organizer->name;
			}
		}
		$event->involved    = $raw_event->involved;
		$event->description = $raw_event->description;
		$event->tags        = $raw_event->tags;
		$event->start       = new DateTime( $raw_event->start );
		$event->end         = null !== $raw_event->end && '' !== $raw_event->end ? new DateTime( $raw_event->end ) : null;
		return $event;
	}

	/**
	 * Calculates the y offset in the timetable.
	 *
	 * @param DateTime $time      The time to calculate the y offset for.
	 * @param number   $start_hour The starting hour of the timetable.
	 *
	 * @return number
	 */
	private function timetable_y_offset( $time, $start_hour ) {
		$hour    = intval( $time->format( 'H' ) ) - $start_hour;
		$minutes = intval( $time->format( 'i' ) );
		return intval( ( $hour * 60 + $minutes ) );
	}

	/**
	 * Loads all sub events for a given event id.
	 *
	 * @param array $params Eventsearch parameters.
	 *
	 * @return array|null
	 */
	private function load_events( $params ) {
		$url = $this->build_url( '/api/v1/eventsearch', $params );
		if ( null === $url ) {
			return null;
		}
		try {
			$response      = wp_remote_get( $url, array( 'timeout' => 60 ) );
			$response_code = wp_remote_retrieve_response_code( $response );
			if ( 200 === $response_code ) {
				$response_body = wp_remote_retrieve_body( $response );
				$response_body = json_decode( $response_body );
			} else {
				$error = new WP_Error( 'broke', 'could not fetch events' );
				$error->add_data(
					array(
						'url'      => $url,
						'response' => $response,
					)
				);
				return $error;
			}
		} catch ( Exception $e ) {
			return null;
		}
		return $response_body->events;
	}

	/**
	 * Builds a full URL for the Lauti instance.
	 *
	 * @param string $uri    The URI to append to the instance URL.
	 * @param array  $params The query parameters to append to the URL.
	 *
	 * @return string|null The full URL or null if the instance URL is not set.
	 */
	private function build_url( $uri, $params ) {
		$options = get_option( 'lauti-calendar' );

		if ( empty( $options['instance-url'] ) ) {
			return null;
		}
		return $options['instance-url'] . $uri . '?' . http_build_query( $params );
	}
}
