<?php
/**
 * Plugin Name: LATW AI Translator for WPML
 * Plugin URI: https://llm-automatic-translation.top/
 * Description: Automatically translate WordPress content using OpenAI LLM with WPML integration.
 * Version: 1.2.0
 * Author: Szymon Zawadzki
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: latw-ai-translator-for-wpml
 * Domain Path: /languages
 * Requires at least: 6.2
 * Tested up to: 6.9
 * Requires PHP: 7.4
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('LATWAITR_VERSION', '1.2.0');
define('LATWAITR_PLUGIN_FILE', __FILE__);
define('LATWAITR_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('LATWAITR_PLUGIN_URL', plugin_dir_url(__FILE__));
define('LATWAITR_PLUGIN_BASENAME', plugin_basename(__FILE__));
define('LATWAITR_MIN_PHP_VERSION', '7.4');
define('LATWAITR_MIN_WP_VERSION', '6.2');

// Require composer autoloader if exists
if (file_exists(LATWAITR_PLUGIN_DIR . 'vendor/autoload.php')) {
    require_once LATWAITR_PLUGIN_DIR . 'vendor/autoload.php';
}

/**
 * Main plugin class
 */
final class LATW_AI_Translator_for_WPML {

    /**
     * Single instance
     */
    private static $instance = null;

    /**
     * Plugin modules
     */
    public $admin;
    public $settings;
    public $llm;
    public $wpml;
    public $queue;
    public $logger;
    public $translator;
    public $acf;
    public $security;
    public $response_checker;

    /**
     * Get instance
     */
    public static function instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->check_requirements();
        $this->init_hooks();
    }

    /**
     * Check plugin requirements
     */
    private function check_requirements() {
        if (version_compare(PHP_VERSION, LATWAITR_MIN_PHP_VERSION, '<')) {
            add_action('admin_notices', array($this, 'php_version_notice'));
            return false;
        }

        if (version_compare(get_bloginfo('version'), LATWAITR_MIN_WP_VERSION, '<')) {
            add_action('admin_notices', array($this, 'wp_version_notice'));
            return false;
        }

        // Check if WPML is active
        if (!defined('ICL_SITEPRESS_VERSION')) {
            add_action('admin_notices', array($this, 'wpml_missing_notice'));
        }

        return true;
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        register_activation_hook(LATWAITR_PLUGIN_FILE, array($this, 'activate'));
        register_deactivation_hook(LATWAITR_PLUGIN_FILE, array($this, 'deactivate'));

        add_action('plugins_loaded', array($this, 'init'), 10);
        add_action('init', array($this, 'load_modules'), 20);

        // Database upgrade check
        add_action('admin_init', array($this, 'maybe_upgrade_db'));

        // Register custom cron schedule for response checker
        add_filter('cron_schedules', array('LATWAITR_Response_Checker', 'add_cron_schedules'));
    }

    /**
     * Initialize plugin
     */
    public function init() {
        do_action('latwaitr_before_init');

        // Initialize core components
        $this->load_dependencies();

        do_action('latwaitr_after_init');
    }

    /**
     * Load plugin dependencies
     */
    private function load_dependencies() {
        // Load utility classes
        require_once LATWAITR_PLUGIN_DIR . 'includes/class-latwaitr-database.php';
        require_once LATWAITR_PLUGIN_DIR . 'includes/class-latwaitr-logger.php';
        require_once LATWAITR_PLUGIN_DIR . 'includes/class-latwaitr-security.php';
        require_once LATWAITR_PLUGIN_DIR . 'includes/class-latwaitr-cost-estimator.php';

        // Load page builder classes
        require_once LATWAITR_PLUGIN_DIR . 'includes/page-builders/class-latwaitr-wpml-page-builder.php';

        // Load core modules
        require_once LATWAITR_PLUGIN_DIR . 'includes/class-latwaitr-llm-api.php';
        require_once LATWAITR_PLUGIN_DIR . 'includes/class-latwaitr-wpml-integration.php';
        require_once LATWAITR_PLUGIN_DIR . 'includes/class-latwaitr-queue.php';
        require_once LATWAITR_PLUGIN_DIR . 'includes/class-latwaitr-translator.php';
        require_once LATWAITR_PLUGIN_DIR . 'includes/class-latwaitr-acf-integration.php';
        require_once LATWAITR_PLUGIN_DIR . 'includes/class-latwaitr-response-checker.php';

        // Load admin
        if (is_admin()) {
            require_once LATWAITR_PLUGIN_DIR . 'admin/class-latwaitr-admin.php';
            require_once LATWAITR_PLUGIN_DIR . 'admin/class-latwaitr-settings.php';
        }

        // Load AJAX Handler
        require_once LATWAITR_PLUGIN_DIR . 'includes/class-latwaitr-ajax-handler.php';
    }

    /**
     * Load plugin modules
     */
    public function load_modules() {
        // Initialize logger first
        $this->logger = new LATWAITR_Logger();
        $this->security = new LATWAITR_Security();

        // Initialize core modules
        $this->llm = new LATWAITR_LLM_API();
        $this->wpml = new LATWAITR_WPML_Integration();
        $this->queue = new LATWAITR_Queue();
        $this->translator = new LATWAITR_Translator();
        $this->acf = new LATWAITR_ACF_Integration();
        $this->response_checker = new LATWAITR_Response_Checker();

        // Initialize admin
        if (is_admin()) {
            $this->settings = new LATWAITR_Settings();
            $this->admin = new LATWAITR_Admin();
        }

        // Initialize AJAX Handler
        new LATWAITR_AJAX_Handler();

        do_action('latwaitr_modules_loaded');
    }

    /**
     * Activation hook
     */
    public function activate() {
        // Create database tables (includes batch columns)
        LATWAITR_Database::create_tables();

        // Set default options
        $this->set_default_options();

        // Flush rewrite rules
        flush_rewrite_rules();

        // Log activation
        do_action('latwaitr_activated');
    }

    /**
     * Check and run database upgrades
     */
    public function maybe_upgrade_db() {
        if (class_exists('LATWAITR_Database')) {
            LATWAITR_Database::maybe_upgrade();
        }
    }

    /**
     * Deactivation hook
     */
    public function deactivate() {
        // Drop database tables
        LATWAITR_Database::drop_tables();

        // Clear scheduled events
        wp_clear_scheduled_hook('latwaitr_check_response_statuses');

        // Delete all plugin options
        delete_option('latwaitr_openai_api_key');
        delete_option('latwaitr_openai_model');
        delete_option('latwaitr_website_description');
        delete_option('latwaitr_target_languages');
        delete_option('latwaitr_translate_title');
        delete_option('latwaitr_translate_content');
        delete_option('latwaitr_translate_excerpt');
        delete_option('latwaitr_translate_custom_fields');
        delete_option('latwaitr_translate_seo');
        delete_option('latwaitr_translate_taxonomies');
        delete_option('latwaitr_enable_telemetry');
        delete_option('latwaitr_debug_mode');

        // Flush rewrite rules
        flush_rewrite_rules();

        do_action('latwaitr_deactivated');
    }

    /**
     * Set default options
     */
    private function set_default_options() {
        // Get all WPML languages as default target languages
        $default_target_languages = array();
        if (defined('ICL_SITEPRESS_VERSION') && function_exists('icl_get_languages')) {
            global $sitepress;
            if ($sitepress) {
                $languages = $sitepress->get_active_languages();
                if (!empty($languages)) {
                    foreach ($languages as $code => $lang) {
                        $default_target_languages[] = $code;
                    }
                }
            }
        }

        $defaults = array(
            'latwaitr_openai_api_key' => '',
            'latwaitr_target_languages' => $default_target_languages,
            'latwaitr_translate_title' => 'yes',
            'latwaitr_translate_content' => 'yes',
            'latwaitr_translate_excerpt' => 'yes',
            'latwaitr_translate_custom_fields' => 'yes',
            'latwaitr_translate_seo' => 'yes',
            'latwaitr_translate_taxonomies' => 'yes',
            'latwaitr_enable_telemetry' => 'no',
            'latwaitr_debug_mode' => 'no',
        );

        foreach ($defaults as $key => $value) {
            if (get_option($key) === false) {
                add_option($key, $value);
            }
        }
    }

    /**
     * PHP version notice
     */
    public function php_version_notice() {
        $message = sprintf(
            /* translators: 1: Required PHP version, 2: Current PHP version */
            __('LATW AI Translator requires PHP version %1$s or higher. You are running version %2$s.', 'latw-ai-translator-for-wpml'),
            LATWAITR_MIN_PHP_VERSION,
            PHP_VERSION
        );
        printf('<div class="error"><p>%s</p></div>', esc_html($message));
    }

    /**
     * WP version notice
     */
    public function wp_version_notice() {
        $message = sprintf(
            /* translators: 1: Required WordPress version, 2: Current WordPress version */
            __('LATW AI Translator requires WordPress version %1$s or higher. You are running version %2$s.', 'latw-ai-translator-for-wpml'),
            LATWAITR_MIN_WP_VERSION,
            get_bloginfo('version')
        );
        printf('<div class="error"><p>%s</p></div>', esc_html($message));
    }

    /**
     * WPML missing notice
     */
    public function wpml_missing_notice() {
        $message = __('LATW AI Translator requires WPML plugin to be installed and activated.', 'latw-ai-translator-for-wpml');
        printf('<div class="error"><p>%s</p></div>', esc_html($message));
    }
}

/**
 * Get plugin instance
 */
function LATWAITR() {
    return LATW_AI_Translator_for_WPML::instance();
}


// Initialize plugin
LATWAITR();
