<?php
/**
 * Universal WPML Page Builder integration for LATWAITR
 *
 * Detects Bricks or Elementor pages and delegates string extraction /
 * translation restoration entirely to WPML's Page Builder API.
 *
 * Flow:
 *  1. detect($post_id)            — returns builder config or null
 *  2. extract_strings($post_id)   — fires wpml_page_builder_register_strings,
 *                                   captures the flat string map
 *  3. restore_translations()      — pushes to WPML ST, fires
 *                                   wpml_page_builder_string_translated so
 *                                   the builder rebuilds its meta on the target post
 *
 * @package LATW_AI_Translator_for_WPML
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class LATWAITR_WPML_Page_Builder {

	/**
	 * @var LATWAITR_Logger
	 */
	private $logger;

	/**
	 * Per-request cache for detect() results.
	 *
	 * @var array
	 */
	private $detect_cache = array();

	public function __construct() {
		$this->logger = LATWAITR()->logger;
	}

	// -------------------------------------------------------------------------
	// Builder registry
	// -------------------------------------------------------------------------

	/**
	 * Return the list of supported page builders.
	 *
	 * Each entry is an array with:
	 *  - kind      : WPML package kind (must match what the builder uses)
	 *  - kind_slug : lowercase slug for the package
	 *  - meta_key  : postmeta key whose presence indicates this builder
	 *  - name_fn   : callable($post_id) → package name string
	 *
	 * Extend via the 'latwaitr_wpml_page_builder_registry' filter.
	 *
	 * @return array[]
	 */
	private static function get_builder_registry() {
		return apply_filters( 'latwaitr_wpml_page_builder_registry', array(
			array(
				'kind'      => 'Bricks',
				'kind_slug' => 'bricks-builder',
				'meta_key'  => '_bricks_page_content_2',
				'name_fn'   => function( $id ) { return 'bricks-' . $id; },
			),
			array(
				'kind'      => 'Elementor',
				'kind_slug' => 'elementor',
				'meta_key'  => '_elementor_data',
				'name_fn'   => function( $id ) { return (string) $id; },
			),
		) );
	}

	// -------------------------------------------------------------------------
	// Public API
	// -------------------------------------------------------------------------

	/**
	 * Detect which page builder is used for a post.
	 *
	 * Results are cached per request to avoid repeated meta and package queries.
	 *
	 * @param int $post_id
	 * @return array|null  Builder config array, or null if no supported builder found.
	 */
	public function detect( $post_id ) {
		if ( array_key_exists( $post_id, $this->detect_cache ) ) {
			return $this->detect_cache[ $post_id ];
		}
		$result = $this->do_detect( $post_id );
		$this->detect_cache[ $post_id ] = $result;
		return $result;
	}

	/**
	 * Internal detection logic (uncached).
	 *
	 * First checks the static builder registry (fast meta lookup).
	 * Falls back to querying WPML ST string packages (works for any
	 * WPML-compatible builder that has been processed at least once).
	 *
	 * @param int $post_id
	 * @return array|null
	 */
	private function do_detect( $post_id ) {
		// Fast path: check static registry (Bricks, Elementor, etc.)
		foreach ( self::get_builder_registry() as $builder ) {
			if ( ! empty( get_post_meta( $post_id, $builder['meta_key'], true ) ) ) {
				return $builder;
			}
		}
		// Slow path: query WPML ST packages (works for any WPML-compatible builder)
		if ( defined( 'WPML_ST_VERSION' ) ) {
			return $this->probe_wpml_packages( $post_id );
		}
		return null;
	}

	/**
	 * Detect builder kind via WPML ST string packages.
	 *
	 * This works for any builder that integrates with WPML (Divi, WPBakery,
	 * Beaver Builder, etc.) as long as WPML has registered packages for the
	 * post at least once (on save, WPML dashboard visit, or previous translation).
	 *
	 * @param int $post_id
	 * @return array|null  Builder config with kind/kind_slug/meta_key/name_fn, or null.
	 */
	private function probe_wpml_packages( $post_id ) {
		$packages = apply_filters( 'wpml_st_get_post_string_packages', array(), $post_id );
		foreach ( $packages as $pkg ) {
			if ( isset( $pkg->kind ) && ! empty( $pkg->kind ) ) {
				$pkg_name = $pkg->name;
				return array(
					'kind'      => $pkg->kind,
					'kind_slug' => sanitize_title( $pkg->kind ),
					'meta_key'  => null,
					'name_fn'   => function( $id ) use ( $pkg_name ) { return $pkg_name; },
				);
			}
		}
		return null;
	}

	/**
	 * Extract translatable strings by delegating to the builder's WPML integration.
	 *
	 * Fires the three-step WPML Page Builder registration sequence. The builder's
	 * own WPML integration parses the relevant meta and calls wpml_register_string
	 * for each text field. We intercept those calls (at PHP_INT_MAX priority, after
	 * WPML ST's own handler saves them to the DB) to build the string map.
	 *
	 * @param int $post_id
	 * @return array  Envelope: [ 'kind' => 'BuilderName', 'strings' => [ 'string_name' => 'original_value', ... ] ]
	 *                or empty array if no builder detected or no strings found.
	 */
	public function extract_strings( $post_id ) {
		$builder = $this->detect( $post_id );
		if ( ! $builder ) {
			return array();
		}

		$post = get_post( $post_id );
		if ( ! $post ) {
			return array();
		}

		$captured = array();

		$capture_handler = function ( $value, $name, $package ) use ( &$captured ) {
			if ( is_string( $value ) && trim( $value ) !== '' ) {
				$captured[ $name ] = $value;
			}
		};
		add_action( 'wpml_register_string', $capture_handler, PHP_INT_MAX, 3 );

		$package = $this->make_wpml_package( $post_id, $builder );

		do_action( 'wpml_start_string_package_registration', $package );
		do_action( 'wpml_page_builder_register_strings', $post, $package );
		do_action( 'wpml_delete_unused_package_strings', $package );

		remove_action( 'wpml_register_string', $capture_handler, PHP_INT_MAX );

		if ( empty( $captured ) ) {
			return array();
		}

		$this->logger->debug( sprintf( '%s: extracted strings envelope via WPML Page Builder API', $builder['kind'] ), array(
			'post_id' => $post_id,
			'count'   => count( $captured ),
			'names'   => array_keys( $captured ),
		) );

		return array(
			'kind'    => $builder['kind'],
			'strings' => $captured,
		);
	}

	/**
	 * Push translated strings to WPML and trigger the builder to rebuild its meta.
	 *
	 * Step 1 (optional): stores translations in WPML ST via icl_add_string_translation()
	 *   so that WPML's own interface stays in sync. Gracefully skipped when WPML ST
	 *   is not active or the package cannot be found.
	 *
	 * Step 2 (required): fires wpml_page_builder_string_translated. The builder hooks
	 *   into this action and rebuilds its meta on $target_post_id.
	 *
	 * @param int         $source_post_id  Original post ID (to find the WPML package).
	 * @param int         $target_post_id  Translated post ID (builder writes here).
	 * @param array       $translated      [ 'string_name' => 'translated_value', ... ]
	 * @param string      $target_lang     Target language code (e.g. 'de').
	 * @param string|null $kind_override   Builder kind from envelope (skips detect() call).
	 * @return bool  True if the rebuild signal was sent, false on empty input.
	 */
	public function restore_translations( $source_post_id, $target_post_id, array $translated, $target_lang, $kind_override = null ) {
		if ( $kind_override !== null ) {
			$builder_kind = $kind_override;
			$builder      = array( 'kind' => $builder_kind );
		} else {
			$builder = $this->detect( $source_post_id );
			if ( ! $builder ) {
				$this->logger->warning( 'WPML Page Builder: no builder detected on source post', array(
					'source_post_id' => $source_post_id,
				) );
				return false;
			}
			$builder_kind = $builder['kind'];
		}

		if ( empty( $translated ) ) {
			$this->logger->warning( sprintf( '%s: no translations to restore', $builder_kind ), array(
				'source_post_id' => $source_post_id,
			) );
			return false;
		}

		// Step 1 (optional): persist in WPML ST for tracking / TM.
		if ( defined( 'WPML_ST_VERSION' ) && function_exists( 'icl_add_string_translation' ) ) {
			try {
				$this->push_to_wpml( $source_post_id, $builder, $translated, $target_lang );
			} catch ( \Throwable $e ) {
				$this->logger->warning( sprintf( '%s: push_to_wpml failed (WPML ST error, skipping)', $builder_kind ), array(
					'source_post_id' => $source_post_id,
					'error'          => $e->getMessage(),
				) );
			}
		}

		// Step 2: signal the builder to rebuild its meta on the target post.
		try {
			do_action( 'wpml_page_builder_string_translated', $builder_kind, $target_post_id, get_post( $source_post_id ), $translated, $target_lang );
		} catch ( \Throwable $e ) {
			$this->logger->warning( sprintf( '%s: wpml_page_builder_string_translated threw an exception (layout rebuild may be incomplete)', $builder_kind ), array(
				'source_post_id' => $source_post_id,
				'target_post_id' => $target_post_id,
				'error'          => $e->getMessage(),
			) );
		}

		$this->logger->info( sprintf( '%s: triggered layout rebuild via wpml_page_builder_string_translated', $builder_kind ), array(
			'source_post_id' => $source_post_id,
			'target_post_id' => $target_post_id,
			'target_lang'    => $target_lang,
			'string_count'   => count( $translated ),
		) );

		return true;
	}

	// -------------------------------------------------------------------------
	// Private helpers
	// -------------------------------------------------------------------------

	/**
	 * Store translated strings in WPML String Translation DB.
	 *
	 * Looks up the string package for the source post, then calls
	 * icl_add_string_translation() for each string. Gracefully skipped when the
	 * package is not found.
	 *
	 * @param int    $source_post_id
	 * @param array  $builder
	 * @param array  $translated
	 * @param string $target_lang
	 */
	private function push_to_wpml( $source_post_id, array $builder, array $translated, $target_lang ) {
		$packages = apply_filters( 'wpml_st_get_post_string_packages', array(), $source_post_id );

		$matched_package = null;
		foreach ( $packages as $pkg ) {
			if ( isset( $pkg->kind ) && $pkg->kind === $builder['kind'] ) {
				$matched_package = $pkg;
				break;
			}
		}

		if ( ! $matched_package ) {
			$this->logger->debug( sprintf( '%s: WPML package not found for post, skipping WPML ST push', $builder['kind'] ), array(
				'source_post_id' => $source_post_id,
			) );
			return;
		}

		$status  = defined( 'ICL_TM_COMPLETE' ) ? ICL_TM_COMPLETE : 10;
		$pushed  = 0;
		$missing = 0;

		foreach ( $translated as $name => $value ) {
			if ( empty( $value ) || ! is_string( $value ) ) {
				continue;
			}

			$string_id = apply_filters( 'wpml_string_id_from_package', 0, $matched_package, $name, '' );

			if ( ! $string_id ) {
				$missing++;
				continue;
			}

			icl_add_string_translation( $string_id, $target_lang, $value, $status );
			$pushed++;
		}

		$this->logger->debug( sprintf( '%s: pushed translations to WPML ST', $builder['kind'] ), array(
			'source_post_id' => $source_post_id,
			'pushed'         => $pushed,
			'missing'        => $missing,
		) );
	}

	/**
	 * Build the WPML package descriptor for a given post and builder.
	 *
	 * @param int   $post_id
	 * @param array $builder
	 * @return array
	 */
	private function make_wpml_package( $post_id, array $builder ) {
		$name = ( $builder['name_fn'] !== null )
			? call_user_func( $builder['name_fn'], $post_id )
			: strtolower( $builder['kind'] ) . '-' . $post_id;

		return array(
			'kind'      => $builder['kind'],
			'kind_slug' => $builder['kind_slug'],
			'name'      => $name,
			'title'     => get_the_title( $post_id ),
			'post_id'   => $post_id,
			'edit_link' => get_edit_post_link( $post_id ),
			'view_link' => get_permalink( $post_id ),
		);
	}
}
