<?php
/**
 * WPML Integration for LATW
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_WPML_Integration
{

    /**
     * Logger
     */
    private $logger;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->logger = LATWAITR()->logger;

        // Check if WPML is active
        if (!$this->is_wpml_active()) {
            return;
        }

        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks()
    {
        // Hook into WPML
        add_action('wpml_loaded', array($this, 'wpml_loaded'));
    }

    /**
     * WPML loaded hook
     */
    public function wpml_loaded()
    {
        $this->logger->debug('WPML loaded and integrated');
    }

    /**
     * Check if WPML is active
     */
    public function is_wpml_active()
    {
        return defined('ICL_SITEPRESS_VERSION') && function_exists('wpml_get_language_information');
    }

    /**
     * Get active languages
     */
    public function get_active_languages()
    {
        if (!$this->is_wpml_active()) {
            return array();
        }

        global $sitepress;
        $languages = $sitepress->get_active_languages();

        return $languages;
    }

    /**
     * Get default language
     */
    public function get_default_language()
    {
        if (!$this->is_wpml_active()) {
            return 'en';
        }

        global $sitepress;
        return $sitepress->get_default_language();
    }

    /**
     * Get post language (using WPML's most reliable method)
     */
    public function get_post_language($post_id)
    {
        if (!$this->is_wpml_active()) {
            return $this->get_default_language();
        }

        // Get post type
        $post = get_post($post_id);
        if (!$post) {
            return $this->get_default_language();
        }

        // Use WPML filter - most reliable source of truth
        $language_code = apply_filters('wpml_element_language_code', null, array(
            'element_id' => $post_id,
            'element_type' => 'post_' . $post->post_type
        ));

        // Fallback to sitepress method
        if (empty($language_code)) {
            global $sitepress;
            if (isset($sitepress)) {
                $language_details = $sitepress->get_element_language_details($post_id, 'post_' . $post->post_type);
                if ($language_details && !empty($language_details->language_code)) {
                    $language_code = $language_details->language_code;
                }
            }
        }

        // Final fallback
        if (empty($language_code)) {
            $language_code = $this->get_default_language();
        }

        return $language_code;
    }

    /**
     * Get translation ID
     */
    public function get_translation_id($post_id, $target_lang)
    {
        if (!$this->is_wpml_active()) {
            return false;
        }

        global $sitepress;
        $post_type = get_post_type($post_id);

        return $sitepress->get_object_id($post_id, $post_type, false, $target_lang);
    }

    /**
     * Check if post has translation
     */
    public function has_translation($post_id, $target_lang)
    {
        $translation_id = $this->get_translation_id($post_id, $target_lang);
        return !empty($translation_id);
    }

    /**
     * Create translation
     */
    public function create_translation($source_post_id, $target_lang, $translated_data)
    {
        if (!$this->is_wpml_active()) {
            return new WP_Error('wpml_not_active', __('WPML is not active', 'latw-ai-translator-for-wpml'));
        }

        global $sitepress;

        $source_post = get_post($source_post_id);
        if (!$source_post) {
            return new WP_Error('invalid_post', __('Source post not found', 'latw-ai-translator-for-wpml'));
        }

        $source_lang = $this->get_post_language($source_post_id);

        // Ensure source post is set for native editor
        $current_editor = get_post_meta($source_post_id, '_wpml_post_translation_editor_native', true);
        if ($current_editor !== 'yes') {
            update_post_meta($source_post_id, '_wpml_post_translation_editor_native', 'yes');
        }

        // Check if translation already exists
        $existing_translation = $this->get_translation_id($source_post_id, $target_lang);

        if ($existing_translation) {
            // Update existing translation
            return $this->update_translation($existing_translation, $translated_data, $source_post_id);
        }

        $title = $translated_data['title'] ?? $source_post->post_title;

        // Prepare translated post data
        $translated_post = array(
            'post_title' => $title,
            'post_name' => sanitize_title($title),
            'post_content' => $translated_data['content'] ?? $source_post->post_content,
            'post_excerpt' => $translated_data['excerpt'] ?? $source_post->post_excerpt,
            'post_status' => 'draft',
            'post_type' => $source_post->post_type,
            'post_author' => $source_post->post_author,
            'post_parent' => 0,
            'menu_order' => $source_post->menu_order,
            'comment_status' => $source_post->comment_status,
            'ping_status' => $source_post->ping_status,
        );

        // Create the translated post
        $translated_post_id = wp_insert_post($translated_post);

        if (is_wp_error($translated_post_id)) {
            return $translated_post_id;
        }

        // Set language and link to original
        $this->link_translation($source_post_id, $translated_post_id, $source_lang, $target_lang);

        // Copy post meta
        $this->copy_post_meta($source_post_id, $translated_post_id, $translated_data);

        // Handle taxonomies
        $this->handle_taxonomies($source_post_id, $translated_post_id, $source_lang, $target_lang, $translated_data);

        $this->logger->info('Translation created', array(
            'source_post_id' => $source_post_id,
            'translated_post_id' => $translated_post_id,
            'source_lang' => $source_lang,
            'target_lang' => $target_lang,
        ));

        return $translated_post_id;
    }

    /**
     * Update translation
     */
    public function update_translation($post_id, $translated_data, $source_post_id)
    {
        $update_data = array(
            'ID' => $post_id,
        );

        if (isset($translated_data['title'])) {
            $update_data['post_title'] = $translated_data['title'];
        }

        if (isset($translated_data['content'])) {
            $update_data['post_content'] = $translated_data['content'];
        }

        if (isset($translated_data['excerpt'])) {
            $update_data['post_excerpt'] = $translated_data['excerpt'];
        }

        $result = wp_update_post($update_data);

        if (is_wp_error($result)) {
            return $result;
        }

        // Update meta fields
        $this->copy_post_meta($source_post_id, $post_id, $translated_data);

        // Re-apply taxonomy assignments (update path skipped this previously)
        $source_lang = $this->get_post_language($source_post_id);
        $target_lang = $this->get_post_language($post_id);
        $this->handle_taxonomies($source_post_id, $post_id, $source_lang, $target_lang, $translated_data);

        return $post_id;
    }

    /**
     * Link translation to original
     */
    private function link_translation($source_id, $translation_id, $source_lang, $target_lang)
    {
        if (!$this->is_wpml_active()) {
            return false;
        }

        global $wpdb;
        $post_type = get_post_type($source_id);

        $trid = apply_filters(
            'wpml_element_trid',
            null,
            $source_id,
            'post_' . $post_type
        );
        if (!$trid) {
            // Create new translation group
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $trid = $wpdb->get_var(
                $wpdb->prepare(
                    'SELECT MAX(trid) FROM %i WHERE element_type = %s',
                    $wpdb->prefix . 'icl_translations',
                    'post_' . $post_type
                )
            );
            $trid = $trid ? $trid + 1 : 1;
        }

        $this->latwaitr_cleanup_orphan_translation($trid, $target_lang);
        do_action(
            'wpml_set_element_language_details',
            array(
                'element_id' => $translation_id,
                'element_type' => 'post_' . $post_type,
                'trid' => $trid,
                'language_code' => $target_lang,
                'source_language_code' => $source_lang,
            )
        );

        return true;
    }

    private function latwaitr_cleanup_orphan_translation($trid, $lang)
    {
        global $wpdb;

        return $wpdb->query(
            $wpdb->prepare(
                "
            DELETE FROM {$wpdb->prefix}icl_translations
            WHERE trid = %d
              AND language_code = %s
              AND element_id IS NULL
            ",
                $trid,
                $lang
            )
        );
    }

    /**
     * Copy post meta
     */
    private function copy_post_meta($source_id, $target_id, $translated_data)
    {
        $meta = get_post_meta($source_id);

        // Blacklisted meta keys that should never be copied
        $blacklist = array(
            '_elementor_element_cache',           // Elementor cache - should be regenerated
            '_wpml_pb_strings',                   // Internal extraction key, not to be copied
            '_latwaitr_elementor_original_data',  // Legacy cleanup
            '_latwaitr_elementor_string_objects', // Legacy cleanup
        );

        // Handle page builder translations (Bricks, Elementor, etc.)
        if (isset($translated_data['meta']['_wpml_pb_strings']) && !empty($translated_data['meta']['_wpml_pb_strings'])) {
            $pb          = new LATWAITR_WPML_Page_Builder();
            $pb_data     = $translated_data['meta']['_wpml_pb_strings'];
            $target_lang = $this->get_post_language($target_id);

            if (isset($pb_data['strings'], $pb_data['kind'])
                && is_array($pb_data['strings'])
                && is_string($pb_data['kind'])
            ) {
                // New envelope format: { kind: 'Divi', strings: { ... } }
                $pb->restore_translations($source_id, $target_id, $pb_data['strings'], $target_lang, $pb_data['kind']);
            } else {
                // Old flat format (backward compat for in-flight responses)
                $pb->restore_translations($source_id, $target_id, $pb_data, $target_lang);
            }
        }

        $pb_meta_keys_to_skip = array('_elementor_data', '_bricks_page_content_2');

        if (isset($translated_data['meta']['_wpml_pb_strings']['kind'])
            && is_string($translated_data['meta']['_wpml_pb_strings']['kind'])
        ) {
            $pb_kind              = $translated_data['meta']['_wpml_pb_strings']['kind'];
            $pb_meta_keys_to_skip = apply_filters('latwaitr_pb_meta_keys_to_skip', $pb_meta_keys_to_skip, $pb_kind, $source_id);
        }

        foreach ($meta as $key => $values) {
            // Skip blacklisted keys
            if (in_array($key, $blacklist)) {
                continue;
            }

            // Skip page-builder meta keys when universal strings were used
            if (isset($translated_data['meta']['_wpml_pb_strings']) && in_array($key, $pb_meta_keys_to_skip)) {
                continue;
            }

            // If this meta was translated, always copy it (even if it starts with _)
            $was_translated = isset($translated_data['meta'][$key]);

            // Skip private WordPress meta unless it was translated or is on copyable list
            if (substr($key, 0, 1) === '_' && !$was_translated && !in_array($key, $this->get_copyable_meta_keys())) {
                continue;
            }

            foreach ($values as $value) {
                // Check if this meta was translated
                if ($was_translated) {
                    $value = $translated_data['meta'][$key];
                }

                update_post_meta($target_id, $key, maybe_unserialize($value));
            }
        }
    }

    /**
     * Get copyable meta keys
     *
     * Private meta keys (starting with _) that should always be copied from
     * the source post to the translation, even when not sent for translation.
     * If the key was also translated (present in translated_data['meta']),
     * the translated value takes precedence over the source value.
     */
    private function get_copyable_meta_keys()
    {
        $keys = array(
            '_thumbnail_id',

            // Yoast SEO
            '_yoast_wpseo_title',
            '_yoast_wpseo_metadesc',
            '_yoast_wpseo_focuskw',
            '_yoast_wpseo_opengraph-title',
            '_yoast_wpseo_opengraph-description',
            '_yoast_wpseo_twitter-title',
            '_yoast_wpseo_twitter-description',
            '_yoast_wpseo_schema_page_type',
            '_yoast_wpseo_schema_article_type',
            '_yoast_wpseo_meta-robots-noindex',
            '_yoast_wpseo_meta-robots-nofollow',
            '_yoast_wpseo_canonical',

            // Rank Math
            'rank_math_title',
            'rank_math_description',
            'rank_math_focus_keyword',
            'rank_math_og_title',
            'rank_math_og_description',
            'rank_math_twitter_title',
            'rank_math_twitter_description',
            'rank_math_robots',
            'rank_math_canonical_url',
            'rank_math_schema_Article',

            // SEOPress
            '_seopress_titles_title',
            '_seopress_titles_desc',
            '_seopress_social_fb_title',
            '_seopress_social_fb_desc',
            '_seopress_social_twitter_title',
            '_seopress_social_twitter_desc',
            '_seopress_robots_canonical',
            '_seopress_robots_index',
            '_seopress_robots_follow',

            // All in One SEO (AIOSEO)
            '_aioseo_title',
            '_aioseo_description',
            '_aioseo_og_title',
            '_aioseo_og_description',
            '_aioseo_twitter_title',
            '_aioseo_twitter_description',
        );

        return apply_filters('latwaitr_copyable_meta_keys', $keys);
    }

    /**
     * Handle taxonomies
     */
    private function handle_taxonomies($source_id, $target_id, $source_lang, $target_lang, $translated_data)
    {
        if (get_option('latwaitr_translate_taxonomies', 'yes') !== 'yes') {
            return;
        }

        $post_type = get_post_type($source_id);
        $taxonomies = get_object_taxonomies($post_type);

        foreach ($taxonomies as $taxonomy) {
            $terms = wp_get_object_terms($source_id, $taxonomy, array('fields' => 'ids'));

            if (empty($terms)) {
                continue;
            }

            $translated_terms = array();

            foreach ($terms as $term_id) {
                // Get translated term
                $translated_term_id = $this->get_translated_term($term_id, $taxonomy, $target_lang);

                if ($translated_term_id) {
                    $translated_terms[] = $translated_term_id;
                } else {
                    // Term has no translation — create a copy in the target language
                    $new_term_id = $this->create_term_translation($term_id, $taxonomy, $source_lang, $target_lang);
                    if ($new_term_id) {
                        $translated_terms[] = $new_term_id;
                    }
                }
            }

            if (!empty($translated_terms)) {
                wp_set_object_terms($target_id, $translated_terms, $taxonomy);
            }
        }
    }

    /**
     * Get translated term
     */
    private function get_translated_term($term_id, $taxonomy, $target_lang)
    {
        if (!$this->is_wpml_active()) {
            return $term_id;
        }

        global $sitepress;

        return $sitepress->get_object_id($term_id, $taxonomy, false, $target_lang);
    }

    /**
     * Create a term translation by copying the original term to the target language.
     */
    private function create_term_translation($term_id, $taxonomy, $source_lang, $target_lang)
    {
        $original_term = get_term($term_id, $taxonomy);
        if (!$original_term || is_wp_error($original_term)) {
            return false;
        }

        // Handle parent hierarchy — ensure parent is translated first
        $translated_parent = 0;
        if ($original_term->parent) {
            $translated_parent = $this->get_translated_term($original_term->parent, $taxonomy, $target_lang);
            if (!$translated_parent) {
                $translated_parent = $this->create_term_translation($original_term->parent, $taxonomy, $source_lang, $target_lang);
                if (!$translated_parent) {
                    $translated_parent = 0;
                }
            }
        }

        // Use a language-suffixed slug to avoid conflicts with the source term
        // (WPML convention: terms in different languages have unique slugs).
        $target_slug = $original_term->slug . '-' . $target_lang;

        $result = wp_insert_term($original_term->name, $taxonomy, array(
            'slug'        => $target_slug,
            'description' => $original_term->description,
            'parent'      => $translated_parent,
        ));

        if (is_wp_error($result)) {
            if ($result->get_error_code() === 'term_exists') {
                // Term already exists in WP — get its ID.
                $existing_id = $result->get_error_data();
                if (is_array($existing_id)) {
                    $existing_id = $existing_id['term_id'] ?? 0;
                }
                $new_term_id = (int) $existing_id;
                if ( ! $new_term_id ) {
                    $this->logger->info('Term exists but could not determine its ID', array(
                        'term_id'  => $term_id,
                        'taxonomy' => $taxonomy,
                    ));
                    return false;
                }

                // Guard: if the existing term IS the source term, do NOT create a
                // self-translation in WPML. Return the source term as a temporary
                // fallback so the translated post still gets a category assigned,
                // without corrupting WPML's translation table.
                if ($new_term_id === (int) $term_id) {
                    $this->logger->debug('Term copy has same name as source term; using source as temporary fallback (no WPML linking)', array(
                        'term_id'     => $term_id,
                        'taxonomy'    => $taxonomy,
                        'target_lang' => $target_lang,
                    ));
                    return $new_term_id; // assigned to post, but NOT linked in WPML
                }

                $this->logger->debug('Term already exists, linking existing term to WPML', array(
                    'source_term_id'   => $term_id,
                    'existing_term_id' => $new_term_id,
                    'taxonomy'         => $taxonomy,
                    'target_lang'      => $target_lang,
                ));
            } else {
                $this->logger->info('Failed to create term translation', array(
                    'term_id'  => $term_id,
                    'taxonomy' => $taxonomy,
                    'error'    => $result->get_error_message(),
                ));
                return false;
            }
        } else {
            $new_term_id = $result['term_id'];
        }

        // Get the trid of the original term
        $trid = apply_filters('wpml_element_trid', null, $term_id, 'tax_' . $taxonomy);

        if ($trid) {
            do_action('wpml_set_element_language_details', array(
                'element_id'           => $new_term_id,
                'element_type'         => 'tax_' . $taxonomy,
                'trid'                 => $trid,
                'language_code'        => $target_lang,
                'source_language_code' => $source_lang,
            ));
        }

        $this->logger->info('Created term translation (copy)', array(
            'original_term_id' => $term_id,
            'new_term_id'      => $new_term_id,
            'taxonomy'         => $taxonomy,
            'name'             => $original_term->name,
            'target_lang'      => $target_lang,
        ));

        return $new_term_id;
    }

    /**
     * Get term language from WPML
     */
    public function get_term_language($term_id, $taxonomy)
    {
        if (!$this->is_wpml_active()) {
            return $this->get_default_language();
        }

        $language_code = apply_filters('wpml_element_language_code', null, array(
            'element_id' => $term_id,
            'element_type' => 'tax_' . $taxonomy,
        ));

        if (empty($language_code)) {
            global $sitepress;
            if (isset($sitepress)) {
                $language_details = $sitepress->get_element_language_details($term_id, 'tax_' . $taxonomy);
                if ($language_details && !empty($language_details->language_code)) {
                    $language_code = $language_details->language_code;
                }
            }
        }

        if (empty($language_code)) {
            $language_code = $this->get_default_language();
        }

        return $language_code;
    }

    /**
     * Get translated term ID
     */
    public function get_translated_term_id($term_id, $taxonomy, $target_lang)
    {
        if (!$this->is_wpml_active()) {
            return false;
        }

        global $sitepress;
        return $sitepress->get_object_id($term_id, $taxonomy, false, $target_lang);
    }

    /**
     * Create a term translation (public wrapper for use by response checker)
     */
    public function create_term_translation_public($term_id, $taxonomy, $source_lang, $target_lang)
    {
        return $this->create_term_translation($term_id, $taxonomy, $source_lang, $target_lang);
    }

    /**
     * Get languages for translation
     */
    public function get_languages_for_translation($post_id)
    {
        if (!$this->is_wpml_active()) {
            return array();
        }

        $all_languages = $this->get_active_languages();
        $post_lang = $this->get_post_language($post_id);
        $target_languages = get_option('latwaitr_target_languages', array());

        $available = array();

        foreach ($all_languages as $lang_code => $lang_data) {
            // Skip source language
            if ($lang_code === $post_lang) {
                continue;
            }

            // Skip if not in target languages
            if (!empty($target_languages) && !in_array($lang_code, $target_languages)) {
                continue;
            }

            $available[$lang_code] = $lang_data;
        }

        return $available;
    }

}
