<?php
/**
 * Main Translator class for LATW
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_Translator {

    /**
     * Logger
     */
    private $logger;

    /**
     * LLM API
     */
    private $llm;

    /**
     * WPML Integration
     */
    private $wpml;

    /**
     * Security
     */
    private $security;

    /**
     * Universal WPML page builder (lazy)
     */
    private $wpml_pb;

    /**
     * Constructor
     */
    public function __construct() {
        $this->logger = LATWAITR()->logger;
        $this->llm = LATWAITR()->llm;
        $this->wpml = LATWAITR()->wpml;
        $this->security = LATWAITR()->security;
    }

    /**
     * Get universal WPML page builder instance (lazy)
     */
    private function get_wpml_pb() {
        if ( ! isset( $this->wpml_pb ) ) {
            $this->wpml_pb = new LATWAITR_WPML_Page_Builder();
        }
        return $this->wpml_pb;
    }

    /**
     * Gather content to translate (public for batch API usage)
     */
    public function gather_content($post_id) {
        $post = get_post($post_id);
        $content = array();

        // Detect page builder (Bricks, Elementor, etc.)
        $pb = $this->get_wpml_pb()->detect( $post_id );
        $is_pb_page = ( $pb !== null );

        // Title
        if (get_option('latwaitr_translate_title') === 'yes') {
            $content['title'] = $post->post_title;
        }

        // Content - skip if page builder page (content is in page builder meta)
        if (get_option('latwaitr_translate_content') === 'yes' && !$is_pb_page) {
            $content['content'] = $post->post_content;
        } elseif ($is_pb_page) {
            $this->logger->debug( sprintf( 'Skipping post_content extraction for %s page (will be rebuilt by WPML Page Builder API)', $pb['kind'] ), array(
                'post_id' => $post_id,
            ));
        }

        // Excerpt
        if (get_option('latwaitr_translate_excerpt') === 'yes' && !empty($post->post_excerpt)) {
            $content['excerpt'] = $post->post_excerpt;
        }

        // Custom fields
        if (get_option('latwaitr_translate_custom_fields') === 'yes') {
            $custom_fields = $this->get_translatable_custom_fields($post_id);
            if (!empty($custom_fields)) {
                $content['meta'] = $custom_fields;
            }
        }

        // SEO fields (Yoast, Rank Math, etc.)
        if (get_option('latwaitr_translate_seo') === 'yes') {
            $seo_fields = $this->get_seo_fields($post_id);
            if (!empty($seo_fields)) {
                if (!isset($content['meta'])) {
                    $content['meta'] = array();
                }
                $content['meta'] = array_merge($content['meta'], $seo_fields);
            }
        }

        // Page Builder fields (Elementor, Bricks, etc.)
        $builder_fields = $this->get_page_builder_fields($post_id);
        if (!empty($builder_fields)) {
            if (!isset($content['meta'])) {
                $content['meta'] = array();
            }
            $content['meta'] = array_merge($content['meta'], $builder_fields);
        }

        return apply_filters('latwaitr_content_to_translate', $content, $post_id);
    }

    /**
     * Get translatable custom fields
     */
    private function get_translatable_custom_fields($post_id) {
        $custom_fields = array();
        $all_meta = get_post_meta($post_id);

        // Get field whitelist/blacklist
        $whitelist = get_option('latwaitr_custom_fields_whitelist', array());
        $blacklist = get_option('latwaitr_custom_fields_blacklist', array());

        foreach ($all_meta as $key => $values) {
            // Skip private fields
            if (substr($key, 0, 1) === '_') {
                continue;
            }

            // Check whitelist/blacklist
            if (!empty($whitelist) && !in_array($key, $whitelist)) {
                continue;
            }

            if (!empty($blacklist) && in_array($key, $blacklist)) {
                continue;
            }

            foreach ($values as $value) {
                if (is_string($value) && !empty($value)) {
                    $custom_fields[$key] = $value;
                    break; // Only take first value
                }
            }
        }

        return $custom_fields;
    }

    /**
     * Get SEO fields for translation.
     *
     * Returns human-readable text fields from common SEO plugins so they can be
     * sent to OpenAI for translation. Non-translatable fields (robots directives,
     * canonical URLs, schema type enums) are intentionally excluded — they are
     * handled by get_copyable_meta_keys() in WPML integration and copied as-is.
     */
    private function get_seo_fields($post_id) {
        $translatable_keys = array(
            // Yoast SEO
            '_yoast_wpseo_title',
            '_yoast_wpseo_metadesc',
            '_yoast_wpseo_focuskw',
            '_yoast_wpseo_opengraph-title',
            '_yoast_wpseo_opengraph-description',
            '_yoast_wpseo_twitter-title',
            '_yoast_wpseo_twitter-description',

            // Rank Math
            'rank_math_title',
            'rank_math_description',
            'rank_math_focus_keyword',
            'rank_math_og_title',
            'rank_math_og_description',
            'rank_math_twitter_title',
            'rank_math_twitter_description',

            // SEOPress
            '_seopress_titles_title',
            '_seopress_titles_desc',
            '_seopress_social_fb_title',
            '_seopress_social_fb_desc',
            '_seopress_social_twitter_title',
            '_seopress_social_twitter_desc',

            // All in One SEO (AIOSEO)
            '_aioseo_title',
            '_aioseo_description',
            '_aioseo_og_title',
            '_aioseo_og_description',
            '_aioseo_twitter_title',
            '_aioseo_twitter_description',
        );

        $seo_fields = array();

        foreach ($translatable_keys as $key) {
            $value = get_post_meta($post_id, $key, true);
            if (!empty($value) && is_string($value)) {
                $seo_fields[$key] = $value;
            }
        }

        return $seo_fields;
    }

    /**
     * Get Page Builder fields (via WPML Page Builder API)
     */
    private function get_page_builder_fields($post_id) {
        $builder_fields = array();

        $strings = $this->get_wpml_pb()->extract_strings( $post_id );
        if ( ! empty( $strings ) ) {
            $builder_fields['_wpml_pb_strings'] = $strings;
        }

        return $builder_fields;
    }
}
