<?php
/**
 * Response Status Checker for LATW
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_Response_Checker {

    /**
     * Cron hook name
     */
    const CRON_HOOK = 'latwaitr_check_response_statuses';

    /**
     * Logger
     */
    private $logger;

    /**
     * Constructor
     */
    public function __construct() {
        $this->logger = LATWAITR()->logger;
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Register cron action
        add_action(self::CRON_HOOK, array($this, 'check_all_pending_responses'));

        // Schedule cron if not scheduled
        $timestamp = wp_next_scheduled(self::CRON_HOOK);
        if ( ! $timestamp || $timestamp < time() - 300 ) {
            wp_clear_scheduled_hook(self::CRON_HOOK);
            wp_schedule_event(time(), 'every_minute', self::CRON_HOOK);
        }
    }

    /**
     * Add custom cron schedule (every 1 minute)
     */
    public static function add_cron_schedules($schedules) {
        $schedules['every_minute'] = array(
            'interval' => 60,
            'display' => __('Every Minute', 'latw-ai-translator-for-wpml')
        );
        return $schedules;
    }

    /**
     * Check all pending background responses
     */
    public function check_all_pending_responses() {
        // Get all translations with pending response statuses
        $pending_statuses = array(
            'queued',
            'in_progress'
        );

        $translations = LATWAITR_Database::get_translations_by_response_status($pending_statuses, 50);

        if (empty($translations)) {
            $this->logger->debug('No pending responses found');
            return;
        }

        $this->logger->info('Found ' . count($translations) . ' pending responses');

        foreach ($translations as $translation) {
            $this->check_single_response($translation);
        }
    }

    /**
     * Check single response status
     */
    private function check_single_response($translation) {
        $translation_id = $translation['id'];
        $response_id = $translation['response_id'];

        try {
            $llm = LATWAITR()->llm;

            // Get response status from OpenAI
            $response_data = $llm->get_response_status($response_id);

            if (is_wp_error($response_data)) {
                $this->logger->error('OpenAI response status check returned error', array(
                    'translation_id' => $translation_id,
                    'response_id' => $response_id,
                    'error_message' => $response_data->get_error_message(),
                    'error_code' => $response_data->get_error_code(),
                    'error_data' => $response_data->get_error_data(),
                ));
                throw new Exception($response_data->get_error_message());
            }

            $response_status = $response_data['status'];

            $this->logger->debug('Response status: ' . $response_status, array(
                'translation_id' => $translation_id,
                'response_id' => $response_id,
            ));

            // Update response status in DB
            LATWAITR_Database::update_translation($translation_id, array(
                'status' => $response_status,
                'response_status' => $response_status,
            ));

            // Handle completed response
            if ($response_status === 'completed') {
                $this->process_completed_response($translation, $response_data);
            }

            // Handle failed/cancelled/incomplete
            if (in_array($response_status, array('failed', 'cancelled', 'incomplete'))) {
                $this->logger->error('OpenAI response has non-success status', array(
                    'translation_id' => $translation_id,
                    'response_id' => $response_id,
                    'response_status' => $response_status,
                    'incomplete_details' => $response_data['incomplete_details'] ?? null,
                    'error' => $response_data['error'] ?? null,
                    'source_post_id' => $translation['source_post_id'] ?? null,
                    'target_lang' => $translation['target_lang'] ?? null,
                ));
                $this->handle_failed_response($translation, $response_data);
            }

        } catch (\Throwable $e) {
            $this->logger->error('Response status check failed', array(
                'translation_id' => $translation_id,
                'response_id' => $response_id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ));
        }
    }

    /**
     * Process completed response
     */
    private function process_completed_response($translation, $response_data) {
        $translation_id = $translation['id'];
        $response_id = $translation['response_id'];

        try {
            // Parse the translation from response
            $translated_content = $this->parse_response_translation($response_data);

            if (is_wp_error($translated_content)) {
                $this->logger->error('Failed to parse translation from OpenAI response', array(
                    'translation_id' => $translation_id,
                    'response_id' => $response_id,
                    'error_code' => $translated_content->get_error_code(),
                    'error_message' => $translated_content->get_error_message(),
                    'response_status' => $response_data['status'] ?? 'unknown',
                    'output_count' => isset($response_data['output']) ? count($response_data['output']) : 0,
                ));
                throw new Exception($translated_content->get_error_message());
            }

            $translated_data = $translated_content['translated_content'];
            $input_tokens = $translated_content['input_tokens_used'];
            $output_tokens = $translated_content['output_tokens_used'];

            $element_type = isset($translation['element_type']) ? $translation['element_type'] : 'post';

            if ($element_type === 'term') {
                $this->process_completed_term_translation($translation, $translated_data, $input_tokens, $output_tokens);
                return;
            }

            // Create WPML translation
            $post_id = $translation['source_post_id'];
            $target_lang = $translation['target_lang'];

            $wpml = LATWAITR()->wpml;
            $translated_post_id = $wpml->create_translation($post_id, $target_lang, $translated_data);

            if (is_wp_error($translated_post_id)) {
                throw new Exception($translated_post_id->get_error_message());
            }

            // Publish immediately after creation/update, before any other operation
            // that could throw an exception and leave the post as draft.
            wp_publish_post($translated_post_id);

            // Calculate cost
            $model = get_option('latwaitr_openai_model', LATWAITR_LLM_API::DEFAULT_MODEL);
            $cost_data = LATWAITR_Cost_Estimator::calculate_translation_cost($input_tokens, $output_tokens, $model);
            $cost = $cost_data['total_cost'];

            // Update database
            LATWAITR_Database::update_translation($translation_id, array(
                'post_id' => $translated_post_id,
                'status' => 'completed',
                'input_tokens_used' => $input_tokens,
                'output_tokens_used' => $output_tokens,
                'actual_cost' => $cost,
                'response_completed_at' => current_time('mysql', true),
                'completed_at' => current_time('mysql', true),
            ));

            $this->logger->info('Background translation completed', array(
                'translation_id' => $translation_id,
                'response_id' => $response_id,
                'translated_post_id' => $translated_post_id,
            ));

        } catch (\Throwable $e) {
            LATWAITR_Database::update_translation($translation_id, array(
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ));

            $this->logger->error('Response processing failed', array(
                'translation_id' => $translation_id,
                'response_id' => $response_id,
                'source_post_id' => $translation['source_post_id'] ?? null,
                'target_lang' => $translation['target_lang'] ?? null,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ));
        }
    }

    /**
     * Process completed term translation
     */
    private function process_completed_term_translation($translation, $translated_data, $input_tokens, $output_tokens) {
        $translation_id = $translation['id'];
        $term_id = $translation['source_post_id'];
        $target_lang = $translation['target_lang'];
        $taxonomy = isset($translation['taxonomy']) ? $translation['taxonomy'] : '';

        try {
            $original_term = get_term($term_id, $taxonomy);
            if (!$original_term || is_wp_error($original_term)) {
                throw new Exception('Original term not found: ' . $term_id);
            }

            $source_lang = $translation['source_lang'];

            $translated_name = isset($translated_data['name']) ? $translated_data['name'] : $original_term->name;
            $translated_description = isset($translated_data['description']) ? $translated_data['description'] : $original_term->description;

            $wpml = LATWAITR()->wpml;

            // Handle parent hierarchy
            $translated_parent = 0;
            if ($original_term->parent) {
                $translated_parent = $wpml->get_translated_term_id($original_term->parent, $taxonomy, $target_lang);
                if (!$translated_parent) {
                    // Create parent translation (untranslated copy) recursively
                    $translated_parent = $wpml->create_term_translation_public($original_term->parent, $taxonomy, $source_lang, $target_lang);
                    if (!$translated_parent) {
                        $translated_parent = 0;
                    }
                }
            }

            // Check if translation already exists.
            // Guard: if get_translated_term_id() returns the source term's own ID,
            // it means the term was previously self-linked (fallback copy had the same
            // name). In that case, create a proper translated term instead.
            $existing_translation = $wpml->get_translated_term_id($term_id, $taxonomy, $target_lang);
            if ($existing_translation && (int) $existing_translation === (int) $term_id) {
                $existing_translation = null;
            }

            if ($existing_translation) {
                // Update existing translated term.
                // Always provide an explicit slug built from the translated name so
                // WordPress doesn't fall back to the old slug (e.g. "oil-pl") which
                // may conflict with another term created by a parallel copy operation.
                $target_slug  = sanitize_title($translated_name);
                $update_args  = array(
                    'name'        => $translated_name,
                    'slug'        => $target_slug,
                    'description' => $translated_description,
                    'parent'      => $translated_parent,
                );
                $update_result = wp_update_term($existing_translation, $taxonomy, $update_args);

                if (is_wp_error($update_result)) {
                    // Slug still conflicts — append lang to guarantee uniqueness.
                    $update_args['slug'] = $target_slug . '-' . $target_lang;
                    $update_result       = wp_update_term($existing_translation, $taxonomy, $update_args);
                }

                if (is_wp_error($update_result)) {
                    throw new Exception('Failed to update term translation: ' . $update_result->get_error_message());
                }

                $new_term_id = $existing_translation;
            } else {
                // Create new translated term.
                $result = wp_insert_term($translated_name, $taxonomy, array(
                    'slug'        => sanitize_title($translated_name),
                    'description' => $translated_description,
                    'parent'      => $translated_parent,
                ));

                if (is_wp_error($result)) {
                    if ($result->get_error_code() === 'term_exists') {
                        // A term with this translated name already exists — use it.
                        $existing_id = $result->get_error_data();
                        $new_term_id = (int) (is_array($existing_id) ? ($existing_id['term_id'] ?? 0) : $existing_id);
                        if (!$new_term_id) {
                            throw new Exception('term_exists but could not determine term ID');
                        }
                    } else {
                        throw new Exception('Failed to create term: ' . $result->get_error_message());
                    }
                } else {
                    $new_term_id = $result['term_id'];
                }

                // Link to WPML.
                $trid = apply_filters('wpml_element_trid', null, $term_id, 'tax_' . $taxonomy);

                if ($trid) {
                    do_action('wpml_set_element_language_details', array(
                        'element_id'           => $new_term_id,
                        'element_type'         => 'tax_' . $taxonomy,
                        'trid'                 => $trid,
                        'language_code'        => $target_lang,
                        'source_language_code' => $source_lang,
                    ));
                }
            }

            // Calculate cost
            $model = get_option('latwaitr_openai_model', LATWAITR_LLM_API::DEFAULT_MODEL);
            $cost_data = LATWAITR_Cost_Estimator::calculate_translation_cost($input_tokens, $output_tokens, $model);
            $cost = $cost_data['total_cost'];

            // Update database
            LATWAITR_Database::update_translation($translation_id, array(
                'post_id' => $new_term_id,
                'status' => 'completed',
                'input_tokens_used' => $input_tokens,
                'output_tokens_used' => $output_tokens,
                'actual_cost' => $cost,
                'response_completed_at' => current_time('mysql', true),
                'completed_at' => current_time('mysql', true),
            ));

            $this->logger->info('Background term translation completed', array(
                'translation_id' => $translation_id,
                'original_term_id' => $term_id,
                'new_term_id' => $new_term_id,
                'taxonomy' => $taxonomy,
                'target_lang' => $target_lang,
            ));

        } catch (Exception $e) {
            LATWAITR_Database::update_translation($translation_id, array(
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ));

            $this->logger->error('Term translation processing failed', array(
                'translation_id' => $translation_id,
                'term_id' => $term_id,
                'taxonomy' => $taxonomy,
                'target_lang' => $target_lang,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ));
        }
    }

    /**
     * Parse response translation (similar to existing parse method)
     */
    private function parse_response_translation($response_data) {
        // Find the last message output (skip reasoning)
        $lastOutput = null;
        foreach ($response_data['output'] as $index => $output) {
            if ($output['type'] === 'message') {
                $lastOutput = $index;
            }
        }

        // Check content filter
        if ($response_data['status'] === 'incomplete'
            && ($response_data['incomplete_details']['reason'] ?? null) === 'content_filter') {
            $this->logger->error('OpenAI content filter blocked translation', array(
                'response_status' => $response_data['status'],
                'incomplete_details' => $response_data['incomplete_details'] ?? null,
            ));
            return new WP_Error('error_content_filter',
                __('Content blocked by OpenAI safety filter', 'latw-ai-translator-for-wpml'));
        }

        if ($lastOutput === null || !isset($response_data['output'][$lastOutput]['content'][0]['text'])) {
            $this->logger->error('Invalid OpenAI response format - no message output found', array(
                'response_status' => $response_data['status'] ?? 'unknown',
                'output_types' => array_map(function($o) { return $o['type'] ?? 'unknown'; }, $response_data['output'] ?? array()),
                'output_count' => count($response_data['output'] ?? array()),
                'last_output_index' => $lastOutput,
            ));
            return new WP_Error('invalid_response',
                __('Invalid API response format', 'latw-ai-translator-for-wpml'));
        }

        $json_content = $response_data['output'][$lastOutput]['content'][0]['text'];
        $translated_data = json_decode($json_content, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->logger->error('Failed to parse OpenAI translation JSON', array(
                'json_error' => json_last_error_msg(),
                'json_error_code' => json_last_error(),
                'raw_content_preview' => substr($json_content, 0, 500),
                'content_length' => strlen($json_content),
            ));
            return new WP_Error('json_parse_error',
                __('Failed to parse JSON response: ', 'latw-ai-translator-for-wpml') . json_last_error_msg());
        }

        return array(
            'translated_content' => $translated_data,
            'input_tokens_used' => $response_data['usage']['input_tokens'] ?? 0,
            'output_tokens_used' => $response_data['usage']['output_tokens'] ?? 0,
        );
    }

    /**
     * Handle failed response
     */
    private function handle_failed_response($translation, $response_data) {
        $translation_id = $translation['id'];
        $retry_count = intval($translation['retry_count']);
        $max_retries = 3;

        $error_message = 'Response ' . $response_data['status'];
        if (isset($response_data['incomplete_details']['reason'])) {
            $error_message .= ': ' . $response_data['incomplete_details']['reason'];
        }

        $this->logger->error('Handling failed OpenAI response', array(
            'translation_id' => $translation_id,
            'response_id' => $translation['response_id'] ?? null,
            'response_status' => $response_data['status'],
            'error_message' => $error_message,
            'incomplete_details' => $response_data['incomplete_details'] ?? null,
            'error_details' => $response_data['error'] ?? null,
            'retry_count' => $retry_count,
            'max_retries' => $max_retries,
            'source_post_id' => $translation['source_post_id'] ?? null,
            'target_lang' => $translation['target_lang'] ?? null,
        ));

        if ($retry_count < $max_retries) {
            // Retry: re-submit response
            $this->logger->info('Retrying failed response', array(
                'translation_id' => $translation_id,
                'retry_count' => $retry_count + 1,
            ));

            LATWAITR_Database::update_translation($translation_id, array(
                'retry_count' => $retry_count + 1,
                'status' => 'preparing',
                'response_id' => null,
                'error_message' => 'Retrying (attempt ' . ($retry_count + 1) . '): ' . $error_message,
            ));

            // Re-queue using LATWAITR_Queue
            $queue = LATWAITR()->queue;
            $queue->queue_translation(
                $translation['source_post_id'],
                $translation['target_lang']
            );

        } else {
            // Max retries reached
            $this->logger->error('Max retries exceeded - translation permanently failed', array(
                'translation_id' => $translation_id,
                'response_id' => $translation['response_id'] ?? null,
                'retry_count' => $retry_count,
                'last_error' => $error_message,
                'source_post_id' => $translation['source_post_id'] ?? null,
                'target_lang' => $translation['target_lang'] ?? null,
            ));

            LATWAITR_Database::update_translation($translation_id, array(
                'status' => 'failed',
                'error_message' => 'Max retries exceeded. Last error: ' . $error_message,
            ));
        }
    }
}
