<?php
/**
 * Queue handler for LATW (Background Responses API version)
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_Queue {

    /**
     * Logger
     */
    private $logger;

    /**
     * Constructor
     */
    public function __construct() {
        $this->logger = LATWAITR()->logger;
    }

    /**
     * Queue translation (immediately submits background response to OpenAI)
     */
    public function queue_translation($post_id, $target_lang, $options = array()) {
        $this->logger->info('Queueing translation', array(
            'post_id' => $post_id,
            'target_lang' => $target_lang,
        ));

        // Check OpenAI API key
        $api_key = get_option('latwaitr_openai_api_key');
        if (empty($api_key)) {
            return new WP_Error('no_api_key', __('OpenAI API Key is not configured. Please add your API key in the settings.', 'latw-ai-translator-for-wpml'));
        }

        // Check rate limiting
        $pending_count = $this->get_pending_response_count();
        $max_concurrent = 50;

        if ($pending_count >= $max_concurrent) {
            return new WP_Error('queue_full', __('Too many translations in progress(Max 50). Please wait.', 'latw-ai-translator-for-wpml'));
        }

        $defaults = array(
            'priority' => 10,
            'auto' => false,
            'publish' => false,
        );

        $options = wp_parse_args($options, $defaults);

        // Get source language
        $wpml = LATWAITR()->wpml;
        $source_lang = $wpml->get_post_language($post_id);

        // Create database record for tracking
        $translation_id = LATWAITR_Database::log_translation(array(
            'post_id' => 0, // Will be filled after translation
            'source_lang' => $source_lang,
            'target_lang' => $target_lang,
            'source_post_id' => $post_id,
            'status' => 'preparing',
            'translation_type' => $options['auto'] ? 'auto' : 'on-demand',
        ));

        // Immediately submit background translation
        $this->submit_background_translation($translation_id, $post_id, $target_lang);

        $this->logger->info('Translation queued and background response submitted', array(
            'translation_id' => $translation_id,
            'post_id' => $post_id,
            'target_lang' => $target_lang,
        ));

        return $translation_id;
    }

    /**
     * Submit background translation to OpenAI (non-blocking)
     */
    private function submit_background_translation($translation_id, $post_id, $target_lang) {
        try {
            // Gather content
            $translator = LATWAITR()->translator;
            $content_to_translate = $translator->gather_content($post_id);

            if (is_wp_error($content_to_translate)) {
                throw new Exception($content_to_translate->get_error_message());
            }

            if (empty($content_to_translate)) {
                throw new Exception('No content to translate');
            }

            $source_lang = LATWAITR()->wpml->get_post_language($post_id);

            $llm = LATWAITR()->llm;

            // Submit background translation request
            $response_data = $llm->submit_background_translation($content_to_translate, $source_lang, $target_lang);

            if (is_wp_error($response_data)) {
                throw new Exception($response_data->get_error_message());
            }

            $response_id = $response_data['id'];
            $response_status = $response_data['status'];

            // Update database with response info
            LATWAITR_Database::update_translation($translation_id, array(
                'status' => $response_status, // queued, in_progress, etc.
                'response_id' => $response_id,
                'response_status' => $response_status,
                'response_submitted_at' => current_time('mysql', true),
                'request_data' => json_encode($content_to_translate),
            ));

            $this->logger->info('Background response submitted successfully', array(
                'translation_id' => $translation_id,
                'response_id' => $response_id,
                'response_status' => $response_status,
            ));

        } catch (Exception $e) {
            LATWAITR_Database::update_translation($translation_id, array(
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ));

            $this->logger->error('Background response submission failed', array(
                'translation_id' => $translation_id,
                'error' => $e->getMessage(),
            ));
        }
    }

    /**
     * Queue term translation (immediately submits background response to OpenAI)
     */
    public function queue_term_translation($term_id, $target_lang) {
        $this->logger->info('Queueing term translation', array(
            'term_id' => $term_id,
            'target_lang' => $target_lang,
        ));

        $api_key = get_option('latwaitr_openai_api_key');
        if (empty($api_key)) {
            return new WP_Error('no_api_key', __('OpenAI API Key is not configured. Please add your API key in the settings.', 'latw-ai-translator-for-wpml'));
        }

        // Check rate limiting
        $pending_count = $this->get_pending_response_count();
        if ($pending_count >= 50) {
            return new WP_Error('queue_full', __('Too many translations in progress(Max 50). Please wait.', 'latw-ai-translator-for-wpml'));
        }

        // Get term details
        $term = get_term($term_id);
        if (!$term || is_wp_error($term)) {
            return new WP_Error('invalid_term', __('Term not found', 'latw-ai-translator-for-wpml'));
        }

        // Get source language
        $wpml = LATWAITR()->wpml;
        $source_lang = $wpml->get_term_language($term_id, $term->taxonomy);

        if ($source_lang === $target_lang) {
            return new WP_Error('same_language', __('Source and target language are the same', 'latw-ai-translator-for-wpml'));
        }

        // Create database record
        $translation_id = LATWAITR_Database::log_translation(array(
            'post_id' => 0,
            'source_lang' => $source_lang,
            'target_lang' => $target_lang,
            'source_post_id' => $term_id,
            'status' => 'preparing',
            'translation_type' => 'on-demand',
            'element_type' => 'term',
            'taxonomy' => $term->taxonomy,
        ));

        // Submit background translation
        $this->submit_background_term_translation($translation_id, $term_id, $source_lang, $target_lang);

        $this->logger->info('Term translation queued and background response submitted', array(
            'translation_id' => $translation_id,
            'term_id' => $term_id,
            'taxonomy' => $term->taxonomy,
            'target_lang' => $target_lang,
        ));

        return $translation_id;
    }

    /**
     * Submit background term translation to OpenAI (non-blocking)
     */
    private function submit_background_term_translation($translation_id, $term_id, $source_lang, $target_lang) {
        try {
            $term = get_term($term_id);
            if (!$term || is_wp_error($term)) {
                throw new Exception('Term not found');
            }

            $content = array(
                'name' => $term->name,
                'description' => $term->description,
            );

            $llm = LATWAITR()->llm;

            $response_data = $llm->submit_background_translation($content, $source_lang, $target_lang);

            if (is_wp_error($response_data)) {
                throw new Exception($response_data->get_error_message());
            }

            $response_id = $response_data['id'];
            $response_status = $response_data['status'];

            LATWAITR_Database::update_translation($translation_id, array(
                'status' => $response_status,
                'response_id' => $response_id,
                'response_status' => $response_status,
                'response_submitted_at' => current_time('mysql', true),
                'request_data' => json_encode($content),
            ));

            $this->logger->info('Background term translation submitted successfully', array(
                'translation_id' => $translation_id,
                'response_id' => $response_id,
                'response_status' => $response_status,
            ));

        } catch (Exception $e) {
            LATWAITR_Database::update_translation($translation_id, array(
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ));

            $this->logger->error('Background term translation submission failed', array(
                'translation_id' => $translation_id,
                'error' => $e->getMessage(),
            ));
        }
    }

    /**
     * Get count of pending background responses
     */
    private function get_pending_response_count() {
        global $wpdb;
        $table_name = LATWAITR_Database::get_table_name();

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $count = $wpdb->get_var(
            $wpdb->prepare(
                'SELECT COUNT(*) FROM %i WHERE status IN (%s, %s) AND response_id IS NOT NULL',
                $table_name,
                'queued',
                'in_progress'
            )
        );

        return intval($count);
    }

    /**
     * Get queue status for display
     */
    public function get_queue_status() {
        global $wpdb;
        $table_name = LATWAITR_Database::get_table_name();

        $stats = array(
            'pending' => 0,
            'in_progress' => 0,
            'completed' => 0,
            'failed' => 0,
        );

        // Count response statuses
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $stats['in_progress'] = $wpdb->get_var(
            $wpdb->prepare(
                'SELECT COUNT(*) FROM %i WHERE status IN (%s, %s) AND response_id IS NOT NULL',
                $table_name,
                'queued',
                'in_progress'
            )
        );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $stats['completed'] = $wpdb->get_var(
            $wpdb->prepare(
                'SELECT COUNT(*) FROM %i WHERE status = %s AND created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)',
                $table_name,
                'completed'
            )
        );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $stats['failed'] = $wpdb->get_var(
            $wpdb->prepare(
                'SELECT COUNT(*) FROM %i WHERE status = %s AND created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)',
                $table_name,
                'failed'
            )
        );

        return $stats;
    }

    /**
     * Get queue items with details
     */
    public function get_queue_items() {
        global $wpdb;
        $table_name = LATWAITR_Database::get_table_name();

        // Get recent translations
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $translations = $wpdb->get_results(
            $wpdb->prepare(
                'SELECT * FROM %i WHERE status IN (%s, %s, %s) ORDER BY created_at DESC LIMIT 50',
                $table_name,
                'queued',
                'in_progress',
                'preparing'
            ),
            ARRAY_A
        );

        $items = array();

        foreach ($translations as $translation) {
            $source_post_id = $translation['source_post_id'];
            $element_type = isset($translation['element_type']) ? $translation['element_type'] : 'post';

            if ($element_type === 'term') {
                $term = get_term($source_post_id);
                if ($term && !is_wp_error($term)) {
                    $items[] = array(
                        'translation_id' => $translation['id'],
                        'post_id' => $source_post_id,
                        'post_title' => $term->name,
                        'post_type' => $translation['taxonomy'] ?? $term->taxonomy,
                        'target_lang' => $translation['target_lang'],
                        'status' => $translation['status'],
                        'response_id' => $translation['response_id'],
                        'response_status' => $translation['response_status'],
                        'scheduled' => strtotime($translation['created_at']),
                    );
                }
            } else {
                $post = get_post($source_post_id);
                if ($post) {
                    $items[] = array(
                        'translation_id' => $translation['id'],
                        'post_id' => $source_post_id,
                        'post_title' => $post->post_title,
                        'post_type' => $post->post_type,
                        'target_lang' => $translation['target_lang'],
                        'status' => $translation['status'],
                        'response_id' => $translation['response_id'],
                        'response_status' => $translation['response_status'],
                        'scheduled' => strtotime($translation['created_at']),
                    );
                }
            }
        }

        return $items;
    }
}
