<?php
/**
 * LLM API Wrapper for OpenAI
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_LLM_API
{

    /**
     * OpenAI API endpoint
     */
    const API_ENDPOINT = 'https://api.openai.com/v1/responses';

    /**
     * API timeout
     */
    const API_TIMEOUT = 30;

    /**
     * Default model
     */
    const DEFAULT_MODEL = 'gpt-5-nano';

    /**
     * Default translation prompt template
     */
    const DEFAULT_PROMPT = 'You are a professional translation engine, not a conversational assistant.

TASK:
Translate all translatable text values in the provided JSON from {{source_language}} to {{destination_language}}.

STRICT RULES:
- Do NOT ask questions.
- Do NOT request confirmation.
- Do NOT comment on content length.
- Do NOT explain your actions.
- Do NOT add notes, warnings, or metadata.
- Do NOT split the output into parts.
- Always return the FULL translation in a single response.

OUTPUT FORMAT:
- Return a JSON object with the exact same structure and keys as the input.
- Translate only human-readable text.
- Preserve all HTML tags, attributes, URLs, IDs, class names, and structure.
- Preserve original newline characters as real line breaks (use \\n, never \\\\n).
- Do NOT re-serialize or stringify JSON values.
- Return ONLY valid, directly parseable JSON.

{{website_context}}';

    /**
     * API key
     */
    private $api_key;

    /**
     * Logger
     */
    private $logger;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->api_key = get_option('latwaitr_openai_api_key');
        $this->logger = LATWAITR()->logger;
    }

    /**
     * Get language name
     */
    private function get_language_name($code)
    {
        return apply_filters( 'wpml_translated_language_name', NULL, $code, "en" );
    }

    /**
     * Get the default prompt template
     *
     * @return string
     */
    public static function get_default_prompt()
    {
        return self::DEFAULT_PROMPT;
    }

    /**
     * Build JSON translation prompt for Responses API
     */
    private function build_json_translation_prompt($content, $source_lang, $target_lang)
    {
        // Get custom prompt or use default
        $prompt_template = get_option('latwaitr_translation_prompt', '');
        if (empty($prompt_template)) {
            $prompt_template = self::DEFAULT_PROMPT;
        }

        // Replace language variables
        $instruction = str_replace(
            array('{{source_language}}', '{{destination_language}}'),
            array($this->get_language_name($source_lang), $this->get_language_name($target_lang)),
            $prompt_template
        );

        // Handle website context variable
        $website_description = get_option('latwaitr_website_description', '');
        if (!empty($website_description)) {
            $website_context = "Website context:\n" . $website_description;
        } else {
            $website_context = '';
        }
        $instruction = str_replace('{{website_context}}', $website_context, $instruction);

        // Prepare content as JSON (do NOT escape Unicode or slashes)
        $content_json = json_encode(
            $content,
            JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT
        );

        // Final input for Responses API
        return $instruction . "\n\nINPUT JSON:\n" . $content_json;
    }

    /**
     * Make API request with JSON response format (Responses API)
     */
    private function make_json_request($input_text, $model = null)
    {
        if ($model === null) {
            $model = self::DEFAULT_MODEL;
        }

        $body = array(
            'model' => $model,
            "reasoning" => array(
                "effort" => "minimal"
            ),
            'input' => $input_text,
            'text' => array(
                'format' => array(
                    'type' => 'json_object'
                )
            )
        );

        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode($body),
            'timeout' => self::API_TIMEOUT,
            'method' => 'POST',
        );

        $this->logger->debug('OpenAI API Request', array(
            'endpoint' => self::API_ENDPOINT,
            'method' => 'POST',
            'model' => $model,
            'input_length' => strlen($input_text),
            'request_body' => $body,
        ));

        $response = wp_remote_request(self::API_ENDPOINT, $args);

        if (is_wp_error($response)) {
            $this->logger->error('OpenAI API Request Failed (WP Error)', array(
                'error_message' => $response->get_error_message(),
                'error_code' => $response->get_error_code(),
            ));
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        $this->logger->debug('OpenAI API Response', array(
            'status_code' => $status_code,
            'response_body' => $data,
        ));

        if ($status_code !== 200) {
            $error_message = $data['error']['message'] ?? 'Unknown API error';
            $error_type = $data['error']['type'] ?? 'unknown';
            $error_code = $data['error']['code'] ?? 'unknown';
            $this->logger->error('OpenAI API Error Response', array(
                'status_code' => $status_code,
                'error_type' => $error_type,
                'error_code' => $error_code,
                'error_message' => $error_message,
                'error_param' => $data['error']['param'] ?? null,
                'response_body' => $body,
            ));
            return new WP_Error('api_error', $error_message, array('status' => $status_code));
        }

        return $data;
    }

    /**
     * Submit background translation request (non-blocking)
     *
     * @param array $content Content to translate
     * @param string $source_lang Source language
     * @param string $target_lang Target language
     * @return array|WP_Error Response with response_id or error
     */
    public function submit_background_translation($content, $source_lang, $target_lang)
    {
        if (empty($this->api_key)) {
            return new WP_Error('no_api_key', __('OpenAI API key is not configured', 'latw-ai-translator-for-wpml'));
        }

        if (empty($content)) {
            return new WP_Error('empty_content', __('Content to translate is empty', 'latw-ai-translator-for-wpml'));
        }

        // Build the JSON translation prompt
        $prompt = $this->build_json_translation_prompt($content, $source_lang, $target_lang);

        // Get model
        $model = get_option('latwaitr_openai_model', self::DEFAULT_MODEL);

        $body = array(
            'model' => $model,
            'reasoning' => array(
                'effort' => 'minimal'
            ),
            'input' => $prompt,
            'text' => array(
                'format' => array(
                    'type' => 'json_object'
                )
            ),
            'background' => true
        );

        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode($body),
            'timeout' => 30,
            'method' => 'POST',
        );

        $this->logger->debug('OpenAI Background Translation Request', array(
            'endpoint' => self::API_ENDPOINT,
            'method' => 'POST',
            'model' => $model,
            'source_lang' => $source_lang,
            'target_lang' => $target_lang,
            'input_length' => strlen($prompt),
            'request_body' => $body,
        ));

        $response = wp_remote_request(self::API_ENDPOINT, $args);

        if (is_wp_error($response)) {
            $this->logger->error('OpenAI Background Translation Request Failed (WP Error)', array(
                'source_lang' => $source_lang,
                'target_lang' => $target_lang,
                'error_message' => $response->get_error_message(),
                'error_code' => $response->get_error_code(),
                'error_data' => $response->get_error_data(),
            ));
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body_response = wp_remote_retrieve_body($response);
        $data = json_decode($body_response, true);

        if ($data === null && !empty($body_response)) {
            $this->logger->error('OpenAI Background Translation - Failed to parse JSON response', array(
                'status_code' => $status_code,
                'json_error' => json_last_error_msg(),
                'raw_response_preview' => substr($body_response, 0, 1000),
            ));
            return new WP_Error('json_parse_error', 'Failed to parse API response: ' . json_last_error_msg());
        }

        $this->logger->debug('OpenAI Background Translation Response', array(
            'status_code' => $status_code,
            'response_body' => $data,
        ));

        if ($status_code !== 200) {
            $error_message = $data['error']['message'] ?? 'Unknown API error';
            $error_type = $data['error']['type'] ?? 'unknown';
            $error_code = $data['error']['code'] ?? 'unknown';
            $this->logger->error('OpenAI Background Translation Error', array(
                'status_code' => $status_code,
                'error_type' => $error_type,
                'error_code' => $error_code,
                'error_message' => $error_message,
                'error_param' => $data['error']['param'] ?? null,
                'source_lang' => $source_lang,
                'target_lang' => $target_lang,
                'model' => $model,
                'response_body' => $body_response,
            ));
            return new WP_Error('api_error', $error_message, array('status' => $status_code));
        }

        $this->logger->info('Background translation submitted', array(
            'response_id' => $data['id'],
            'status' => $data['status']
        ));

        return $data;
    }

    /**
     * Get response status from OpenAI
     *
     * @param string $response_id OpenAI response ID
     * @return array|WP_Error Response status data or error
     */
    public function get_response_status($response_id)
    {
        $endpoint = self::API_ENDPOINT . '/' . $response_id;

        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->api_key,
            ),
            'timeout' => self::API_TIMEOUT,
            'method' => 'GET',
        );

        $this->logger->debug('OpenAI Get Response Status Request', array(
            'endpoint' => $endpoint,
            'method' => 'GET',
            'response_id' => $response_id,
        ));

        $response = wp_remote_request($endpoint, $args);

        if (is_wp_error($response)) {
            $this->logger->error('OpenAI Get Response Status Failed (WP Error)', array(
                'response_id' => $response_id,
                'endpoint' => $endpoint,
                'error_message' => $response->get_error_message(),
                'error_code' => $response->get_error_code(),
                'error_data' => $response->get_error_data(),
            ));
            return $response;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);

        if ($data === null && !empty($body)) {
            $this->logger->error('OpenAI Get Response Status - Failed to parse JSON response', array(
                'response_id' => $response_id,
                'status_code' => $status_code,
                'json_error' => json_last_error_msg(),
                'raw_response_preview' => substr($body, 0, 1000),
            ));
            return new WP_Error('json_parse_error', 'Failed to parse status response: ' . json_last_error_msg());
        }

        $this->logger->debug('OpenAI Get Response Status Response', array(
            'response_id' => $response_id,
            'status_code' => $status_code,
            'response_status' => $data['status'] ?? 'unknown',
            'response_body' => $data,
        ));

        if ($status_code !== 200) {
            $error_message = $data['error']['message'] ?? 'Unknown error during status check';
            $error_type = $data['error']['type'] ?? 'unknown';
            $error_code = $data['error']['code'] ?? 'unknown';
            $this->logger->error('OpenAI Get Response Status Error', array(
                'response_id' => $response_id,
                'status_code' => $status_code,
                'error_type' => $error_type,
                'error_code' => $error_code,
                'error_message' => $error_message,
                'error_param' => $data['error']['param'] ?? null,
                'response_body' => $body,
            ));
            return new WP_Error('response_status_failed', $error_message);
        }

        return $data;
    }
}
