<?php
/**
 * ACF Integration for LATW
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_ACF_Integration {

    /**
     * Logger
     */
    private $logger;

    /**
     * LLM API
     */
    private $llm;

    /**
     * Supported field types
     */
    private $supported_fields = array(
        'text',
        'textarea',
        'wysiwyg',
        'repeater',
        'flexible_content',
        'group',
    );

    /**
     * Constructor
     */
    public function __construct() {
        $this->logger = LATWAITR()->logger;

        if (!$this->is_acf_active()) {
            return;
        }

        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        add_filter('latwaitr_content_to_translate', array($this, 'add_acf_fields'), 10, 2);
        add_action('latwaitr_after_translation', array($this, 'update_acf_fields'), 10, 3);
    }

    /**
     * Check if ACF is active
     */
    public function is_acf_active() {
        return function_exists('get_field') && function_exists('get_field_objects');
    }

    /**
     * Add ACF fields to content to translate
     */
    public function add_acf_fields($content, $post_id) {
        $acf_fields = $this->get_translatable_acf_fields($post_id);

        if (!empty($acf_fields)) {
            if (!isset($content['acf'])) {
                $content['acf'] = array();
            }

            $content['acf'] = array_merge($content['acf'], $acf_fields);
        }

        return $content;
    }

    /**
     * Get translatable ACF fields
     */
    public function get_translatable_acf_fields($post_id) {
        if (!$this->is_acf_active()) {
            return array();
        }

        $field_objects = get_field_objects($post_id);

        if (!$field_objects) {
            return array();
        }

        $translatable = array();

        foreach ($field_objects as $field_key => $field) {
            if (!$this->is_field_translatable($field)) {
                continue;
            }

            $value = $this->get_field_value($field, $post_id);

            if (!empty($value)) {
                $translatable[$field_key] = $value;
            }
        }

        return $translatable;
    }

    /**
     * Check if field is translatable
     */
    private function is_field_translatable($field) {
        // Check if field type is supported
        if (!in_array($field['type'], $this->supported_fields)) {
            return false;
        }

        // Check if field is marked as translatable (custom ACF setting)
        if (isset($field['latwaitr_translate']) && $field['latwaitr_translate'] === false) {
            return false;
        }

        return true;
    }

    /**
     * Get field value for translation
     */
    private function get_field_value($field, $post_id) {
        $type = $field['type'];
        $name = $field['name'];

        switch ($type) {
            case 'text':
            case 'textarea':
            case 'wysiwyg':
                return get_field($name, $post_id, false);

            case 'repeater':
                return $this->get_repeater_value($name, $post_id);

            case 'flexible_content':
                return $this->get_flexible_content_value($name, $post_id);

            case 'group':
                return $this->get_group_value($name, $post_id);

            default:
                return null;
        }
    }

    /**
     * Get repeater field value
     */
    private function get_repeater_value($field_name, $post_id) {
        $rows = get_field($field_name, $post_id, false);

        if (!is_array($rows)) {
            return null;
        }

        $translatable_rows = array();

        foreach ($rows as $row_index => $row) {
            foreach ($row as $sub_field_name => $sub_field_value) {
                if (is_string($sub_field_value) && !empty($sub_field_value)) {
                    $translatable_rows[$row_index][$sub_field_name] = $sub_field_value;
                }
            }
        }

        return $translatable_rows;
    }

    /**
     * Get flexible content field value
     */
    private function get_flexible_content_value($field_name, $post_id) {
        $layouts = get_field($field_name, $post_id, false);

        if (!is_array($layouts)) {
            return null;
        }

        $translatable_layouts = array();

        foreach ($layouts as $layout_index => $layout) {
            foreach ($layout as $sub_field_name => $sub_field_value) {
                if ($sub_field_name === 'acf_fc_layout') {
                    continue; // Skip layout identifier
                }

                if (is_string($sub_field_value) && !empty($sub_field_value)) {
                    $translatable_layouts[$layout_index][$sub_field_name] = $sub_field_value;
                }
            }
        }

        return $translatable_layouts;
    }

    /**
     * Get group field value
     */
    private function get_group_value($field_name, $post_id) {
        $group = get_field($field_name, $post_id, false);

        if (!is_array($group)) {
            return null;
        }

        $translatable_group = array();

        foreach ($group as $sub_field_name => $sub_field_value) {
            if (is_string($sub_field_value) && !empty($sub_field_value)) {
                $translatable_group[$sub_field_name] = $sub_field_value;
            }
        }

        return $translatable_group;
    }

    /**
     * Update ACF fields with translations
     */
    public function update_acf_fields($source_post_id, $translated_post_id, $translated_content) {
        if (!isset($translated_content['acf']) || !is_array($translated_content['acf'])) {
            return;
        }

        foreach ($translated_content['acf'] as $field_key => $value) {
            update_field($field_key, $value, $translated_post_id);
        }

        $this->logger->debug('ACF fields updated', array(
            'source_post_id' => $source_post_id,
            'translated_post_id' => $translated_post_id,
            'fields_count' => count($translated_content['acf']),
        ));
    }

    /**
     * Translate ACF field value
     */
    public function translate_acf_field($field_value, $source_lang, $target_lang, $field_type = 'text') {
        if (empty($field_value)) {
            return $field_value;
        }

        $this->llm = LATWAITR()->llm;

        switch ($field_type) {
            case 'repeater':
            case 'flexible_content':
                return $this->translate_complex_field($field_value, $source_lang, $target_lang);

            case 'group':
                return $this->translate_group_field($field_value, $source_lang, $target_lang);

            default:
                $result = $this->llm->translate($field_value, $source_lang, $target_lang, array(
                    'field_type' => 'acf',
                    'preserve_html' => $field_type === 'wysiwyg',
                ));

                if (is_wp_error($result)) {
                    return $field_value; // Return original on error
                }

                return $result['translated_text'];
        }
    }

    /**
     * Translate complex field (repeater, flexible content)
     */
    private function translate_complex_field($field_value, $source_lang, $target_lang) {
        if (!is_array($field_value)) {
            return $field_value;
        }

        $translated = array();

        foreach ($field_value as $row_index => $row) {
            $translated[$row_index] = array();

            foreach ($row as $sub_field_name => $sub_field_value) {
                if ($sub_field_name === 'acf_fc_layout') {
                    $translated[$row_index][$sub_field_name] = $sub_field_value;
                    continue;
                }

                if (is_string($sub_field_value)) {
                    $result = $this->llm->translate($sub_field_value, $source_lang, $target_lang, array(
                        'field_type' => 'acf',
                    ));

                    $translated[$row_index][$sub_field_name] = is_wp_error($result) ? $sub_field_value : $result['translated_text'];
                } else {
                    $translated[$row_index][$sub_field_name] = $sub_field_value;
                }
            }
        }

        return $translated;
    }

    /**
     * Translate group field
     */
    private function translate_group_field($field_value, $source_lang, $target_lang) {
        if (!is_array($field_value)) {
            return $field_value;
        }

        $translated = array();

        foreach ($field_value as $sub_field_name => $sub_field_value) {
            if (is_string($sub_field_value)) {
                $result = $this->llm->translate($sub_field_value, $source_lang, $target_lang, array(
                    'field_type' => 'acf',
                ));

                $translated[$sub_field_name] = is_wp_error($result) ? $sub_field_value : $result['translated_text'];
            } else {
                $translated[$sub_field_name] = $sub_field_value;
            }
        }

        return $translated;
    }

    /**
     * Get supported field types
     */
    public function get_supported_field_types() {
        return $this->supported_fields;
    }
}
