/**
 * Admin JavaScript for LLM Translator
 */
(function($) {
    'use strict';

    $(document).ready(function() {
        // Translations Panel - Select All Posts
        $('.latwaitr-select-all-posts').on('change', function() {
            var postType = $(this).data('post-type');
            var checked = $(this).is(':checked');
            $('.latwaitr-select-post[data-post-type="' + postType + '"]').prop('checked', checked);
        });

        // Translations Panel - Select All Languages for a post
        $('.latwaitr-select-all-langs').on('change', function() {
            var postId = $(this).data('post-id');
            var checked = $(this).is(':checked');
            // Only affect non-disabled and non-locked checkboxes
            $('.latwaitr-select-lang[data-post-id="' + postId + '"]').each(function() {
                if (!isLanguageLocked($(this)) && !$(this).prop('disabled')) {
                    $(this).prop('checked', checked);
                }
            });
        });

        // Translations Panel - Individual language checkbox changed
        $('.latwaitr-select-lang').on('change', function() {
            var postId = $(this).data('post-id');
            // Only count non-locked checkboxes for the "all" state
            var $availableLangs = $('.latwaitr-select-lang[data-post-id="' + postId + '"]').filter(function() {
                return !isLanguageLocked($(this));
            });
            var $allLangsCheckbox = $('.latwaitr-select-all-langs[data-post-id="' + postId + '"]');

            // Update "All Languages" checkbox based on available (non-locked) checkboxes
            var $enabledLangs = $availableLangs.filter(':not(:disabled)');
            var allChecked = $enabledLangs.length > 0 && $enabledLangs.length === $enabledLangs.filter(':checked').length;
            $allLangsCheckbox.prop('checked', allChecked);
        });

        /**
         * Check if a language checkbox is locked
         */
        function isLanguageLocked($checkbox) {
            return $checkbox.data('is-locked') === 1 || $checkbox.data('is-locked') === '1';
        }

        /**
         * Check if a language code is supported
         */
        function isLanguageCodeSupported(langCode) {
            var normalized = langCode.toLowerCase().substring(0, 2);
            return normalized === "en";
        }

        // Translations Panel - Translate Single Post
        $('.latwaitr-translate-single').on('click', function(e) {
            e.preventDefault();
            var $button = $(this);
            var postId = $button.data('post-id');

            // Get selected languages for this post
            var selectedLangs = getSelectedLanguagesForPost(postId);

            if (selectedLangs.length === 0) {
                alert('Please select at least one language to translate');
                return;
            }

            queueTranslations([postId], selectedLangs, $button);
        });

        // Translations Panel - Translate Selected
        $('.latwaitr-translate-selected').on('click', function(e) {
            e.preventDefault();
            var $button = $(this);
            var postType = $button.data('post-type');
            var $selected = $('.latwaitr-select-post[data-post-type="' + postType + '"]:checked');

            if ($selected.length === 0) {
                alert('Please select at least one post to translate');
                return;
            }

            var postIds = [];
            var allSelectedLangs = [];
            $selected.each(function() {
                var postId = $(this).val();
                postIds.push(postId);

                // Get selected languages for this post
                var selectedLangs = getSelectedLanguagesForPost(postId);
                $.each(selectedLangs, function(i, lang) {
                    if (allSelectedLangs.indexOf(lang) === -1) {
                        allSelectedLangs.push(lang);
                    }
                });
            });

            if (allSelectedLangs.length === 0) {
                alert('Please select at least one language to translate');
                return;
            }

            if (!confirm('Queue ' + postIds.length + ' post(s) for translation?')) {
                return;
            }

            queueTranslations(postIds, allSelectedLangs, $button);
        });

        /**
         * Get selected languages for a specific post
         */
        function getSelectedLanguagesForPost(postId) {
            var selectedLangs = [];
            $('.latwaitr-select-lang[data-post-id="' + postId + '"]:checked').each(function() {
                // Skip locked languages
                if (!isLanguageLocked($(this))) {
                    selectedLangs.push($(this).data('lang-code'));
                }
            });
            return selectedLangs;
        }

        /**
         * Queue translations via AJAX
         */
        function queueTranslations(postIds, targetLanguages, $button) {
            // Disable button
            $button.prop('disabled', true);
            var originalText = $button.text();
            $button.text('Queueing...');

            // Disable and mark language checkboxes immediately
            $.each(postIds, function(i, postId) {
                $.each(targetLanguages, function(j, langCode) {
                    var $checkbox = $('.latwaitr-select-lang[data-post-id="' + postId + '"][data-lang-code="' + langCode + '"]');
                    $checkbox.prop('disabled', true).prop('checked', false);
                    $checkbox.closest('label').css('opacity', '0.6');

                    // Add status indicator
                    var $statusSpan = $('<span style="color: #f0b849; font-size: 11px;"> (⏳ Queued)</span>');
                    $checkbox.parent().append($statusSpan);
                });
            });

            $.ajax({
                url: latwaitrAdmin.ajax_url,
                type: 'POST',
                data: {
                    action: 'latwaitr_queue_translations',
                    post_ids: postIds,
                    target_languages: targetLanguages,
                    _wpnonce: latwaitrAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        alert(response.data.message);
                        // Reload page to refresh queue
                        location.reload();
                    } else {
                        alert('Error: ' + (response.data.message || 'Failed to queue translations'));
                        // Re-enable button and checkboxes on error
                        $button.prop('disabled', false);
                        $button.text(originalText);

                        // Re-enable checkboxes on error
                        $.each(postIds, function(i, postId) {
                            $.each(targetLanguages, function(j, langCode) {
                                var $checkbox = $('.latwaitr-select-lang[data-post-id="' + postId + '"][data-lang-code="' + langCode + '"]');
                                $checkbox.prop('disabled', false).prop('checked', true);
                                $checkbox.closest('label').css('opacity', '1');
                                $checkbox.parent().find('span').remove();
                            });
                        });
                    }
                },
                error: function(xhr, status, error) {
                    alert('AJAX Error: ' + error);
                    // Re-enable button and checkboxes on error
                    $button.prop('disabled', false);
                    $button.text(originalText);

                    // Re-enable checkboxes on error
                    $.each(postIds, function(i, postId) {
                        $.each(targetLanguages, function(j, langCode) {
                            var $checkbox = $('.latwaitr-select-lang[data-post-id="' + postId + '"][data-lang-code="' + langCode + '"]');
                            $checkbox.prop('disabled', false).prop('checked', true);
                            $checkbox.closest('label').css('opacity', '1');
                            $checkbox.parent().find('span').remove();
                        });
                    });
                }
            });
        }

        // ========================================
        // CATEGORIES TAB - TERM TRANSLATIONS
        // ========================================

        // Select all terms
        $('.latwaitr-select-all-terms').on('change', function() {
            var checked = $(this).is(':checked');
            $('.latwaitr-select-term').prop('checked', checked);
        });

        // Select all languages for a term
        $(document).on('change', '.latwaitr-term-select-all-langs', function() {
            var termId = $(this).data('term-id');
            var checked = $(this).is(':checked');
            $('.latwaitr-term-select-lang[data-term-id="' + termId + '"]').each(function() {
                if (!$(this).prop('disabled')) {
                    $(this).prop('checked', checked);
                }
            });
        });

        // Individual term language checkbox changed
        $(document).on('change', '.latwaitr-term-select-lang', function() {
            var termId = $(this).data('term-id');
            var $availableLangs = $('.latwaitr-term-select-lang[data-term-id="' + termId + '"]');
            var $allLangsCheckbox = $('.latwaitr-term-select-all-langs[data-term-id="' + termId + '"]');
            var $enabledLangs = $availableLangs.filter(':not(:disabled)');
            var allChecked = $enabledLangs.length > 0 && $enabledLangs.length === $enabledLangs.filter(':checked').length;
            $allLangsCheckbox.prop('checked', allChecked);
        });

        // Translate single term
        $(document).on('click', '.latwaitr-translate-single-term', function(e) {
            e.preventDefault();
            var $button = $(this);
            var termId = $button.data('term-id');

            var selectedLangs = getSelectedLanguagesForTerm(termId);

            if (selectedLangs.length === 0) {
                alert('Please select at least one language to translate');
                return;
            }

            queueTermTranslations([termId], selectedLangs, $button);
        });

        // Translate selected terms
        $(document).on('click', '.latwaitr-translate-selected-terms', function(e) {
            e.preventDefault();
            var $button = $(this);
            var $selected = $('.latwaitr-select-term:checked');

            if ($selected.length === 0) {
                alert('Please select at least one term to translate');
                return;
            }

            var termIds = [];
            var allSelectedLangs = [];
            $selected.each(function() {
                var termId = $(this).val();
                termIds.push(termId);

                var selectedLangs = getSelectedLanguagesForTerm(termId);
                $.each(selectedLangs, function(i, lang) {
                    if (allSelectedLangs.indexOf(lang) === -1) {
                        allSelectedLangs.push(lang);
                    }
                });
            });

            if (allSelectedLangs.length === 0) {
                alert('Please select at least one language to translate');
                return;
            }

            if (!confirm('Queue ' + termIds.length + ' term(s) for translation?')) {
                return;
            }

            queueTermTranslations(termIds, allSelectedLangs, $button);
        });

        /**
         * Get selected languages for a specific term
         */
        function getSelectedLanguagesForTerm(termId) {
            var selectedLangs = [];
            $('.latwaitr-term-select-lang[data-term-id="' + termId + '"]:checked').each(function() {
                selectedLangs.push($(this).data('lang-code'));
            });
            return selectedLangs;
        }

        /**
         * Queue term translations via AJAX
         */
        function queueTermTranslations(termIds, targetLanguages, $button) {
            $button.prop('disabled', true);
            var originalText = $button.text();
            $button.text('Queueing...');

            // Disable and mark language checkboxes immediately
            $.each(termIds, function(i, termId) {
                $.each(targetLanguages, function(j, langCode) {
                    var $checkbox = $('.latwaitr-term-select-lang[data-term-id="' + termId + '"][data-lang-code="' + langCode + '"]');
                    $checkbox.prop('disabled', true).prop('checked', false);
                    $checkbox.closest('label').css('opacity', '0.6');
                    var $statusSpan = $('<span style="color: #f0b849; font-size: 11px;"> (⏳ Queued)</span>');
                    $checkbox.parent().append($statusSpan);
                });
            });

            $.ajax({
                url: latwaitrAdmin.ajax_url,
                type: 'POST',
                data: {
                    action: 'latwaitr_queue_term_translations',
                    term_ids: termIds,
                    target_languages: targetLanguages,
                    _wpnonce: latwaitrAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        alert(response.data.message);
                        location.reload();
                    } else {
                        alert('Error: ' + (response.data.message || 'Failed to queue translations'));
                        $button.prop('disabled', false);
                        $button.text(originalText);

                        $.each(termIds, function(i, termId) {
                            $.each(targetLanguages, function(j, langCode) {
                                var $checkbox = $('.latwaitr-term-select-lang[data-term-id="' + termId + '"][data-lang-code="' + langCode + '"]');
                                $checkbox.prop('disabled', false).prop('checked', true);
                                $checkbox.closest('label').css('opacity', '1');
                                $checkbox.parent().find('span').remove();
                            });
                        });
                    }
                },
                error: function(xhr, status, error) {
                    alert('AJAX Error: ' + error);
                    $button.prop('disabled', false);
                    $button.text(originalText);

                    $.each(termIds, function(i, termId) {
                        $.each(targetLanguages, function(j, langCode) {
                            var $checkbox = $('.latwaitr-term-select-lang[data-term-id="' + termId + '"][data-lang-code="' + langCode + '"]');
                            $checkbox.prop('disabled', false).prop('checked', true);
                            $checkbox.closest('label').css('opacity', '1');
                            $checkbox.parent().find('span').remove();
                        });
                    });
                }
            });
        }

        // ========================================
        // CHECK RESPONSES (MANUAL CRON)
        // ========================================
        $('#latwaitr-check-responses-btn').on('click', function(e) {
            e.preventDefault();
            var $button = $(this);
            var originalHtml = $button.html();
            $button.prop('disabled', true);
            $button.html('<span class="dashicons dashicons-update dashicons-update-spin" style="vertical-align: middle; margin-top: -2px;"></span> Checking...');

            $.ajax({
                url: latwaitrAdmin.ajax_url,
                type: 'POST',
                data: {
                    action: 'latwaitr_check_responses',
                    _wpnonce: latwaitrAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        alert(response.data.message);
                        location.reload();
                    } else {
                        alert('Error: ' + (response.data.message || 'Failed to check responses'));
                    }
                },
                error: function(xhr, status, error) {
                    alert('AJAX Error: ' + error);
                },
                complete: function() {
                    $button.prop('disabled', false);
                    $button.html(originalHtml);
                }
            });
        });

        // ========================================
        // TRANSLATE AGAIN MODAL
        // ========================================

        var $modal = $('#latwaitr-translate-again-modal');
        var $step1 = $('.latwaitr-modal-step-1');
        var $step2 = $('.latwaitr-modal-step-2');
        var currentPostTypeData = null;
        var currentMode = 'posts'; // 'posts' or 'terms'
        var currentTermsData = null;

        // Open modal
        $('#latwaitr-translate-again-btn').on('click', function() {
            $modal.fadeIn(200);
            $step1.show();
            $step2.hide();
            $('#latwaitr-retranslate-post-type').val('');
        });

        // Close modal
        $('.latwaitr-modal-close, .latwaitr-modal-overlay').on('click', function() {
            $modal.fadeOut(200);
        });

        // Prevent modal content clicks from closing modal
        $('.latwaitr-modal-content').on('click', function(e) {
            e.stopPropagation();
        });

        // Load posts/terms when post type selected
        $('#latwaitr-load-translated-posts').on('click', function() {
            var postType = $('#latwaitr-retranslate-post-type').val();

            if (!postType) {
                alert('Please select a post type');
                return;
            }

            var $button = $(this);
            $button.prop('disabled', true).text('Loading...');

            function handleAjaxError(xhr, status, error, label) {
                var errorMessage = error || status || 'Unknown error';
                if (xhr.responseText) {
                    try {
                        var resp = JSON.parse(xhr.responseText);
                        if (resp.data && resp.data.message) {
                            errorMessage = resp.data.message;
                        }
                    } catch (e) {
                        if (xhr.responseText.length < 500) {
                            errorMessage = xhr.responseText;
                        }
                    }
                }
                alert('Error loading ' + label + ': ' + errorMessage);
                $button.prop('disabled', false).text('Load Posts');
            }

            if (postType === 'categories') {
                currentMode = 'terms';
                $.ajax({
                    url: latwaitrAdmin.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'latwaitr_get_translated_terms',
                        _wpnonce: latwaitrAdmin.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            currentTermsData = response.data;
                            renderRetranslationTerms(response.data);
                            $step1.hide();
                            $step2.show();
                        } else {
                            alert('Error: ' + (response.data.message || 'Failed to load terms'));
                        }
                        $button.prop('disabled', false).text('Load Posts');
                    },
                    error: function(xhr, status, error) { handleAjaxError(xhr, status, error, 'terms'); }
                });
            } else {
                currentMode = 'posts';
                $.ajax({
                    url: latwaitrAdmin.ajax_url,
                    type: 'POST',
                    data: {
                        action: 'latwaitr_get_translated_posts',
                        post_type: postType,
                        _wpnonce: latwaitrAdmin.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            currentPostTypeData = response.data;
                            renderRetranslationPosts(response.data);
                            $step1.hide();
                            $step2.show();
                        } else {
                            alert('Error: ' + (response.data.message || 'Failed to load posts'));
                        }
                        $button.prop('disabled', false).text('Load Posts');
                    },
                    error: function(xhr, status, error) { handleAjaxError(xhr, status, error, 'posts'); }
                });
            }
        });

        // Render posts in modal
        function renderRetranslationPosts(data) {
            var posts = data.posts;
            var activeLanguages = data.active_languages;
            var $container = $('#latwaitr-retranslate-posts-container');

            if (!posts || posts.length === 0) {
                $container.html('<p>No posts with translations found for this post type.</p>');
                return;
            }

            var html = '<table class="wp-list-table widefat fixed striped">';
            html += '<thead><tr>';
            html += '<th style="width: 40px;"><input type="checkbox" id="latwaitr-retranslate-select-all"></th>';
            html += '<th>Title</th>';
            html += '<th>Status</th>';
            html += '<th>Source Language</th>';
            html += '<th>Existing Translations</th>';
            html += '<th>Last Modified</th>';
            html += '</tr></thead>';
            html += '<tbody>';

            var renderedPostsCount = 0;

            $.each(posts, function(i, post) {
                // Filter out locked languages
                var availableTranslations = [];
                $.each(post.existing_translations, function(j, langCode) {
                    if (isLanguageCodeSupported(langCode)) {
                        availableTranslations.push(langCode);
                    }
                });

                // Skip posts that have no available translations
                if (availableTranslations.length === 0) {
                    return true; // continue to next post
                }

                renderedPostsCount++;

                html += '<tr>';
                html += '<td><input type="checkbox" class="latwaitr-retranslate-select-post" value="' + post.id + '"></td>';
                html += '<td><strong>' + escapeHtml(post.title) + '</strong></td>';
                html += '<td>';
                if (post.status === 'publish') {
                    html += '<span style="color: #00a32a;">● Published</span>';
                } else {
                    html += '<span style="color: #dba617;">● Draft</span>';
                }
                html += '</td>';

                // Source language
                var sourceLangName = activeLanguages[post.source_lang] ?
                    activeLanguages[post.source_lang].native_name : post.source_lang;
                html += '<td>' + escapeHtml(sourceLangName) + '</td>';

                // Existing translations with checkboxes (only available languages)
                html += '<td><div class="latwaitr-language-checkboxes">';
                html += '<label style="display: block; margin-bottom: 5px;">';
                html += '<input type="checkbox" class="latwaitr-retranslate-select-all-langs" data-post-id="' + post.id + '" checked>';
                html += '<strong>All Languages</strong>';
                html += '</label>';

                $.each(availableTranslations, function(j, langCode) {
                    var langName = activeLanguages[langCode] ?
                        activeLanguages[langCode].display_name : langCode;
                    html += '<label style="display: block; margin-left: 15px;">';
                    html += '<input type="checkbox" class="latwaitr-retranslate-select-lang" data-post-id="' + post.id + '" data-lang-code="' + langCode + '" data-is-locked="0" checked>';
                    html += escapeHtml(langName);
                    html += '</label>';
                });

                html += '</div></td>';
                html += '<td>' + escapeHtml(post.modified) + '</td>';
                html += '</tr>';
            });

            // Check if any posts were actually rendered after filtering
            if (renderedPostsCount === 0) {
                $container.html('<p>No posts with translations found for this post type.</p>');
                return;
            }

            html += '</tbody></table>';
            $container.html(html);

            // Attach event handlers for new checkboxes
            attachRetranslationCheckboxHandlers();
        }

        // Render terms in retranslation modal (same structure as renderRetranslationPosts)
        function renderRetranslationTerms(data) {
            var terms = data.terms;
            var activeLanguages = data.active_languages;
            var $container = $('#latwaitr-retranslate-posts-container');

            if (!terms || terms.length === 0) {
                $container.html('<p>No terms with translations found.</p>');
                return;
            }

            var html = '<table class="wp-list-table widefat fixed striped">';
            html += '<thead><tr>';
            html += '<th style="width: 40px;"><input type="checkbox" id="latwaitr-retranslate-select-all"></th>';
            html += '<th>Name</th>';
            html += '<th>Taxonomy</th>';
            html += '<th>Source Language</th>';
            html += '<th>Existing Translations</th>';
            html += '</tr></thead>';
            html += '<tbody>';

            var renderedCount = 0;

            $.each(terms, function(i, term) {
                // Filter out locked languages (reuse same logic as posts)
                var availableTranslations = [];
                $.each(term.existing_translations, function(j, langCode) {
                    if (isLanguageCodeSupported(langCode)) {
                        availableTranslations.push(langCode);
                    }
                });

                if (availableTranslations.length === 0) {
                    return true; // continue
                }

                renderedCount++;

                var sourceLangName = activeLanguages[term.source_lang] ?
                    activeLanguages[term.source_lang].native_name : term.source_lang;

                // Use same classes and data-post-id as posts so attachRetranslationCheckboxHandlers() works
                html += '<tr>';
                html += '<td><input type="checkbox" class="latwaitr-retranslate-select-post" value="' + term.term_id + '"></td>';
                html += '<td><strong>' + escapeHtml(term.name) + '</strong></td>';
                html += '<td>' + escapeHtml(term.taxonomy_label) + '</td>';
                html += '<td>' + escapeHtml(sourceLangName) + '</td>';

                html += '<td><div class="latwaitr-language-checkboxes">';
                html += '<label style="display: block; margin-bottom: 5px;">';
                html += '<input type="checkbox" class="latwaitr-retranslate-select-all-langs" data-post-id="' + term.term_id + '" checked>';
                html += '<strong>All Languages</strong>';
                html += '</label>';

                $.each(availableTranslations, function(j, langCode) {
                    var langName = activeLanguages[langCode] ?
                        activeLanguages[langCode].display_name : langCode;
                    html += '<label style="display: block; margin-left: 15px;">';
                    html += '<input type="checkbox" class="latwaitr-retranslate-select-lang" data-post-id="' + term.term_id + '" data-lang-code="' + langCode + '" data-is-locked="0" checked>';
                    html += escapeHtml(langName);
                    html += '</label>';
                });

                html += '</div></td>';
                html += '</tr>';
            });

            if (renderedCount === 0) {
                $container.html('<p>No terms with translations found.</p>');
                return;
            }

            html += '</tbody></table>';
            $container.html(html);

            attachRetranslationCheckboxHandlers();
        }

        // Attach checkbox handlers for retranslation modal
        function attachRetranslationCheckboxHandlers() {
            // Select all posts
            $('#latwaitr-retranslate-select-all').on('change', function() {
                var checked = $(this).is(':checked');
                $('.latwaitr-retranslate-select-post').prop('checked', checked);
            });

            // Select all languages for a post (only non-locked)
            $(document).on('change', '.latwaitr-retranslate-select-all-langs', function() {
                var postId = $(this).data('post-id');
                var checked = $(this).is(':checked');
                // Only affect non-locked checkboxes
                $('.latwaitr-retranslate-select-lang[data-post-id="' + postId + '"]').each(function() {
                    if (!isLanguageLocked($(this))) {
                        $(this).prop('checked', checked);
                    }
                });
            });

            // Individual language checkbox changed
            $(document).on('change', '.latwaitr-retranslate-select-lang', function() {
                var postId = $(this).data('post-id');
                // Only count non-locked checkboxes
                var $availableLangs = $('.latwaitr-retranslate-select-lang[data-post-id="' + postId + '"]').filter(function() {
                    return !isLanguageLocked($(this));
                });
                var $allLangsCheckbox = $('.latwaitr-retranslate-select-all-langs[data-post-id="' + postId + '"]');

                var allChecked = $availableLangs.length > 0 && $availableLangs.length === $availableLangs.filter(':checked').length;
                $allLangsCheckbox.prop('checked', allChecked);
            });
        }

        // Get selected languages for a specific post in retranslation modal
        function getSelectedRetranslationLanguagesForPost(postId) {
            var selectedLangs = [];
            $('.latwaitr-retranslate-select-lang[data-post-id="' + postId + '"]:checked').each(function() {
                // Skip locked languages
                if (!isLanguageLocked($(this))) {
                    selectedLangs.push($(this).data('lang-code'));
                }
            });
            return selectedLangs;
        }

        // Translate selected posts/terms from modal
        $('#latwaitr-retranslate-selected').on('click', function() {
            var $button = $(this);

            if (currentMode === 'terms') {
                var $selected = $('.latwaitr-retranslate-select-post:checked');

                if ($selected.length === 0) {
                    alert('Please select at least one term to translate');
                    return;
                }

                var termIds = [];
                var allSelectedLangs = [];

                $selected.each(function() {
                    var termId = $(this).val();
                    termIds.push(termId);

                    var selectedLangs = getSelectedRetranslationLanguagesForPost(termId);
                    $.each(selectedLangs, function(i, lang) {
                        if (allSelectedLangs.indexOf(lang) === -1) {
                            allSelectedLangs.push(lang);
                        }
                    });
                });

                if (allSelectedLangs.length === 0) {
                    alert('Please select at least one language to translate');
                    return;
                }

                if (!confirm('Queue ' + termIds.length + ' term(s) for retranslation?')) {
                    return;
                }

                queueTermTranslations(termIds, allSelectedLangs, $button);
            } else {
                var $selected = $('.latwaitr-retranslate-select-post:checked');

                if ($selected.length === 0) {
                    alert('Please select at least one post to translate');
                    return;
                }

                var postIds = [];
                var allSelectedLangs = [];

                $selected.each(function() {
                    var postId = $(this).val();
                    postIds.push(postId);

                    var selectedLangs = getSelectedRetranslationLanguagesForPost(postId);
                    $.each(selectedLangs, function(i, lang) {
                        if (allSelectedLangs.indexOf(lang) === -1) {
                            allSelectedLangs.push(lang);
                        }
                    });
                });

                if (allSelectedLangs.length === 0) {
                    alert('Please select at least one language to translate');
                    return;
                }

                if (!confirm('Queue ' + postIds.length + ' post(s) for retranslation?')) {
                    return;
                }

                queueTranslations(postIds, allSelectedLangs, $button);
            }
        });

        // Back button
        $('#latwaitr-retranslate-back').on('click', function() {
            $step2.hide();
            $step1.show();
        });

        // Helper function to escape HTML
        function escapeHtml(text) {
            var map = {
                '&': '&amp;',
                '<': '&lt;',
                '>': '&gt;',
                '"': '&quot;',
                "'": '&#039;'
            };
            return text.replace(/[&<>"']/g, function(m) { return map[m]; });
        }

        // ========================================
        // SETTINGS PAGE - RESET PROMPT
        // ========================================
        $('#latwaitr_reset_prompt').on('click', function() {
            if (typeof latwaitrAdmin !== 'undefined' && latwaitrAdmin.defaultPrompt) {
                if (confirm(latwaitrAdmin.i18n.resetPromptConfirm || 'Are you sure you want to reset the prompt to default?')) {
                    $('#latwaitr_translation_prompt').val(latwaitrAdmin.defaultPrompt);
                }
            }
        });
    });

})(jQuery);
