<?php
/**
 * Admin Handler for LATW
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_Admin {

    /**
     * Constructor
     */
    public function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_filter('plugin_action_links_' . LATWAITR_PLUGIN_BASENAME, array($this, 'plugin_action_links'));
    }

    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        // Main menu page
        add_menu_page(
            __('AI Translator', 'latw-ai-translator-for-wpml'),
            __('AI Translator', 'latw-ai-translator-for-wpml'),
            'manage_options',
            'latwaitr-dashboard',
            array($this, 'render_dashboard'),
            'dashicons-translation',
            80
        );

        // Dashboard submenu
        add_submenu_page(
            'latwaitr-dashboard',
            __('Dashboard', 'latw-ai-translator-for-wpml'),
            __('Dashboard', 'latw-ai-translator-for-wpml'),
            'manage_options',
            'latwaitr-dashboard',
            array($this, 'render_dashboard')
        );

        // Translations submenu
        add_submenu_page(
            'latwaitr-dashboard',
            __('Translations', 'latw-ai-translator-for-wpml'),
            __('Translations', 'latw-ai-translator-for-wpml'),
            'manage_options',
            'latwaitr-translations',
            array($this, 'render_translations')
        );

        // Settings submenu
        add_submenu_page(
            'latwaitr-dashboard',
            __('Settings', 'latw-ai-translator-for-wpml'),
            __('Settings', 'latw-ai-translator-for-wpml'),
            'manage_options',
            'latwaitr-settings',
            array($this, 'render_settings')
        );

        // Logs
        add_submenu_page(
            'latwaitr-dashboard',
            __('Logs', 'latw-ai-translator-for-wpml'),
            __('Logs', 'latw-ai-translator-for-wpml'),
            'manage_options',
            'latwaitr-logs',
            array($this, 'render_logs')
        );
    }

    /**
     * Enqueue admin scripts and styles
     */
    public function enqueue_scripts($hook) {
        // Only load on our plugin pages
        if (strpos($hook, 'latw') === false) {
            return;
        }

        wp_enqueue_style(
            'latwaitr-admin-css',
            LATWAITR_PLUGIN_URL . 'admin/css/admin.css',
            array(),
            LATWAITR_VERSION
        );

        wp_enqueue_script(
            'latwaitr-admin-js',
            LATWAITR_PLUGIN_URL . 'admin/js/admin.js',
            array('jquery'),
            LATWAITR_VERSION,
            true
        );

        wp_localize_script('latwaitr-admin-js', 'latwaitrAdmin', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('latwaitr_action'),
            'strings' => array(
                'confirm_translate' => __('Are you sure you want to translate this post?', 'latw-ai-translator-for-wpml'),
                'translating' => __('Translating...', 'latw-ai-translator-for-wpml'),
                'success' => __('Translation completed successfully', 'latw-ai-translator-for-wpml'),
                'error' => __('Translation failed', 'latw-ai-translator-for-wpml'),
            ),
            'defaultPrompt' => LATWAITR_LLM_API::get_default_prompt(),
            'i18n' => array(
                'resetPromptConfirm' => __('Are you sure you want to reset the prompt to default?', 'latw-ai-translator-for-wpml'),
            ),
        ));
    }

    /**
     * Add plugin action links
     */
    public function plugin_action_links($links) {
        $settings_link = sprintf(
            '<a href="%s">%s</a>',
            admin_url('admin.php?page=latwaitr-settings'),
            __('Settings', 'latw-ai-translator-for-wpml')
        );

        array_unshift($links, $settings_link);

        return $links;
    }

    /**
     * Render dashboard page
     */
    public function render_dashboard() {
        $latwaitr_stats = LATWAITR_Database::get_stats(30);
        $latwaitr_queue_status = LATWAITR()->queue->get_queue_status();
        $latwaitr_monthly_cost = LATWAITR_Database::get_monthly_costs();

        include LATWAITR_PLUGIN_DIR . 'admin/views/dashboard.php';
    }

    /**
     * Render translations page
     */
    public function render_translations() {

        // Get target languages from settings
        $latwaitr_target_languages = get_option('latwaitr_target_languages', array());

        // Get URL parameters
        $latwaitr_current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : '';
        $latwaitr_current_page = isset($_GET['paged']) ? absint($_GET['paged']) : 1;
        $latwaitr_search_query = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
        $latwaitr_per_page = 20;

        // Get all post types with posts needing translation (for tabs)
        $latwaitr_post_types = $this->get_post_types_needing_translation($latwaitr_target_languages);

        // Get categories tab data
        $latwaitr_categories_tab = $this->get_taxonomy_tab_data($latwaitr_target_languages);

        // If no tab selected, use the first available post type
        if (empty($latwaitr_current_tab) && !empty($latwaitr_post_types)) {
            $first_type = reset($latwaitr_post_types);
            $latwaitr_current_tab = $first_type['name'];
        }

        // Check if categories tab is active
        $latwaitr_is_categories_tab = ($latwaitr_current_tab === 'categories');

        // Get posts/terms for current tab with pagination and search
        $latwaitr_posts_to_translate = array();
        $latwaitr_terms_data = null;

        if ($latwaitr_is_categories_tab) {
            $latwaitr_terms_data = $this->get_terms_needing_translation(
                $latwaitr_target_languages,
                $latwaitr_current_page,
                $latwaitr_per_page,
                $latwaitr_search_query
            );
        } elseif (!empty($latwaitr_current_tab)) {
            $latwaitr_posts_to_translate = $this->get_posts_needing_translation(
                $latwaitr_target_languages,
                $latwaitr_current_tab,
                $latwaitr_current_page,
                $latwaitr_per_page,
                $latwaitr_search_query
            );
        }

        // Get current tab data
        $latwaitr_current_tab_data = isset($latwaitr_posts_to_translate[$latwaitr_current_tab]) ? $latwaitr_posts_to_translate[$latwaitr_current_tab] : null;

        // Get queue status from database
        $latwaitr_queue_items = LATWAITR()->queue->get_queue_items();

        // Build map of queued languages per post: post_id => [lang_code => status]
        $latwaitr_queued_languages = array();
        foreach ($latwaitr_queue_items as $latwaitr_item) {
            $latwaitr_post_id = $latwaitr_item['post_id'];
            $latwaitr_lang = $latwaitr_item['target_lang'];
            $latwaitr_status = $latwaitr_item['status']; // 'queued' or 'in-progress'

            if (!isset($latwaitr_queued_languages[$latwaitr_post_id])) {
                $latwaitr_queued_languages[$latwaitr_post_id] = array();
            }

            $latwaitr_queued_languages[$latwaitr_post_id][$latwaitr_lang] = $latwaitr_status;
        }

        // Base URL for pagination and tabs
        $latwaitr_base_url = admin_url('admin.php?page=latwaitr-translations');

        include LATWAITR_PLUGIN_DIR . 'admin/views/translations.php';
    }

    /**
     * Get all post types that should be available for translation.
     * Includes all public post types plus selected private ones with translatable content.
     *
     * @return array Post type objects keyed by post type name
     */
    private function get_translatable_post_types() {
        $post_types = get_post_types(array('public' => true), 'objects');

        $private_translatable = array('wp_block', 'wp_template', 'wp_template_part');
        foreach ($private_translatable as $pt_name) {
            if (post_type_exists($pt_name) && !isset($post_types[$pt_name])) {
                $post_types[$pt_name] = get_post_type_object($pt_name);
            }
        }

        return $post_types;
    }

    /**
     * Get posts needing translation with pagination and search
     *
     * @param array  $target_languages Target languages
     * @param string $post_type        Specific post type to get (optional)
     * @param int    $page             Page number (1-indexed)
     * @param int    $per_page         Items per page
     * @param string $search           Search query
     * @return array Posts data with pagination info
     */
    private function get_posts_needing_translation($target_languages, $post_type = null, $page = 1, $per_page = 20, $search = '') {

        global $sitepress;
        $wpml = LATWAITR()->wpml;
        $posts_by_type = array();

        // Get translatable post types
        $post_types = $this->get_translatable_post_types();

        foreach ($post_types as $pt) {
            // If specific post type requested, skip others
            if ($post_type !== null && $pt->name !== $post_type) {
                continue;
            }

            $args = array(
                'post_type' => $pt->name,
                'post_status' => array('publish', 'draft'),
                'posts_per_page' => -1,
                'orderby' => 'modified',
                'order' => 'DESC',
                'suppress_filters' => true, // Get posts from all languages, not just current admin language
            );

            // Add search query if provided
            if (!empty($search)) {
                $args['s'] = $search;
            }

            $posts = get_posts($args);
            $processed_trids = array(); // Track processed translation groups
            $all_posts_needing_translation = array();

            foreach ($posts as $post) {
                // Get translation group ID (trid)
                $trid = $sitepress->get_element_trid($post->ID, 'post_' . $pt->name);

                // Skip if we already processed this translation group
                if (in_array($trid, $processed_trids)) {
                    continue;
                }

                // Get all translations first so we can identify the original language
                $translations = $sitepress->get_element_translations($trid, 'post_' . $pt->name);

                // Find the original post in this translation group
                $original_post_id = $post->ID;
                $source_lang = $wpml->get_post_language($post->ID);

                foreach ($translations as $lang_code => $translation) {
                    if (!empty($translation->original)) {
                        $original_post_id = $translation->element_id;
                        $source_lang = $lang_code;
                        break;
                    }
                }

                // Mark group as processed now that we know the original language
                // (must happen after resolving the original, so we don't lock out the group
                // when an English translation is encountered before the non-English original)
                $processed_trids[] = $trid;

                // Skip if original is English (nothing to translate to English)
                if (strtolower(substr($source_lang, 0, 2)) === 'en') {
                    continue;
                }

                // Check if English translation exists and if WPML flags it as needing update
                $translation_exists = false;
                $translation_id = null;
                $needs_update = false;

                foreach ($translations as $lang_code => $translation) {
                    if (strtolower(substr($lang_code, 0, 2)) === 'en' && !empty($translation->element_id)) {
                        $translation_exists = true;
                        $translation_id = $translation->element_id;
                        // Use WPML's needs_update flag instead of comparing post_modified dates
                        $needs_update = !empty($translation->needs_update);
                        break;
                    }
                }

                // Check if English translation is missing or needs update
                $needs_translation = false;
                if (!$translation_exists) {
                    $needs_translation = true;
                } elseif ($needs_update) {
                    $needs_translation = true;
                } elseif ($translation_id) {
                    // Double-check: translation ID exists but post doesn't - treat as missing
                    $translated_post = get_post($translation_id);
                    if (!$translated_post) {
                        $needs_translation = true;
                    }
                }

                if ($needs_translation) {
                    $original_post = get_post($original_post_id);

                    $all_posts_needing_translation[] = array(
                        'id' => $original_post_id,
                        'title' => $original_post->post_title,
                        'status' => $original_post->post_status,
                        'modified' => $original_post->post_modified,
                        'source_lang' => $source_lang,
                        'missing_languages' => array('en'),
                    );
                }
            }

            // Only add post type if it has posts needing translation
            if (!empty($all_posts_needing_translation)) {
                $total_posts = count($all_posts_needing_translation);
                $total_pages = ceil($total_posts / $per_page);
                $offset = ($page - 1) * $per_page;

                // Paginate results
                $paginated_posts = array_slice($all_posts_needing_translation, $offset, $per_page);

                $posts_by_type[$pt->name] = array(
                    'label' => $pt->label,
                    'posts' => $paginated_posts,
                    'total_posts' => $total_posts,
                    'total_pages' => $total_pages,
                    'current_page' => $page,
                    'per_page' => $per_page,
                );
            }
        }

        return $posts_by_type;
    }

    /**
     * Get all post types that have posts needing translation (for tabs)
     *
     * @param array $target_languages Target languages
     * @return array Post types with counts
     */
    private function get_post_types_needing_translation($target_languages) {
        global $sitepress;
        $wpml = LATWAITR()->wpml;
        $post_type_counts = array();

        // Get translatable post types
        $post_types = $this->get_translatable_post_types();

        foreach ($post_types as $pt) {
            $args = array(
                'post_type' => $pt->name,
                'post_status' => array('publish', 'draft'),
                'posts_per_page' => -1,
                'orderby' => 'modified',
                'order' => 'DESC',
                'suppress_filters' => true, // Get posts from all languages, not just current admin language
            );

            $posts = get_posts($args);
            $processed_trids = array();
            $count = 0;

            foreach ($posts as $post) {
                $trid = $sitepress->get_element_trid($post->ID, 'post_' . $pt->name);

                if (in_array($trid, $processed_trids)) {
                    continue;
                }

                // Get all translations first so we can identify the original language
                $translations = $sitepress->get_element_translations($trid, 'post_' . $pt->name);

                // Find the original post in this translation group
                $original_post_id = $post->ID;
                $source_lang = $wpml->get_post_language($post->ID);

                foreach ($translations as $lang_code => $translation) {
                    if (!empty($translation->original)) {
                        $original_post_id = $translation->element_id;
                        $source_lang = $lang_code;
                        break;
                    }
                }

                // Mark group as processed now that we know the original language
                $processed_trids[] = $trid;

                // Skip if original is English (nothing to translate to English)
                if (strtolower(substr($source_lang, 0, 2)) === 'en') {
                    continue;
                }

                // Check if English translation exists and if WPML flags it as needing update
                $translation_exists = false;
                $translation_id = null;
                $needs_update = false;

                foreach ($translations as $lang_code => $translation) {
                    if (strtolower(substr($lang_code, 0, 2)) === 'en' && !empty($translation->element_id)) {
                        $translation_exists = true;
                        $translation_id = $translation->element_id;
                        // Use WPML's needs_update flag instead of comparing post_modified dates
                        $needs_update = !empty($translation->needs_update);
                        break;
                    }
                }

                $needs_translation = false;
                if (!$translation_exists) {
                    $needs_translation = true;
                } elseif ($needs_update) {
                    $needs_translation = true;
                } elseif ($translation_id) {
                    // Double-check: translation ID exists but post doesn't - treat as missing
                    $translated_post = get_post($translation_id);
                    if (!$translated_post) {
                        $needs_translation = true;
                    }
                }

                if ($needs_translation) {
                    $count++;
                }
            }

            if ($count > 0) {
                $post_type_counts[$pt->name] = array(
                    'name' => $pt->name,
                    'label' => $pt->label,
                    'count' => $count,
                );
            }
        }

        return $post_type_counts;
    }

    /**
     * Get taxonomy tab data (count of terms needing translation)
     *
     * @param array $target_languages Target languages
     * @return array|null Tab data or null if no terms need translation
     */
    private function get_taxonomy_tab_data($target_languages) {
        if (empty($target_languages)) {
            return null;
        }

        global $sitepress;
        if (!$sitepress) {
            return null;
        }

        $total_count = 0;
        $taxonomies = get_taxonomies(array('public' => true), 'objects');

        foreach ($taxonomies as $taxonomy) {
            $terms = get_terms(array(
                'taxonomy' => $taxonomy->name,
                'hide_empty' => false,
            ));

            if (is_wp_error($terms) || empty($terms)) {
                continue;
            }

            $processed_trids = array();

            foreach ($terms as $term) {
                $trid = apply_filters('wpml_element_trid', null, $term->term_id, 'tax_' . $taxonomy->name);

                if (in_array($trid, $processed_trids)) {
                    continue;
                }
                $processed_trids[] = $trid;

                // Check if this is the original term
                $element_lang = apply_filters('wpml_element_language_code', null, array(
                    'element_id' => $term->term_id,
                    'element_type' => 'tax_' . $taxonomy->name,
                ));

                if (empty($element_lang)) {
                    continue;
                }

                $translations = $sitepress->get_element_translations($trid, 'tax_' . $taxonomy->name);

                // FREE version: only check English translation
                $has_missing = false;
                if ($element_lang !== 'en') {
                    $translation_exists = false;
                    foreach ($translations as $lang_code => $translation) {
                        if ($lang_code === 'en' && !empty($translation->element_id)) {
                            $translation_exists = true;
                            break;
                        }
                    }
                    if (!$translation_exists) {
                        $has_missing = true;
                    }
                }

                if ($has_missing) {
                    $total_count++;
                }
            }
        }

        if ($total_count === 0) {
            return null;
        }

        return array(
            'name' => 'categories',
            'label' => 'Categories',
            'count' => $total_count,
        );
    }

    /**
     * Get terms needing translation with pagination and search
     *
     * @param array  $target_languages Target languages
     * @param int    $page             Page number
     * @param int    $per_page         Items per page
     * @param string $search           Search query
     * @return array Terms data with pagination info
     */
    private function get_terms_needing_translation($target_languages, $page = 1, $per_page = 20, $search = '') {
        if (empty($target_languages)) {
            return array('terms' => array(), 'total_terms' => 0, 'total_pages' => 0);
        }

        global $sitepress;
        if (!$sitepress) {
            return array('terms' => array(), 'total_terms' => 0, 'total_pages' => 0);
        }

        $all_terms = array();
        $taxonomies = get_taxonomies(array('public' => true), 'objects');

        foreach ($taxonomies as $taxonomy) {
            $term_args = array(
                'taxonomy' => $taxonomy->name,
                'hide_empty' => false,
            );

            if (!empty($search)) {
                $term_args['search'] = $search;
            }

            $terms = get_terms($term_args);

            if (is_wp_error($terms) || empty($terms)) {
                continue;
            }

            $processed_trids = array();

            foreach ($terms as $term) {
                $trid = apply_filters('wpml_element_trid', null, $term->term_id, 'tax_' . $taxonomy->name);

                if (in_array($trid, $processed_trids)) {
                    continue;
                }
                $processed_trids[] = $trid;

                $element_lang = apply_filters('wpml_element_language_code', null, array(
                    'element_id' => $term->term_id,
                    'element_type' => 'tax_' . $taxonomy->name,
                ));

                if (empty($element_lang)) {
                    continue;
                }

                $translations = $sitepress->get_element_translations($trid, 'tax_' . $taxonomy->name);

                // Find the original term in this translation group
                $original_term_id = $term->term_id;
                $source_lang = $element_lang;
                foreach ($translations as $lang_code => $translation) {
                    if (!empty($translation->original) && $translation->original == 1) {
                        $original_term_id = $translation->element_id;
                        $source_lang = $lang_code;
                        break;
                    }
                }

                // FREE version: only check English translation
                $missing_languages = array();
                if ($source_lang !== 'en') {
                    $translation_exists = false;
                    foreach ($translations as $lang_code => $translation) {
                        if ($lang_code === 'en' && !empty($translation->element_id)) {
                            $translation_exists = true;
                            break;
                        }
                    }
                    if (!$translation_exists) {
                        $missing_languages[] = 'en';
                    }
                }

                if (!empty($missing_languages)) {
                    $original_term = get_term($original_term_id, $taxonomy->name);
                    if (!$original_term || is_wp_error($original_term)) {
                        continue;
                    }

                    $all_terms[] = array(
                        'term_id' => $original_term_id,
                        'name' => $original_term->name,
                        'taxonomy' => $taxonomy->name,
                        'taxonomy_label' => $taxonomy->label,
                        'slug' => $original_term->slug,
                        'description' => $original_term->description,
                        'parent' => $original_term->parent,
                        'count' => $original_term->count,
                        'source_lang' => $source_lang,
                        'missing_languages' => $missing_languages,
                    );
                }
            }
        }

        $total_terms = count($all_terms);
        $total_pages = ceil($total_terms / $per_page);
        $offset = ($page - 1) * $per_page;
        $paginated_terms = array_slice($all_terms, $offset, $per_page);

        return array(
            'terms' => $paginated_terms,
            'total_terms' => $total_terms,
            'total_pages' => $total_pages,
        );
    }

    /**
     * Render settings page
     */
    public function render_settings() {
        // Use existing settings instance
        LATWAITR()->settings->render();
    }

    /**
     * Render logs page
     */
    public function render_logs() {
        $latwaitr_logger = LATWAITR()->logger;
        $latwaitr_logs = $latwaitr_logger->get_recent_logs(200);

        include LATWAITR_PLUGIN_DIR . 'admin/views/logs.php';
    }

    /**
     * Get posts with existing translations (for retranslation)
     * Returns posts with existing English translations
     *
     * @param string $post_type The post type to filter by
     * @return array Posts with their existing translations
     */
    public function get_posts_with_translations($post_type) {
        if (empty($post_type)) {
            return array();
        }

        global $sitepress;

        // Check if WPML is available
        if (!$sitepress || !is_object($sitepress)) {
            return array();
        }

        $wpml = LATWAITR()->wpml;
        $posts_with_translations = array();

        $args = array(
            'post_type' => $post_type,
            'post_status' => array('publish', 'draft'),
            'posts_per_page' => -1,
            'orderby' => 'modified',
            'order' => 'DESC',
        );

        $posts = get_posts($args);
        $processed_trids = array(); // Track processed translation groups

        foreach ($posts as $post) {
            // Get translation group ID (trid)
            $trid = $sitepress->get_element_trid($post->ID, 'post_' . $post_type);

            // Skip if we already processed this translation group
            if (in_array($trid, $processed_trids)) {
                continue;
            }

            $processed_trids[] = $trid;

            // Get the original/source post in this translation group
            $source_lang = $wpml->get_post_language($post->ID);

            // Get all translations in this group
            $translations = $sitepress->get_element_translations($trid, 'post_' . $post_type);

            // Find the original post
            $original_post_id = $post->ID;
            foreach ($translations as $lang_code => $translation) {
                if ($translation->original == 1) {
                    $original_post_id = $translation->element_id;
                    $source_lang = $lang_code;
                }
            }

            // Skip if source language is English (can't translate from English to English)
            if (strtolower(substr($source_lang, 0, 2)) === 'en') {
                continue;
            }

            // Check if English translation exists
            $has_english_translation = false;
            foreach ($translations as $lang_code => $translation) {
                if (strtolower(substr($lang_code, 0, 2)) === 'en' && $translation->element_id) {
                    $has_english_translation = true;
                    break;
                }
            }

            // Only include posts that have English translation
            if ($has_english_translation) {
                $original_post = get_post($original_post_id);

                $posts_with_translations[] = array(
                    'id' => $original_post_id,
                    'title' => $original_post->post_title,
                    'status' => $original_post->post_status,
                    'modified' => $original_post->post_modified,
                    'source_lang' => $source_lang,
                    'existing_translations' => array('en'),
                );
            }
        }

        return $posts_with_translations;
    }

    /**
     * Get terms that have English translations (for Translate Again feature)
     */
    public function get_terms_with_translations() {
        global $sitepress;

        if (!$sitepress) {
            return array();
        }

        $terms_with_translations = array();
        $taxonomies = get_taxonomies(array('public' => true), 'objects');
        $processed_trids = array();

        foreach ($taxonomies as $taxonomy) {
            $terms = get_terms(array(
                'taxonomy' => $taxonomy->name,
                'hide_empty' => false,
            ));

            if (is_wp_error($terms) || empty($terms)) {
                continue;
            }

            foreach ($terms as $term) {
                $trid = apply_filters('wpml_element_trid', null, $term->term_id, 'tax_' . $taxonomy->name);

                if (in_array($trid, $processed_trids)) {
                    continue;
                }

                $translations = $sitepress->get_element_translations($trid, 'tax_' . $taxonomy->name);

                // Find original term
                $original_term_id = $term->term_id;
                $source_lang = apply_filters('wpml_element_language_code', null, array(
                    'element_id' => $term->term_id,
                    'element_type' => 'tax_' . $taxonomy->name,
                ));

                foreach ($translations as $lang_code => $translation) {
                    if (!empty($translation->original)) {
                        $original_term_id = $translation->element_id;
                        $source_lang = $lang_code;
                        break;
                    }
                }

                $processed_trids[] = $trid;

                // Skip if original is English
                if (empty($source_lang) || strtolower(substr($source_lang, 0, 2)) === 'en') {
                    continue;
                }

                // Check if English translation exists
                $has_english_translation = false;
                foreach ($translations as $lang_code => $translation) {
                    if (strtolower(substr($lang_code, 0, 2)) === 'en' && !empty($translation->element_id)) {
                        $has_english_translation = true;
                        break;
                    }
                }

                if ($has_english_translation) {
                    $original_term = get_term($original_term_id, $taxonomy->name);
                    if ($original_term && !is_wp_error($original_term)) {
                        $terms_with_translations[] = array(
                            'term_id'              => $original_term_id,
                            'name'                 => $original_term->name,
                            'taxonomy'             => $taxonomy->name,
                            'taxonomy_label'       => $taxonomy->label,
                            'source_lang'          => $source_lang,
                            'existing_translations' => array('en'),
                        );
                    }
                }
            }
        }

        return $terms_with_translations;
    }
}
