<?php
/**
 * Abstract class for Elementor modules with repeatable items
 *
 * Handles widgets like tabs, accordion, icon-list that have repeating items
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

abstract class LATWAITR_Elementor_Module_With_Items {

    /**
     * Field separator for nested fields
     */
    const FIELD_SEPARATOR = '>';

    /**
     * Get the title/description for a field
     *
     * @param string $field Field name
     * @return string
     */
    abstract protected function get_title($field);

    /**
     * Get array of fields to translate
     *
     * @return array
     */
    abstract protected function get_fields();

    /**
     * Get editor type for a field
     *
     * @param string $field Field name
     * @return string Editor type (LINE, VISUAL, AREA, LINK)
     */
    abstract protected function get_editor_type($field);

    /**
     * Get the name of the items field in settings
     *
     * @return string
     */
    abstract public function get_items_field();

    /**
     * Extract translatable strings from element
     *
     * @param string|int         $node_id Node identifier
     * @param array              $element Element data
     * @param LATWAITR_PB_String[]   $strings Existing strings array
     * @return LATWAITR_PB_String[]
     */
    public function get($node_id, $element, $strings) {
        $items = $this->get_items($element);

        if (!is_array($items)) {
            return $strings;
        }

        foreach ($items as $item) {
            foreach ($this->get_fields() as $key => $field) {
                if (!is_array($field)) {
                    // Simple field
                    $path_parts = explode(self::FIELD_SEPARATOR, $field);
                    $string_value = $this->get_value_from_path($item, $path_parts);

                    if ($string_value && is_string($string_value)) {
                        $strings[] = new LATWAITR_PB_String(
                            $string_value,
                            $this->get_string_name($node_id, $item, $element, $field),
                            $this->get_title($field),
                            $this->get_editor_type($field)
                        );
                    }
                } else {
                    // Array field (nested structure)
                    foreach ($field as $inner_field) {
                        if (isset($item[$key][$inner_field])) {
                            $strings[] = new LATWAITR_PB_String(
                                $item[$key][$inner_field],
                                $this->get_string_name($node_id, $item, $element, $inner_field, $key),
                                $this->get_title($inner_field),
                                $this->get_editor_type($inner_field)
                            );
                        }
                    }
                }
            }
        }

        return $strings;
    }

    /**
     * Update element with translated string
     *
     * @param int|string       $node_id Node identifier
     * @param array            $element Element data
     * @param LATWAITR_PB_String   $pb_string String to update
     * @return array [key, item] or [null, null] if not found
     */
    public function update($node_id, $element, $pb_string) {
        $items = $this->get_items($element);

        if (!is_array($items)) {
            return array(null, null);
        }

        foreach ($items as $key => $item) {
            foreach ($this->get_fields() as $field_key => $field) {
                if (!is_array($field)) {
                    // Simple field
                    if ($this->get_string_name($node_id, $item, $element, $field) === $pb_string->get_name()) {
                        $path_parts = explode(self::FIELD_SEPARATOR, $field);
                        $string_value = $this->get_value_from_path($item, $path_parts);

                        if (is_string($string_value)) {
                            $item = $this->set_value_to_path($item, $path_parts, $pb_string->get_value());
                            return array($key, $item);
                        }
                    }
                } else {
                    // Array field
                    foreach ($field as $inner_field) {
                        if (
                            isset($item[$field_key][$inner_field]) &&
                            $this->get_string_name($node_id, $item, $element, $inner_field, $field_key) === $pb_string->get_name()
                        ) {
                            $item[$field_key][$inner_field] = $pb_string->get_value();
                            return array($key, $item);
                        }
                    }
                }
            }
        }

        return array(null, null);
    }

    /**
     * Get items array from element
     *
     * @param array $element Element data
     * @return array|null
     */
    protected function get_items($element) {
        if (!isset($element['settings'][$this->get_items_field()])) {
            return null;
        }

        return $element['settings'][$this->get_items_field()];
    }

    /**
     * Generate unique string name
     *
     * @param string $node_id Node identifier
     * @param array  $item Item data
     * @param array  $element Element data
     * @param string $field Field name
     * @param string $key Optional key for nested fields
     * @return string
     */
    protected function get_string_name($node_id, $item, $element, $field = '', $key = '') {
        $widget_type = isset($element['widgetType']) ? $element['widgetType'] : $element['elType'];
        $item_id = isset($item['_id']) ? $item['_id'] : '';

        return $widget_type . '-' . $field . '-' . $node_id . '-' . $item_id;
    }

    /**
     * Get value from nested path
     *
     * @param array $data Data array
     * @param array $path Path parts
     * @return mixed|null
     */
    protected function get_value_from_path($data, $path) {
        $current = $data;

        foreach ($path as $key) {
            if (!isset($current[$key])) {
                return null;
            }
            $current = $current[$key];
        }

        return $current;
    }

    /**
     * Set value to nested path
     *
     * @param array  $data Data array
     * @param array  $path Path parts
     * @param mixed  $value Value to set
     * @return array
     */
    protected function set_value_to_path($data, $path, $value) {
        if (empty($path)) {
            return $value;
        }

        $key = array_shift($path);

        if (empty($path)) {
            $data[$key] = $value;
        } else {
            if (!isset($data[$key]) || !is_array($data[$key])) {
                $data[$key] = array();
            }
            $data[$key] = $this->set_value_to_path($data[$key], $path, $value);
        }

        return $data;
    }
}
