<?php
/**
 * Elementor Translatable Nodes Manager
 *
 * Manages translatable strings extraction and update for Elementor widgets
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_Elementor_Translatable_Nodes {

    const SETTINGS_FIELD = 'settings';
    const TYPE = 'widgetType';
    const DEFAULT_HEADING_TAG = 'h2';
    const ELEMENT_TYPE = 'elType';

    /**
     * @var array Cached nodes configuration
     */
    private $nodes_to_translate;

    /**
     * Get translatable strings from an element
     *
     * @param string|int $node_id Element ID
     * @param array      $element Element data
     * @return LATWAITR_PB_String[]
     */
    public function get($node_id, $element) {
        if (!$this->nodes_to_translate) {
            $this->initialize_nodes_to_translate();
        }

        $strings = array();
        $found_config = false;

        foreach ($this->nodes_to_translate as $node_type => $node_data) {
            if ($this->conditions_ok($node_data, $element)) {
                $found_config = true;

                $widget_type = isset($element['widgetType']) ? $element['widgetType'] : 'unknown';
                LATWAITR()->logger->debug('Found widget configuration', array(
                    'widget_type' => $widget_type,
                    'node_type' => $node_type,
                    'has_integration_class' => isset($node_data['integration-class']),
                    'fields_count' => count($node_data['fields'])
                ));

                // Process simple fields
                foreach ($node_data['fields'] as $key => $field) {
                    $field_key = $field['field'];
                    $path_in_flat_field = array_merge(array(self::SETTINGS_FIELD), $this->get_partial_path($field_key));
                    $string_value = $this->get_value_from_path($element, $path_in_flat_field);

                    if (!is_string($string_value)) {
                        $path_in_array_field = array_merge(array(self::SETTINGS_FIELD, $key), $this->get_partial_path($field_key));
                        $string_value = $this->get_value_from_path($element, $path_in_array_field);
                    }

                    if ($string_value && is_string($string_value)) {
                        $strings[] = new LATWAITR_PB_String(
                            $string_value,
                            $this->get_string_name($node_id, $field, $element),
                            $field['type'],
                            $field['editor_type'],
                            $this->get_wrap_tag($element)
                        );
                    }
                }

                // Process integration classes (widgets with items)
                if (isset($node_data['integration-class'])) {
                    $integration_classes = is_array($node_data['integration-class'])
                        ? $node_data['integration-class']
                        : array($node_data['integration-class']);

                    foreach ($integration_classes as $class_name) {
                        LATWAITR()->logger->debug('Calling integration class', array(
                            'widget_type' => $widget_type,
                            'class' => $class_name,
                            'class_exists' => class_exists($class_name)
                        ));

                        if (class_exists($class_name)) {
                            try {
                                $instance = new $class_name();
                                $before_count = count($strings);
                                $strings = $instance->get($node_id, $element, $strings);
                                $after_count = count($strings);

                                LATWAITR()->logger->debug('Integration class extracted strings', array(
                                    'widget_type' => $widget_type,
                                    'class' => $class_name,
                                    'strings_added' => $after_count - $before_count
                                ));
                            } catch (Exception $e) {
                                LATWAITR()->logger->error('Integration class error', array(
                                    'widget_type' => $widget_type,
                                    'class' => $class_name,
                                    'error' => $e->getMessage()
                                ));
                            }
                        }
                    }
                }
            }
        }

        // If no configuration found, try automatic detection
        if (!$found_config) {
            $widget_type = isset($element['widgetType']) ? $element['widgetType'] : 'unknown';
            LATWAITR()->logger->debug('No config found for widget, using auto-detection', array(
                'widget_type' => $widget_type,
                'node_id' => $node_id
            ));
            $strings = $this->auto_detect_strings($node_id, $element, $strings);
        }

        return $strings;
    }

    /**
     * Update element with translated string
     *
     * @param int|string       $node_id Element ID
     * @param array            $element Element data
     * @param LATWAITR_PB_String   $pb_string String to update
     * @return array Updated element
     */
    public function update($node_id, $element, $pb_string) {
        if (!$this->nodes_to_translate) {
            $this->initialize_nodes_to_translate();
        }

        $found_config = false;

        foreach ($this->nodes_to_translate as $node_type => $node_data) {
            if ($this->conditions_ok($node_data, $element)) {
                $found_config = true;

                // Update simple fields
                foreach ($node_data['fields'] as $key => $field) {
                    $field_key = $field['field'];

                    if ($this->get_string_name($node_id, $field, $element) === $pb_string->get_name()) {
                        $path_in_flat_field = array_merge(array(self::SETTINGS_FIELD), $this->get_partial_path($field_key));
                        $string_in_flat_field = $this->get_value_from_path($element, $path_in_flat_field);
                        $path_in_array_field = array_merge(array(self::SETTINGS_FIELD, $key), $this->get_partial_path($field_key));
                        $string_in_array_field = $this->get_value_from_path($element, $path_in_array_field);

                        if (is_string($string_in_flat_field)) {
                            $element = $this->set_value_to_path($element, $path_in_flat_field, $pb_string->get_value());
                        } elseif (is_string($string_in_array_field)) {
                            $element = $this->set_value_to_path($element, $path_in_array_field, $pb_string->get_value());
                        }
                    }
                }

                // Update integration classes
                if (isset($node_data['integration-class'])) {
                    $integration_classes = is_array($node_data['integration-class'])
                        ? $node_data['integration-class']
                        : array($node_data['integration-class']);

                    foreach ($integration_classes as $class_name) {
                        if (class_exists($class_name)) {
                            try {
                                $instance = new $class_name();
                                list($key, $item) = $instance->update($node_id, $element, $pb_string);

                                if ($item) {
                                    // Check if key contains path separator (for complex widgets like Contact)
                                    if (strpos($key, '>') !== false) {
                                        // Key is a path like "group_1_repeater>0"
                                        $path_parts = $this->get_partial_path($key);
                                        $path = array_merge(array(self::SETTINGS_FIELD), $path_parts);
                                    } else {
                                        // Standard repeater: items_field + index
                                        $path = array(self::SETTINGS_FIELD, $instance->get_items_field(), $key);
                                    }
                                    $element = $this->set_value_to_path($element, $path, $item);
                                }
                            } catch (Exception $e) {
                                // Continue on error
                            }
                        }
                    }
                }
            }
        }

        // If no configuration found, try auto-update
        if (!$found_config) {
            $result = $this->auto_update_string($node_id, $element, $pb_string);
            if ($result !== false) {
                $element = $result;
            }
        }

        return $element;
    }

    /**
     * Automatically update string in unknown widget
     *
     * @param string|int       $node_id Element ID
     * @param array            $element Element data
     * @param LATWAITR_PB_String   $pb_string String to update
     * @return array|false Updated element, or false if no update was made
     */
    private function auto_update_string($node_id, $element, $pb_string) {
        if (!isset($element[self::SETTINGS_FIELD]) || !is_array($element[self::SETTINGS_FIELD])) {
            return false;
        }

        $widget_type = isset($element[self::TYPE]) ? $element[self::TYPE] : 'unknown';
        $string_name = $pb_string->get_name();

        // Parse string name to get field name
        // Format: field_name-widget_type-node_id
        // Use known widget_type and node_id to strip the suffix (handles hyphenated widget types)
        $suffix = '-' . $widget_type . '-' . $node_id;
        if (substr($string_name, -strlen($suffix)) === $suffix) {
            $field_name = substr($string_name, 0, -strlen($suffix));
        } else {
            return false;
        }

        // Update the field if it exists
        if (isset($element[self::SETTINGS_FIELD][$field_name])) {
            $element[self::SETTINGS_FIELD][$field_name] = $pb_string->get_value();
            return $element;
        }

        return false;
    }

    /**
     * Get string name/identifier
     *
     * @param string|int $node_id Element ID
     * @param array      $field Field configuration
     * @param array      $settings Element data
     * @return string
     */
    public function get_string_name($node_id, $field, $settings) {
        $field_id = isset($field['field_id']) ? $field['field_id'] : $field['field'];
        $type = isset($settings[self::TYPE]) ? $settings[self::TYPE] : $settings[self::ELEMENT_TYPE];

        return $field_id . '-' . $type . '-' . $node_id;
    }

    /**
     * Get wrap tag for SEO
     *
     * @param array $settings Element settings
     * @return string
     */
    private function get_wrap_tag($settings) {
        if (isset($settings[self::TYPE]) && 'heading' === $settings[self::TYPE]) {
            $header_size = isset($settings[self::SETTINGS_FIELD]['header_size'])
                ? $settings[self::SETTINGS_FIELD]['header_size']
                : self::DEFAULT_HEADING_TAG;

            return $header_size;
        }

        return '';
    }

    /**
     * Check if conditions match
     *
     * @param array $node_data Node configuration
     * @param array $element Element data
     * @return bool
     */
    private function conditions_ok($node_data, $element) {
        foreach ($node_data['conditions'] as $field_key => $field_value) {
            if (!isset($element[$field_key]) || $element[$field_key] !== $field_value) {
                return false;
            }
        }

        return true;
    }

    /**
     * Split field path by separator
     *
     * @param string $field Field path
     * @return array
     */
    private function get_partial_path($field) {
        return explode('>', $field);
    }

    /**
     * Get value from nested path
     *
     * @param array $data Data array
     * @param array $path Path parts
     * @return mixed|null
     */
    private function get_value_from_path($data, $path) {
        $current = $data;

        foreach ($path as $key) {
            if (!isset($current[$key])) {
                return null;
            }
            $current = $current[$key];
        }

        return $current;
    }

    /**
     * Set value to nested path
     *
     * @param array  $data Data array
     * @param array  $path Path parts
     * @param mixed  $value Value to set
     * @return array
     */
    private function set_value_to_path($data, $path, $value) {
        if (empty($path)) {
            return $value;
        }

        $key = array_shift($path);

        if (empty($path)) {
            $data[$key] = $value;
        } else {
            if (!isset($data[$key]) || !is_array($data[$key])) {
                $data[$key] = array();
            }
            $data[$key] = $this->set_value_to_path($data[$key], $path, $value);
        }

        return $data;
    }

    /**
     * Automatically detect translatable strings in unknown widgets
     *
     * @param string|int         $node_id Element ID
     * @param array              $element Element data
     * @param LATWAITR_PB_String[]   $strings Existing strings
     * @return LATWAITR_PB_String[]
     */
    private function auto_detect_strings($node_id, $element, $strings) {
        if (!isset($element[self::SETTINGS_FIELD]) || !is_array($element[self::SETTINGS_FIELD])) {
            return $strings;
        }

        $widget_type = isset($element[self::TYPE]) ? $element[self::TYPE] : 'unknown';

        // Common text field names in Elementor widgets
        $text_field_patterns = array(
            // Direct text fields
            'title', 'text', 'content', 'description', 'heading',
            'label', 'placeholder', 'caption', 'subtitle',
            'button_text', 'link_text', 'item_text',
            'before_text', 'after_text', 'prefix', 'suffix',
            'html', 'editor',
            // Specific widget fields
            'alert_title', 'alert_description',
            'testimonial_content', 'testimonial_name', 'testimonial_job',
            'blockquote_content',
            'inner_text',
        );

        // Patterns that indicate non-translatable CSS/layout/configuration settings
        $non_translatable_patterns = array(
            'typography',   // font family, weight, size, line height, etc.
            'shadow',       // text shadow, box shadow settings
            '_font_',       // font-related settings
        );

        $non_translatable_suffixes = array(
            '_position',    // content_position, background_position, etc.
            '_align',       // text alignment, flex alignment
            '_align_flex',  // flex alignment
            '_style',       // widget style identifiers (style1, style2, etc.)
            '_font_family', // font family names
            '_font_weight', // font weight values
            '_font_size',   // font size values
            '_decoration',  // text decoration (none, underline, etc.)
            '_transform',   // text transform (uppercase, lowercase, etc.)
        );

        // Scan settings for text fields
        foreach ($element[self::SETTINGS_FIELD] as $field_name => $value) {
            // Skip non-text fields
            if (!is_string($value) || empty(trim($value))) {
                continue;
            }

            // Skip very short values (likely not translatable text)
            if (strlen(trim($value)) < 3) {
                continue;
            }

            // Check if field name matches common patterns
            $is_text_field = false;
            foreach ($text_field_patterns as $pattern) {
                if (strpos($field_name, $pattern) !== false) {
                    $is_text_field = true;
                    break;
                }
            }

            if (!$is_text_field) {
                continue;
            }

            // Exclude non-translatable CSS/layout/configuration fields
            $is_non_translatable = false;
            foreach ($non_translatable_patterns as $nt_pattern) {
                if (strpos($field_name, $nt_pattern) !== false) {
                    $is_non_translatable = true;
                    break;
                }
            }
            if (!$is_non_translatable) {
                foreach ($non_translatable_suffixes as $nt_suffix) {
                    if (substr($field_name, -strlen($nt_suffix)) === $nt_suffix) {
                        $is_non_translatable = true;
                        break;
                    }
                }
            }

            if ($is_non_translatable) {
                continue;
            }

            // Determine editor type based on content
            $editor_type = $this->guess_editor_type($value);

            // Create string object
            $strings[] = new LATWAITR_PB_String(
                $value,
                $field_name . '-' . $widget_type . '-' . $node_id,
                ucwords(str_replace('_', ' ', $widget_type)) . ': ' . ucwords(str_replace('_', ' ', $field_name)),
                $editor_type,
                ''
            );
        }

        return $strings;
    }

    /**
     * Guess editor type from content
     *
     * @param string $value Field value
     * @return string Editor type
     */
    private function guess_editor_type($value) {
        // Check for HTML tags (VISUAL editor)
        if (preg_match('/<[a-z][\s\S]*>/i', $value)) {
            return 'VISUAL';
        }

        // Check for URLs
        if (filter_var($value, FILTER_VALIDATE_URL)) {
            return 'LINK';
        }

        // Check for multiline text (AREA)
        if (substr_count($value, "\n") > 1 || strlen($value) > 200) {
            return 'AREA';
        }

        // Default to LINE
        return 'LINE';
    }

    /**
     * Initialize nodes to translate
     */
    public function initialize_nodes_to_translate() {
        $nodes = $this->get_nodes_to_translate();

        LATWAITR()->logger->debug('Initializing widget configurations', array(
            'total_widgets' => count($nodes),
            'widget_types' => array_keys($nodes),
            'form_has_integration_class' => isset($nodes['form']['integration-class']),
            'price_list_has_integration_class' => isset($nodes['price-list']['integration-class']),
            'contact_has_integration_class' => isset($nodes['contact']['integration-class'])
        ));

        /**
         * Filter Elementor widgets configuration
         *
         * Allows developers to add their own custom widgets or modify existing ones
         *
         * @param array $nodes Array of widget configurations
         *
         * Example:
         * add_filter('latwaitr_elementor_widgets_to_translate', function($nodes) {
         *     $nodes['my-custom-widget'] = array(
         *         'conditions' => array('widgetType' => 'my-custom-widget'),
         *         'fields' => array(
         *             array(
         *                 'field' => 'custom_title',
         *                 'type' => 'My Widget: Title',
         *                 'editor_type' => 'LINE',
         *             ),
         *         ),
         *     );
         *     return $nodes;
         * });
         */
        $this->nodes_to_translate = apply_filters('latwaitr_elementor_widgets_to_translate', $nodes);

        LATWAITR()->logger->debug('After latw filter', array(
            'form_has_integration_class' => isset($this->nodes_to_translate['form']['integration-class']),
            'price_list_has_integration_class' => isset($this->nodes_to_translate['price-list']['integration-class']),
            'contact_has_integration_class' => isset($this->nodes_to_translate['contact']['integration-class'])
        ));

        // Also check for WPML configuration (for compatibility)
        if (has_filter('wpml_elementor_widgets_to_translate')) {
            /**
             * Support WPML filters for compatibility
             *
             * This allows using existing WPML widget configurations
             * from third-party plugins without modification
             */
            $wpml_nodes = apply_filters('wpml_elementor_widgets_to_translate', array());
            if (!empty($wpml_nodes) && is_array($wpml_nodes)) {
                // Use array_replace_recursive to preserve our integration-class values
                // WPML nodes are merged but our configs take priority
                foreach ($wpml_nodes as $widget_type => $widget_config) {
                    if (!isset($this->nodes_to_translate[$widget_type])) {
                        // Only add if we don't have this widget yet
                        $this->nodes_to_translate[$widget_type] = $widget_config;
                    }
                }
            }
        }

        LATWAITR()->logger->debug('Final nodes configuration', array(
            'total_widgets' => count($this->nodes_to_translate),
            'form_has_integration_class' => isset($this->nodes_to_translate['form']['integration-class']),
            'price_list_has_integration_class' => isset($this->nodes_to_translate['price-list']['integration-class']),
            'contact_has_integration_class' => isset($this->nodes_to_translate['contact']['integration-class'])
        ));
    }

    /**
     * Get all nodes configuration
     *
     * @return array
     */
    public function get_nodes_to_translate() {
        return array(
            // Container
            'container' => array(
                'conditions' => array(self::ELEMENT_TYPE => 'container'),
                'fields' => array(
                    'link' => array(
                        'field' => 'url',
                        'type' => 'Container: Link URL',
                        'editor_type' => 'LINK',
                    ),
                ),
            ),

            // Heading
            'heading' => array(
                'conditions' => array(self::TYPE => 'heading'),
                'fields' => array(
                    array(
                        'field' => 'title',
                        'type' => 'Heading: Title',
                        'editor_type' => 'LINE',
                    ),
                    'link' => array(
                        'field' => 'url',
                        'type' => 'Heading: Link URL',
                        'editor_type' => 'LINK',
                    ),
                ),
            ),

            // Text Editor
            'text-editor' => array(
                'conditions' => array(self::TYPE => 'text-editor'),
                'fields' => array(
                    array(
                        'field' => 'editor',
                        'type' => 'Text Editor: Text',
                        'editor_type' => 'VISUAL',
                    ),
                ),
            ),

            // Video
            'video' => array(
                'conditions' => array(self::TYPE => 'video'),
                'fields' => array(
                    array(
                        'field' => 'link',
                        'type' => 'Video: Link',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'vimeo_link',
                        'type' => 'Video: Vimeo link',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'youtube_url',
                        'type' => 'Video: Youtube URL',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'vimeo_url',
                        'type' => 'Video: Vimeo URL',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'dailymotion_url',
                        'type' => 'Video: DailyMotion URL',
                        'editor_type' => 'LINE',
                    ),
                    'hosted_url' => array(
                        'field' => 'url',
                        'type' => 'Video: Self hosted',
                        'editor_type' => 'LINE',
                    ),
                    'external_url' => array(
                        'field' => 'url',
                        'type' => 'Video: External hosted',
                        'editor_type' => 'LINE',
                    ),
                ),
            ),

            // Login
            'login' => array(
                'conditions' => array(self::TYPE => 'login'),
                'fields' => array(
                    array(
                        'field' => 'button_text',
                        'type' => 'Login: Button text',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'user_label',
                        'type' => 'Login: User label',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'user_placeholder',
                        'type' => 'Login: User placeholder',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'password_label',
                        'type' => 'Login: Password label',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'password_placeholder',
                        'type' => 'Login: Password placeholder',
                        'editor_type' => 'LINE',
                    ),
                ),
            ),

            // Button
            'button' => array(
                'conditions' => array(self::TYPE => 'button'),
                'fields' => array(
                    array(
                        'field' => 'text',
                        'type' => 'Button: Text',
                        'editor_type' => 'LINE',
                    ),
                    'link' => array(
                        'field' => 'url',
                        'type' => 'Button: Link URL',
                        'editor_type' => 'LINK',
                    ),
                ),
            ),

            // HTML
            'html' => array(
                'conditions' => array(self::TYPE => 'html'),
                'fields' => array(
                    array(
                        'field' => 'html',
                        'type' => 'HTML',
                        'editor_type' => 'AREA',
                    ),
                ),
            ),

            // Alert
            'alert' => array(
                'conditions' => array(self::TYPE => 'alert'),
                'fields' => array(
                    array(
                        'field' => 'alert_title',
                        'type' => 'Alert: Title',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'alert_description',
                        'type' => 'Alert: Description',
                        'editor_type' => 'VISUAL',
                    ),
                ),
            ),

            // Blockquote
            'blockquote' => array(
                'conditions' => array(self::TYPE => 'blockquote'),
                'fields' => array(
                    array(
                        'field' => 'blockquote_content',
                        'type' => 'Blockquote: Content',
                        'editor_type' => 'AREA',
                    ),
                    array(
                        'field' => 'tweet_button_label',
                        'type' => 'Blockquote: Tweet button label',
                        'editor_type' => 'LINE',
                    ),
                ),
            ),

            // Testimonial
            'testimonial' => array(
                'conditions' => array(self::TYPE => 'testimonial'),
                'fields' => array(
                    array(
                        'field' => 'testimonial_content',
                        'type' => 'Testimonial: Content',
                        'editor_type' => 'VISUAL',
                    ),
                    array(
                        'field' => 'testimonial_name',
                        'type' => 'Testimonial: Name',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'testimonial_job',
                        'type' => 'Testimonial: Job',
                        'editor_type' => 'LINE',
                    ),
                    'link' => array(
                        'field' => 'url',
                        'type' => 'Testimonial: Link URL',
                        'editor_type' => 'LINK',
                    ),
                ),
            ),

            // Progress
            'progress' => array(
                'conditions' => array(self::TYPE => 'progress'),
                'fields' => array(
                    array(
                        'field' => 'title',
                        'type' => 'Progress: Title',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'inner_text',
                        'type' => 'Progress: Inner text',
                        'editor_type' => 'LINE',
                    ),
                ),
            ),

            // Counter
            'counter' => array(
                'conditions' => array(self::TYPE => 'counter'),
                'fields' => array(
                    array(
                        'field' => 'starting_number',
                        'type' => 'Counter: Starting number',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'title',
                        'type' => 'Counter: Title',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'prefix',
                        'type' => 'Counter: Prefix',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'suffix',
                        'type' => 'Counter: Suffix',
                        'editor_type' => 'LINE',
                    ),
                ),
            ),

            // Icon Box
            'icon-box' => array(
                'conditions' => array(self::TYPE => 'icon-box'),
                'fields' => array(
                    array(
                        'field' => 'title_text',
                        'type' => 'Icon Box: Title text',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'description_text',
                        'type' => 'Icon Box: Description text',
                        'editor_type' => 'AREA',
                    ),
                    'link' => array(
                        'field' => 'url',
                        'type' => 'Icon Box: Link',
                        'editor_type' => 'LINK',
                    ),
                ),
            ),

            // Image Box
            'image-box' => array(
                'conditions' => array(self::TYPE => 'image-box'),
                'fields' => array(
                    array(
                        'field' => 'title_text',
                        'type' => 'Image Box: Title text',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'description_text',
                        'type' => 'Image Box: Description text',
                        'editor_type' => 'LINE',
                    ),
                    'link' => array(
                        'field' => 'url',
                        'type' => 'Image Box: Link',
                        'editor_type' => 'LINK',
                    ),
                ),
            ),

            // Animated Headline
            'animated-headline' => array(
                'conditions' => array(self::TYPE => 'animated-headline'),
                'fields' => array(
                    array(
                        'field' => 'before_text',
                        'type' => 'Animated Headline: Before text',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'highlighted_text',
                        'type' => 'Animated Headline: Highlighted text',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'rotating_text',
                        'type' => 'Animated Headline: Rotating text',
                        'editor_type' => 'AREA',
                    ),
                    array(
                        'field' => 'after_text',
                        'type' => 'Animated Headline: After text',
                        'editor_type' => 'LINE',
                    ),
                    'link' => array(
                        'field' => 'url',
                        'type' => 'Animated Headline: Link URL',
                        'editor_type' => 'LINK',
                    ),
                ),
            ),

            // Flip Box
            'flip-box' => array(
                'conditions' => array(self::TYPE => 'flip-box'),
                'fields' => array(
                    array(
                        'field' => 'title_text_a',
                        'type' => 'Flip Box: Title text side A',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'description_text_a',
                        'type' => 'Flip Box: Description text side A',
                        'editor_type' => 'AREA',
                    ),
                    array(
                        'field' => 'title_text_b',
                        'type' => 'Flip Box: Title text side B',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'description_text_b',
                        'type' => 'Flip Box: Description text side B',
                        'editor_type' => 'AREA',
                    ),
                    array(
                        'field' => 'button_text',
                        'type' => 'Flip Box: Button text',
                        'editor_type' => 'LINE',
                    ),
                    'link' => array(
                        'field' => 'url',
                        'type' => 'Flip Box: Button link',
                        'editor_type' => 'LINK',
                    ),
                ),
            ),

            // Call to Action
            'call-to-action' => array(
                'conditions' => array(self::TYPE => 'call-to-action'),
                'fields' => array(
                    array(
                        'field' => 'title',
                        'type' => 'Call to action: title',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'description',
                        'type' => 'Call to action: description',
                        'editor_type' => 'VISUAL',
                    ),
                    array(
                        'field' => 'button',
                        'type' => 'Call to action: button',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'ribbon_title',
                        'type' => 'Call to action: ribbon title',
                        'editor_type' => 'LINE',
                    ),
                    'link' => array(
                        'field' => 'url',
                        'type' => 'Call to action: link',
                        'editor_type' => 'LINK',
                    ),
                ),
            ),

            // Toggle
            'toggle' => array(
                'conditions' => array(self::TYPE => 'toggle'),
                'fields' => array(),
                'integration-class' => 'LATWAITR_Elementor_Toggle',
            ),

            // Accordion
            'accordion' => array(
                'conditions' => array(self::TYPE => 'accordion'),
                'fields' => array(),
                'integration-class' => 'LATWAITR_Elementor_Accordion',
            ),

            // Testimonial Carousel
            'testimonial-carousel' => array(
                'conditions' => array(self::TYPE => 'testimonial-carousel'),
                'fields' => array(),
                'integration-class' => 'LATWAITR_Elementor_Testimonial_Carousel',
            ),

            // Tabs
            'tabs' => array(
                'conditions' => array(self::TYPE => 'tabs'),
                'fields' => array(),
                'integration-class' => 'LATWAITR_Elementor_Tabs',
            ),

            // Icon List
            'icon-list' => array(
                'conditions' => array(self::TYPE => 'icon-list'),
                'fields' => array(),
                'integration-class' => 'LATWAITR_Elementor_Icon_List',
            ),

            // Slides
            'slides' => array(
                'conditions' => array(self::TYPE => 'slides'),
                'fields' => array(),
                'integration-class' => 'LATWAITR_Elementor_Slides',
            ),

            // Price Table
            'price-table' => array(
                'conditions' => array(self::TYPE => 'price-table'),
                'fields' => array(
                    array(
                        'field' => 'heading',
                        'type' => 'Price Table: Heading',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'sub_heading',
                        'type' => 'Price Table: Sub heading',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'period',
                        'type' => 'Price Table: Period',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'button_text',
                        'type' => 'Price Table: Button text',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'footer_additional_info',
                        'type' => 'Price Table: Footer additional info',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'ribbon_title',
                        'type' => 'Price Table: Ribbon title',
                        'editor_type' => 'LINE',
                    ),
                    'link' => array(
                        'field' => 'url',
                        'type' => 'Price Table: Button link',
                        'editor_type' => 'LINK',
                    ),
                ),
                'integration-class' => 'LATWAITR_Elementor_Price_Table',
            ),

            // Menu Anchor
            'menu-anchor' => array(
                'conditions' => array(self::TYPE => 'menu-anchor'),
                'fields' => array(
                    array(
                        'field' => 'anchor',
                        'type' => 'Menu Anchor',
                        'editor_type' => 'LINE',
                    ),
                ),
            ),

            // Archive Posts
            'archive-posts' => array(
                'conditions' => array(self::TYPE => 'archive-posts'),
                'fields' => array(
                    array(
                        'field' => 'archive_cards_meta_separator',
                        'type' => 'Archive: Cards Separator',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'archive_cards_read_more_text',
                        'type' => 'Archive: Cards Read More Text',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'nothing_found_message',
                        'type' => 'Archive: Nothing Found Message',
                        'editor_type' => 'AREA',
                    ),
                    array(
                        'field' => 'pagination_prev_label',
                        'type' => 'Archive: Previous Label',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'pagination_next_label',
                        'type' => 'Archive: Next Label',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'archive_classic_meta_separator',
                        'type' => 'Archive: Classic Separator',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'archive_classic_read_more_text',
                        'type' => 'Archive: Classic Read More Text',
                        'editor_type' => 'LINE',
                    ),
                ),
            ),

            // Search Form
            'search-form' => array(
                'conditions' => array(self::TYPE => 'search-form'),
                'fields' => array(
                    array(
                        'field' => 'placeholder',
                        'type' => 'Search: Placeholder',
                        'editor_type' => 'LINE',
                    ),
                ),
            ),

            // Post Navigation
            'post-navigation' => array(
                'conditions' => array(self::TYPE => 'post-navigation'),
                'fields' => array(
                    array(
                        'field' => 'prev_label',
                        'type' => 'Post Navigation: Previous Label',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'next_label',
                        'type' => 'Post Navigation: Next Label',
                        'editor_type' => 'LINE',
                    ),
                ),
            ),

            // Divider
            'divider' => array(
                'conditions' => array(self::TYPE => 'divider'),
                'fields' => array(
                    array(
                        'field' => 'text',
                        'type' => 'Divider: Text',
                        'editor_type' => 'LINE',
                    ),
                ),
            ),

            // Table of Contents
            'table-of-contents' => array(
                'conditions' => array(self::TYPE => 'table-of-contents'),
                'fields' => array(
                    array(
                        'field' => 'title',
                        'type' => 'Table of Contents: Title',
                        'editor_type' => 'LINE',
                    ),
                ),
            ),

            // Author Box
            'author-box' => array(
                'conditions' => array(self::TYPE => 'author-box'),
                'fields' => array(
                    array(
                        'field' => 'author_name',
                        'type' => 'Author: Name',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'author_bio',
                        'type' => 'Author: Bio',
                        'editor_type' => 'AREA',
                    ),
                    'author_website' => array(
                        'field' => 'url',
                        'field_id' => 'author_website',
                        'type' => 'Author: Link',
                        'editor_type' => 'LINK',
                    ),
                    array(
                        'field' => 'link_text',
                        'type' => 'Author: Archive Text',
                        'editor_type' => 'LINE',
                    ),
                    'posts_url' => array(
                        'field' => 'url',
                        'field_id' => 'posts_url',
                        'type' => 'Author: Archive URL',
                        'editor_type' => 'LINK',
                    ),
                ),
            ),

            // Reviews
            'reviews' => array(
                'conditions' => array(self::TYPE => 'reviews'),
                'fields' => array(),
                'integration-class' => 'LATWAITR_Elementor_Reviews',
            ),

            // Galleries (Multiple Gallery widget)
            'galleries' => array(
                'conditions' => array(self::TYPE => 'gallery'),
                'fields' => array(
                    array(
                        'field' => 'show_all_galleries_label',
                        'type' => 'Galleries: All Label',
                        'editor_type' => 'LINE',
                    ),
                    'url' => array(
                        'field' => 'url',
                        'type' => 'Galleries: Gallery custom link',
                        'editor_type' => 'LINK',
                    ),
                ),
                'integration-class' => 'LATWAITR_Elementor_Multiple_Gallery',
            ),

            // Mega Menu (Elementor Pro)
            'mega-menu' => array(
                'conditions' => array(self::TYPE => 'mega-menu'),
                'fields' => array(
                    array(
                        'field' => 'menu_name',
                        'type' => 'Mega Menu: Menu Name',
                        'editor_type' => 'LINE',
                    ),
                ),
                'integration-class' => 'LATWAITR_Elementor_Mega_Menu',
            ),

            // Sitemap (Elementor Pro)
            'sitemap' => array(
                'conditions' => array(self::TYPE => 'sitemap'),
                'fields' => array(),
                'integration-class' => 'LATWAITR_Elementor_Sitemap',
            ),

            // Form (Elementor Pro)
            'form' => array(
                'conditions' => array(self::TYPE => 'form'),
                'fields' => array(
                    array(
                        'field' => 'form_name',
                        'type' => 'Form: Name',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'button_text',
                        'type' => 'Form: Button Text',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'step_next_label',
                        'type' => 'Form: Next Step Label',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'step_previous_label',
                        'type' => 'Form: Previous Step Label',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'success_message',
                        'type' => 'Form: Success Message',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'error_message',
                        'type' => 'Form: Error Message',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'server_message',
                        'type' => 'Form: Server Error Message',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'invalid_message',
                        'type' => 'Form: Invalid Message',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'required_field_message',
                        'type' => 'Form: Required Field Message',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'email_subject',
                        'type' => 'Form: Email Subject',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'email_content',
                        'type' => 'Form: Email Content',
                        'editor_type' => 'AREA',
                    ),
                    array(
                        'field' => 'email_from_name',
                        'type' => 'Form: Email From Name',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'email_subject_2',
                        'type' => 'Form: Email Subject 2',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'email_content_2',
                        'type' => 'Form: Email Content 2',
                        'editor_type' => 'AREA',
                    ),
                    array(
                        'field' => 'email_from_name_2',
                        'type' => 'Form: Email From Name 2',
                        'editor_type' => 'LINE',
                    ),
                ),
                'integration-class' => 'LATWAITR_Elementor_Form',
            ),

            // Price List (Elementor Pro)
            'price-list' => array(
                'conditions' => array(self::TYPE => 'price-list'),
                'fields' => array(),
                'integration-class' => 'LATWAITR_Elementor_Price_List',
            ),

            // Contact (Elementor Pro)
            'contact' => array(
                'conditions' => array(self::TYPE => 'contact'),
                'fields' => array(
                    array(
                        'field' => 'heading_text',
                        'type' => 'Contact: Heading',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'description_text',
                        'type' => 'Contact: Description',
                        'editor_type' => 'AREA',
                    ),
                    array(
                        'field' => 'map_address',
                        'type' => 'Contact: Map Address',
                        'editor_type' => 'LINE',
                    ),
                    // Group 1 subheadings
                    array(
                        'field' => 'group_1_links_subheading',
                        'type' => 'Contact: Group 1 Links Subheading',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'group_1_text_subheading',
                        'type' => 'Contact: Group 1 Text Subheading',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'group_1_text_textarea',
                        'type' => 'Contact: Group 1 Text',
                        'editor_type' => 'AREA',
                    ),
                    array(
                        'field' => 'group_1_social_subheading',
                        'type' => 'Contact: Group 1 Social Subheading',
                        'editor_type' => 'LINE',
                    ),
                    // Group 2 subheadings
                    array(
                        'field' => 'group_2_links_subheading',
                        'type' => 'Contact: Group 2 Links Subheading',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'group_2_text_subheading',
                        'type' => 'Contact: Group 2 Text Subheading',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'group_2_text_textarea',
                        'type' => 'Contact: Group 2 Text',
                        'editor_type' => 'AREA',
                    ),
                    array(
                        'field' => 'group_2_social_subheading',
                        'type' => 'Contact: Group 2 Social Subheading',
                        'editor_type' => 'LINE',
                    ),
                    // Group 3 subheadings
                    array(
                        'field' => 'group_3_links_subheading',
                        'type' => 'Contact: Group 3 Links Subheading',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'group_3_text_subheading',
                        'type' => 'Contact: Group 3 Text Subheading',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'group_3_text_textarea',
                        'type' => 'Contact: Group 3 Text',
                        'editor_type' => 'AREA',
                    ),
                    array(
                        'field' => 'group_3_social_subheading',
                        'type' => 'Contact: Group 3 Social Subheading',
                        'editor_type' => 'LINE',
                    ),
                    // Group 4 subheadings
                    array(
                        'field' => 'group_4_links_subheading',
                        'type' => 'Contact: Group 4 Links Subheading',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'group_4_text_subheading',
                        'type' => 'Contact: Group 4 Text Subheading',
                        'editor_type' => 'LINE',
                    ),
                    array(
                        'field' => 'group_4_text_textarea',
                        'type' => 'Contact: Group 4 Text',
                        'editor_type' => 'AREA',
                    ),
                    array(
                        'field' => 'group_4_social_subheading',
                        'type' => 'Contact: Group 4 Social Subheading',
                        'editor_type' => 'LINE',
                    ),
                ),
                'integration-class' => 'LATWAITR_Elementor_Contact',
            ),
        );
    }
}
