<?php
/**
 * WPML Integration for LATW
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_WPML_Integration
{

    /**
     * Logger
     */
    private $logger;

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->logger = LATWAITR()->logger;

        // Check if WPML is active
        if (!$this->is_wpml_active()) {
            return;
        }

        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks()
    {
        // Hook into WPML
        add_action('wpml_loaded', array($this, 'wpml_loaded'));
    }

    /**
     * WPML loaded hook
     */
    public function wpml_loaded()
    {
        $this->logger->debug('WPML loaded and integrated');
    }

    /**
     * Check if WPML is active
     */
    public function is_wpml_active()
    {
        return defined('ICL_SITEPRESS_VERSION') && function_exists('wpml_get_language_information');
    }

    /**
     * Get active languages
     */
    public function get_active_languages()
    {
        if (!$this->is_wpml_active()) {
            return array();
        }

        global $sitepress;
        $languages = $sitepress->get_active_languages();

        return $languages;
    }

    /**
     * Get default language
     */
    public function get_default_language()
    {
        if (!$this->is_wpml_active()) {
            return 'en';
        }

        global $sitepress;
        return $sitepress->get_default_language();
    }

    /**
     * Get post language (using WPML's most reliable method)
     */
    public function get_post_language($post_id)
    {
        if (!$this->is_wpml_active()) {
            return $this->get_default_language();
        }

        // Get post type
        $post = get_post($post_id);
        if (!$post) {
            return $this->get_default_language();
        }

        // Use WPML filter - most reliable source of truth
        $language_code = apply_filters('wpml_element_language_code', null, array(
            'element_id' => $post_id,
            'element_type' => 'post_' . $post->post_type
        ));

        // Fallback to sitepress method
        if (empty($language_code)) {
            global $sitepress;
            if (isset($sitepress)) {
                $language_details = $sitepress->get_element_language_details($post_id, 'post_' . $post->post_type);
                if ($language_details && !empty($language_details->language_code)) {
                    $language_code = $language_details->language_code;
                }
            }
        }

        // Final fallback
        if (empty($language_code)) {
            $language_code = $this->get_default_language();
        }

        return $language_code;
    }

    /**
     * Get translation ID
     */
    public function get_translation_id($post_id, $target_lang)
    {
        if (!$this->is_wpml_active()) {
            return false;
        }

        global $sitepress;
        $post_type = get_post_type($post_id);

        return $sitepress->get_object_id($post_id, $post_type, false, $target_lang);
    }

    /**
     * Check if post has translation
     */
    public function has_translation($post_id, $target_lang)
    {
        $translation_id = $this->get_translation_id($post_id, $target_lang);
        return !empty($translation_id);
    }

    /**
     * Check if post is an Elementor page
     */
    private function is_elementor_page($post_id)
    {
        $elementor_edit_mode = get_post_meta($post_id, '_elementor_edit_mode', true);
        $elementor_data = get_post_meta($post_id, '_elementor_data', true);
        return ($elementor_edit_mode === 'builder' || !empty($elementor_data));
    }

    /**
     * Create translation
     */
    public function create_translation($source_post_id, $target_lang, $translated_data)
    {
        if (!$this->is_wpml_active()) {
            return new WP_Error('wpml_not_active', __('WPML is not active', 'latw-ai-translator-for-wpml'));
        }

        global $sitepress;

        $source_post = get_post($source_post_id);
        if (!$source_post) {
            return new WP_Error('invalid_post', __('Source post not found', 'latw-ai-translator-for-wpml'));
        }

        $source_lang = $this->get_post_language($source_post_id);

        // Ensure source post is set for native editor
        $current_editor = get_post_meta($source_post_id, '_wpml_post_translation_editor_native', true);
        if ($current_editor !== 'yes') {
            update_post_meta($source_post_id, '_wpml_post_translation_editor_native', 'yes');
        }

        // Check if translation already exists
        $existing_translation = $this->get_translation_id($source_post_id, $target_lang);

        if ($existing_translation) {
            // Update existing translation
            return $this->update_translation($existing_translation, $translated_data, $source_post_id);
        }

        $title = $translated_data['title'] ?? $source_post->post_title;

        // For Elementor pages, post_content will be generated from _elementor_data after translation
        $is_elementor = $this->is_elementor_page($source_post_id);
        if ($is_elementor && !isset($translated_data['content'])) {
            $this->logger->debug('post_content will be generated from translated _elementor_data', array(
                'source_post_id' => $source_post_id,
            ));
        }

        // Prepare translated post data
        $translated_post = array(
            'post_title' => $title,
            'post_name' => sanitize_title($title),
            'post_content' => $translated_data['content'] ?? $source_post->post_content,
            'post_excerpt' => $translated_data['excerpt'] ?? $source_post->post_excerpt,
            'post_status' => 'draft',
            'post_type' => $source_post->post_type,
            'post_author' => $source_post->post_author,
            'post_parent' => 0,
            'menu_order' => $source_post->menu_order,
            'comment_status' => $source_post->comment_status,
            'ping_status' => $source_post->ping_status,
        );

        // Create the translated post
        $translated_post_id = wp_insert_post($translated_post);

        if (is_wp_error($translated_post_id)) {
            return $translated_post_id;
        }

        // Set language and link to original
        $this->link_translation($source_post_id, $translated_post_id, $source_lang, $target_lang);

        // Copy post meta
        $this->copy_post_meta($source_post_id, $translated_post_id, $translated_data);

        // Handle taxonomies
        $this->handle_taxonomies($source_post_id, $translated_post_id, $source_lang, $target_lang, $translated_data);

        $this->logger->info('Translation created', array(
            'source_post_id' => $source_post_id,
            'translated_post_id' => $translated_post_id,
            'source_lang' => $source_lang,
            'target_lang' => $target_lang,
        ));

        return $translated_post_id;
    }

    /**
     * Update translation
     */
    public function update_translation($post_id, $translated_data, $source_post_id)
    {
        $update_data = array(
            'ID' => $post_id,
        );

        if (isset($translated_data['title'])) {
            $update_data['post_title'] = $translated_data['title'];
        }

        if (isset($translated_data['content'])) {
            $update_data['post_content'] = $translated_data['content'];
        } else {
            // For Elementor pages, post_content will be generated from _elementor_data
            $is_elementor = $this->is_elementor_page($post_id);
            if ($is_elementor) {
                $this->logger->debug('post_content will be generated from translated _elementor_data', array(
                    'post_id' => $post_id,
                ));
            }
        }

        if (isset($translated_data['excerpt'])) {
            $update_data['post_excerpt'] = $translated_data['excerpt'];
        }

        $result = wp_update_post($update_data);

        if (is_wp_error($result)) {
            return $result;
        }

        // Update meta fields
        $this->copy_post_meta($source_post_id, $post_id, $translated_data);

        return $post_id;
    }

    /**
     * Link translation to original
     */
    private function link_translation($source_id, $translation_id, $source_lang, $target_lang)
    {
        if (!$this->is_wpml_active()) {
            return false;
        }

        global $wpdb;
        $post_type = get_post_type($source_id);

        $trid = apply_filters(
            'wpml_element_trid',
            null,
            $source_id,
            'post_' . $post_type
        );
        if (!$trid) {
            // Create new translation group
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $trid = $wpdb->get_var(
                $wpdb->prepare(
                    'SELECT MAX(trid) FROM %i WHERE element_type = %s',
                    $wpdb->prefix . 'icl_translations',
                    'post_' . $post_type
                )
            );
            $trid = $trid ? $trid + 1 : 1;
        }

        $this->latwaitr_cleanup_orphan_translation($trid, $target_lang);
        do_action(
            'wpml_set_element_language_details',
            array(
                'element_id' => $translation_id,
                'element_type' => 'post_' . $post_type,
                'trid' => $trid,
                'language_code' => $target_lang,
                'source_language_code' => $source_lang,
            )
        );

        return true;
    }

    private function latwaitr_cleanup_orphan_translation($trid, $lang)
    {
        global $wpdb;

        return $wpdb->query(
            $wpdb->prepare(
                "
            DELETE FROM {$wpdb->prefix}icl_translations
            WHERE trid = %d
              AND language_code = %s
              AND element_id IS NULL
            ",
                $trid,
                $lang
            )
        );
    }

    /**
     * Copy post meta
     */
    private function copy_post_meta($source_id, $target_id, $translated_data)
    {
        $meta = get_post_meta($source_id);

        // Blacklisted meta keys that should never be copied
        $blacklist = array(
            '_elementor_element_cache', // Elementor cache - should be regenerated
            '_elementor_strings', // Temporary key used for translation - should not be copied
        );

        // Handle Elementor translations specially
        if (isset($translated_data['meta']['_elementor_strings']) && !empty($translated_data['meta']['_elementor_strings'])) {
            // Restore Elementor data with translated strings
            $restored_elementor_data = LATWAITR()->translator->restore_elementor_translations(
                $source_id,
                $translated_data['meta']['_elementor_strings']
            );

            if (!empty($restored_elementor_data)) {
                // update_post_meta() automatically does wp_slash(), so we DON'T do it manually
                // Otherwise we get double-slashing which breaks unicode characters
                update_post_meta($target_id, '_elementor_data', wp_slash($restored_elementor_data));

                // Copy other Elementor meta fields that are needed
                $elementor_meta_to_copy = array(
                    '_elementor_edit_mode',
                    '_elementor_template_type',
                    '_elementor_version',
                    '_elementor_pro_version',
                    '_elementor_page_settings',
                );

                foreach ($elementor_meta_to_copy as $elementor_key) {
                    $elementor_value = get_post_meta($source_id, $elementor_key, true);
                    if (!empty($elementor_value)) {
                        update_post_meta($target_id, $elementor_key, $elementor_value);
                    }
                }

                // Generate post_content from translated Elementor data
                $this->generate_post_content_from_elementor($target_id);
            }
        }

        foreach ($meta as $key => $values) {
            // Skip blacklisted keys
            if (in_array($key, $blacklist)) {
                continue;
            }

            // Skip _elementor_data if it was already handled above
            if ($key === '_elementor_data' && isset($translated_data['meta']['_elementor_strings'])) {
                continue;
            }

            // If this meta was translated, always copy it (even if it starts with _)
            $was_translated = isset($translated_data['meta'][$key]);

            // Skip private WordPress meta unless it was translated or is on copyable list
            if (substr($key, 0, 1) === '_' && !$was_translated && !in_array($key, $this->get_copyable_meta_keys())) {
                continue;
            }

            foreach ($values as $value) {
                // Check if this meta was translated
                if ($was_translated) {
                    $value = $translated_data['meta'][$key];
                }

                update_post_meta($target_id, $key, maybe_unserialize($value));
            }
        }
    }

    /**
     * Get copyable meta keys
     */
    private function get_copyable_meta_keys()
    {
        $keys = array(
            '_thumbnail_id',
        );

        return apply_filters('latwaitr_copyable_meta_keys', $keys);
    }

    /**
     * Handle taxonomies
     */
    private function handle_taxonomies($source_id, $target_id, $source_lang, $target_lang, $translated_data)
    {
        if (!get_option('latwaitr_translate_taxonomies') === 'yes') {
            return;
        }

        $post_type = get_post_type($source_id);
        $taxonomies = get_object_taxonomies($post_type);

        foreach ($taxonomies as $taxonomy) {
            $terms = wp_get_object_terms($source_id, $taxonomy, array('fields' => 'ids'));

            if (empty($terms)) {
                continue;
            }

            $translated_terms = array();

            foreach ($terms as $term_id) {
                // Get translated term
                $translated_term_id = $this->get_translated_term($term_id, $taxonomy, $target_lang);

                if ($translated_term_id) {
                    $translated_terms[] = $translated_term_id;
                } else {
                    // Term has no translation — create a copy in the target language
                    $new_term_id = $this->create_term_translation($term_id, $taxonomy, $source_lang, $target_lang);
                    if ($new_term_id) {
                        $translated_terms[] = $new_term_id;
                    }
                }
            }

            if (!empty($translated_terms)) {
                wp_set_object_terms($target_id, $translated_terms, $taxonomy);
            }
        }
    }

    /**
     * Get translated term
     */
    private function get_translated_term($term_id, $taxonomy, $target_lang)
    {
        if (!$this->is_wpml_active()) {
            return $term_id;
        }

        global $sitepress;

        return $sitepress->get_object_id($term_id, $taxonomy, false, $target_lang);
    }

    /**
     * Create a term translation by copying the original term to the target language.
     */
    private function create_term_translation($term_id, $taxonomy, $source_lang, $target_lang)
    {
        $original_term = get_term($term_id, $taxonomy);
        if (!$original_term || is_wp_error($original_term)) {
            return false;
        }

        // Handle parent hierarchy — ensure parent is translated first
        $translated_parent = 0;
        if ($original_term->parent) {
            $translated_parent = $this->get_translated_term($original_term->parent, $taxonomy, $target_lang);
            if (!$translated_parent) {
                $translated_parent = $this->create_term_translation($original_term->parent, $taxonomy, $source_lang, $target_lang);
                if (!$translated_parent) {
                    $translated_parent = 0;
                }
            }
        }

        $result = wp_insert_term($original_term->name, $taxonomy, array(
            'slug' => $original_term->slug,
            'description' => $original_term->description,
            'parent' => $translated_parent,
        ));

        if (is_wp_error($result)) {
            $this->logger->info('Failed to create term translation', array(
                'term_id' => $term_id,
                'taxonomy' => $taxonomy,
                'error' => $result->get_error_message(),
            ));
            return false;
        }

        $new_term_id = $result['term_id'];

        // Get the trid of the original term
        $trid = apply_filters('wpml_element_trid', null, $term_id, 'tax_' . $taxonomy);

        if ($trid) {
            do_action('wpml_set_element_language_details', array(
                'element_id' => $new_term_id,
                'element_type' => 'tax_' . $taxonomy,
                'trid' => $trid,
                'language_code' => $target_lang,
                'source_language_code' => $source_lang,
            ));
        }

        $this->logger->info('Created term translation (copy)', array(
            'original_term_id' => $term_id,
            'new_term_id' => $new_term_id,
            'taxonomy' => $taxonomy,
            'name' => $original_term->name,
            'target_lang' => $target_lang,
        ));

        return $new_term_id;
    }

    /**
     * Get term language from WPML
     */
    public function get_term_language($term_id, $taxonomy)
    {
        if (!$this->is_wpml_active()) {
            return $this->get_default_language();
        }

        $language_code = apply_filters('wpml_element_language_code', null, array(
            'element_id' => $term_id,
            'element_type' => 'tax_' . $taxonomy,
        ));

        if (empty($language_code)) {
            global $sitepress;
            if (isset($sitepress)) {
                $language_details = $sitepress->get_element_language_details($term_id, 'tax_' . $taxonomy);
                if ($language_details && !empty($language_details->language_code)) {
                    $language_code = $language_details->language_code;
                }
            }
        }

        if (empty($language_code)) {
            $language_code = $this->get_default_language();
        }

        return $language_code;
    }

    /**
     * Get translated term ID
     */
    public function get_translated_term_id($term_id, $taxonomy, $target_lang)
    {
        if (!$this->is_wpml_active()) {
            return false;
        }

        global $sitepress;
        return $sitepress->get_object_id($term_id, $taxonomy, false, $target_lang);
    }

    /**
     * Create a term translation (public wrapper for use by response checker)
     */
    public function create_term_translation_public($term_id, $taxonomy, $source_lang, $target_lang)
    {
        return $this->create_term_translation($term_id, $taxonomy, $source_lang, $target_lang);
    }

    /**
     * Get languages for translation
     */
    public function get_languages_for_translation($post_id)
    {
        if (!$this->is_wpml_active()) {
            return array();
        }

        $all_languages = $this->get_active_languages();
        $post_lang = $this->get_post_language($post_id);
        $target_languages = get_option('latwaitr_target_languages', array());

        $available = array();

        foreach ($all_languages as $lang_code => $lang_data) {
            // Skip source language
            if ($lang_code === $post_lang) {
                continue;
            }

            // Skip if not in target languages
            if (!empty($target_languages) && !in_array($lang_code, $target_languages)) {
                continue;
            }

            $available[$lang_code] = $lang_data;
        }

        return $available;
    }

    /**
     * Generate post_content from Elementor data
     * Removes divs and attributes but keeps other HTML tags
     */
    private function generate_post_content_from_elementor($post_id)
    {
        // Check if Elementor is active
        if (!did_action('elementor/loaded')) {
            $this->logger->debug('Elementor not loaded, cannot generate post_content', array(
                'post_id' => $post_id,
            ));
            return false;
        }

        try {
            // Get Elementor plugin instance
            $elementor = \Elementor\Plugin::instance();

            // Check if frontend exists
            if (!isset($elementor->frontend)) {
                $this->logger->debug('Elementor frontend not available', array(
                    'post_id' => $post_id,
                ));
                return false;
            }

            // Get the rendered content
            $content = $elementor->frontend->get_builder_content($post_id, false);

            if (!empty($content)) {
                // Clean HTML - remove divs and attributes
                $cleaned_content = $this->clean_elementor_html($content);

                // Update post_content with cleaned HTML
                wp_update_post(array(
                    'ID' => $post_id,
                    'post_content' => $cleaned_content,
                ));

                $this->logger->debug('Generated post_content from Elementor data', array(
                    'post_id' => $post_id,
                    'content_length' => strlen($cleaned_content),
                ));

                return true;
            } else {
                $this->logger->debug('Elementor rendered empty content', array(
                    'post_id' => $post_id,
                ));
                return false;
            }
        } catch (\Exception $e) {
            $this->logger->error('Error generating post_content from Elementor', array(
                'post_id' => $post_id,
                'error' => $e->getMessage(),
            ));
            return false;
        }
    }

    /**
     * Clean Elementor HTML - remove divs and all attributes
     */
    private function clean_elementor_html($html)
    {
        // Remove all divs but keep their content
        $html = preg_replace('/<div[^>]*>/i', '', $html);
        $html = preg_replace('/<\/div>/i', '', $html);

        // Remove all attributes from remaining tags
        $html = preg_replace('/<(\w+)\s+[^>]*>/i', '<$1>', $html);

        // Clean up multiple spaces and newlines
        $html = preg_replace('/\s+/', ' ', $html);
        $html = trim($html);

        return $html;
    }
}
