<?php
/**
 * Main Translator class for LATW
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_Translator {

    /**
     * Logger
     */
    private $logger;

    /**
     * LLM API
     */
    private $llm;

    /**
     * WPML Integration
     */
    private $wpml;

    /**
     * Security
     */
    private $security;

    /**
     * Constructor
     */
    public function __construct() {
        $this->logger = LATWAITR()->logger;
        $this->llm = LATWAITR()->llm;
        $this->wpml = LATWAITR()->wpml;
        $this->security = LATWAITR()->security;
    }

    /**
     * Gather content to translate (public for batch API usage)
     */
    public function gather_content($post_id) {
        $post = get_post($post_id);
        $content = array();

        // Check if this is an Elementor page
        $is_elementor_page = $this->is_elementor_page($post_id);

        // Title
        if (get_option('latwaitr_translate_title') === 'yes') {
            $content['title'] = $post->post_title;
        }

        // Content - skip if Elementor page (content is in _elementor_data, post_content will be generated from it)
        if (get_option('latwaitr_translate_content') === 'yes' && !$is_elementor_page) {
            $content['content'] = $post->post_content;
        } elseif ($is_elementor_page) {
            $this->logger->debug('Skipping post_content extraction for Elementor page (will be generated from translated _elementor_data)', array(
                'post_id' => $post_id,
            ));
        }

        // Excerpt
        if (get_option('latwaitr_translate_excerpt') === 'yes' && !empty($post->post_excerpt)) {
            $content['excerpt'] = $post->post_excerpt;
        }

        // Custom fields
        if (get_option('latwaitr_translate_custom_fields') === 'yes') {
            $custom_fields = $this->get_translatable_custom_fields($post_id);
            if (!empty($custom_fields)) {
                $content['meta'] = $custom_fields;
            }
        }

        // SEO fields (Yoast, Rank Math, etc.)
        if (get_option('latwaitr_translate_seo') === 'yes') {
            $seo_fields = $this->get_seo_fields($post_id);
            if (!empty($seo_fields)) {
                if (!isset($content['meta'])) {
                    $content['meta'] = array();
                }
                $content['meta'] = array_merge($content['meta'], $seo_fields);
            }
        }

        // Page Builder fields (Elementor, Beaver Builder, etc.)
        $builder_fields = $this->get_page_builder_fields($post_id);
        if (!empty($builder_fields)) {
            if (!isset($content['meta'])) {
                $content['meta'] = array();
            }
            $content['meta'] = array_merge($content['meta'], $builder_fields);
        }

        return apply_filters('latwaitr_content_to_translate', $content, $post_id);
    }

    /**
     * Get translatable custom fields
     */
    private function get_translatable_custom_fields($post_id) {
        $custom_fields = array();
        $all_meta = get_post_meta($post_id);

        // Get field whitelist/blacklist
        $whitelist = get_option('latwaitr_custom_fields_whitelist', array());
        $blacklist = get_option('latwaitr_custom_fields_blacklist', array());

        foreach ($all_meta as $key => $values) {
            // Skip private fields
            if (substr($key, 0, 1) === '_') {
                continue;
            }

            // Check whitelist/blacklist
            if (!empty($whitelist) && !in_array($key, $whitelist)) {
                continue;
            }

            if (!empty($blacklist) && in_array($key, $blacklist)) {
                continue;
            }

            foreach ($values as $value) {
                if (is_string($value) && !empty($value)) {
                    $custom_fields[$key] = $value;
                    break; // Only take first value
                }
            }
        }

        return $custom_fields;
    }

    /**
     * Get SEO fields
     */
    private function get_seo_fields($post_id) {
        $seo_fields = array();

        // Yoast SEO
        $yoast_title = get_post_meta($post_id, '_yoast_wpseo_title', true);
        $yoast_desc = get_post_meta($post_id, '_yoast_wpseo_metadesc', true);

        if (!empty($yoast_title)) {
            $seo_fields['_yoast_wpseo_title'] = $yoast_title;
        }

        if (!empty($yoast_desc)) {
            $seo_fields['_yoast_wpseo_metadesc'] = $yoast_desc;
        }

        // Rank Math
        $rankmath_title = get_post_meta($post_id, 'rank_math_title', true);
        $rankmath_desc = get_post_meta($post_id, 'rank_math_description', true);

        if (!empty($rankmath_title)) {
            $seo_fields['rank_math_title'] = $rankmath_title;
        }

        if (!empty($rankmath_desc)) {
            $seo_fields['rank_math_description'] = $rankmath_desc;
        }

        return $seo_fields;
    }

    /**
     * Check if page is built with Elementor
     *
     * @param int $post_id Post ID
     * @return bool
     */
    private function is_elementor_page($post_id) {
        // Check if Elementor edit mode is enabled
        $elementor_edit_mode = get_post_meta($post_id, '_elementor_edit_mode', true);

        // Check if Elementor data exists
        $elementor_data = get_post_meta($post_id, '_elementor_data', true);

        // It's an Elementor page if:
        // 1. Edit mode is 'builder' OR
        // 2. Elementor data exists and is not empty
        $is_elementor = ($elementor_edit_mode === 'builder' || !empty($elementor_data));

        if ($is_elementor) {
            $this->logger->debug('Detected Elementor page', array(
                'post_id' => $post_id,
                'edit_mode' => $elementor_edit_mode,
                'has_data' => !empty($elementor_data),
            ));
        }

        return $is_elementor;
    }

    /**
     * Get Page Builder fields (Elementor)
     */
    private function get_page_builder_fields($post_id) {
        $builder_fields = array();

        // Elementor only
        $elementor_data = get_post_meta($post_id, '_elementor_data', true);
        if (!empty($elementor_data)) {
            // Parse Elementor data using new system
            $extracted = $this->extract_elementor_translatable_strings_new($elementor_data, $post_id);

            if (!empty($extracted['strings'])) {
                $builder_fields['_elementor_strings'] = $extracted['strings'];
            }
        }

        return $builder_fields;
    }

    /**
     * Extract translatable strings from Elementor data using new system
     */
    private function extract_elementor_translatable_strings_new($elementor_data, $post_id) {
        $strings = array();
        $string_objects = array();

        // Decode JSON
        $data = json_decode($elementor_data, true);
        if (!is_array($data)) {
            return array('strings' => array(), 'string_objects' => array());
        }

        $translatable_nodes = new LATWAITR_Elementor_Translatable_Nodes();

        // Recursively process all elements
        $this->process_elementor_elements_new($data, $translatable_nodes, $strings, $string_objects);

        // Store string objects in post meta for later restoration
        if (!empty($string_objects)) {
            update_post_meta($post_id, '_latwaitr_elementor_string_objects', $string_objects);
        }

        // Store original data for restoration
        update_post_meta($post_id, '_latwaitr_elementor_original_data', wp_slash($elementor_data));

        return array(
            'strings' => $strings,
            'string_objects' => $string_objects
        );
    }

    /**
     * Recursively process Elementor elements
     */
    private function process_elementor_elements_new(&$elements, $translatable_nodes, &$strings, &$string_objects) {
        if (!is_array($elements)) {
            return;
        }

        foreach ($elements as $index => &$element) {
            // Skip invalid elements
            if (!is_array($element)) {
                continue;
            }

            if (!isset($element['id'])) {
                $this->logger->debug('Skipping element without ID', array(
                    'index' => $index,
                    'element_type' => isset($element['elType']) ? $element['elType'] : 'unknown',
                ));
                continue;
            }

            $node_id = $element['id'];

            $widget_type = isset($element['widgetType']) ? $element['widgetType'] : null;
            $el_type = isset($element['elType']) ? $element['elType'] : 'unknown';
            $display_type = $widget_type ? $widget_type : $el_type;

            // Get translatable strings from this element
            try {
                $element_strings = $translatable_nodes->get($node_id, $element);

                if (!empty($element_strings)) {
                    $this->logger->debug('Extracted strings from element', array(
                        'node_id' => $node_id,
                        'widget_type' => $display_type,
                        'string_count' => count($element_strings),
                        'string_names' => array_map(function($s) { return $s->get_name(); }, $element_strings),
                    ));
                } else {
                    // No strings extracted - log full element for debugging
                    $settings_keys = isset($element['settings']) && is_array($element['settings'])
                        ? array_keys($element['settings'])
                        : array();
                    $text_settings = array();
                    if (isset($element['settings']) && is_array($element['settings'])) {
                        foreach ($element['settings'] as $k => $v) {
                            if (is_string($v) && strlen(trim($v)) >= 3) {
                                $text_settings[$k] = mb_substr($v, 0, 100);
                            }
                        }
                    }
                    $this->logger->debug('No strings extracted from element', array(
                        'node_id' => $node_id,
                        'elType' => $el_type,
                        'widgetType' => $widget_type,
                        'display_type' => $display_type,
                        'settings_keys' => $settings_keys,
                        'text_settings_preview' => $text_settings,
                        'has_elements' => isset($element['elements']) && !empty($element['elements']),
                        'children_count' => isset($element['elements']) ? count($element['elements']) : 0,
                        'full_element' => $element,
                    ));
                }

                foreach ($element_strings as $pb_string) {
                    $string_name = $pb_string->get_name();
                    $strings[$string_name] = $pb_string->get_value();

                    // Store string object data for restoration
                    $string_objects[$string_name] = array(
                        'node_id' => $node_id,
                        'title' => $pb_string->get_title(),
                        'editor_type' => $pb_string->get_editor_type(),
                        'wrap_tag' => $pb_string->get_wrap_tag(),
                    );
                }
            } catch (Exception $e) {
                $this->logger->error('Error extracting strings from element', array(
                    'node_id' => $node_id,
                    'widget_type' => $display_type,
                    'error' => $e->getMessage(),
                    'full_element' => $element,
                ));
            }

            // Recursively process nested elements
            if (isset($element['elements']) && is_array($element['elements'])) {
                $this->process_elementor_elements_new($element['elements'], $translatable_nodes, $strings, $string_objects);
            }
        }
    }

    /**
     * Restore translated strings back into Elementor data structure
     */
    public function restore_elementor_translations($source_post_id, $translated_strings) {
        // Get stored data from post meta
        $original_data = get_post_meta($source_post_id, '_latwaitr_elementor_original_data', true);
        $string_objects = get_post_meta($source_post_id, '_latwaitr_elementor_string_objects', true);

        if (empty($original_data) || empty($string_objects)) {
            // No stored data found, return original data
            $fallback_data = get_post_meta($source_post_id, '_elementor_data', true);
            $fallback_data = wp_unslash($fallback_data);
            $this->logger->warning('No stored Elementor data found for restoration', array(
                'source_post_id' => $source_post_id,
                'has_original_data' => !empty($original_data),
                'has_string_objects' => !empty($string_objects),
            ));
            return $fallback_data;
        }

        $this->logger->debug('Restoring Elementor translations', array(
            'source_post_id' => $source_post_id,
            'original_data_length' => strlen($original_data),
            'string_objects_count' => count($string_objects),
            'translated_strings_count' => count($translated_strings),
        ));

        // Decode JSON
        $data = json_decode($original_data, true);
        if (!is_array($data)) {
            $this->logger->error('Failed to decode Elementor data', array(
                'source_post_id' => $source_post_id,
                'json_error' => json_last_error_msg(),
            ));
            return $original_data;
        }

        $this->logger->debug('Decoded Elementor structure', array(
            'elements_count' => count($data),
            'first_element_type' => isset($data[0]['elType']) ? $data[0]['elType'] : 'unknown',
        ));

        // Create PB String objects and use new system to update
        $translatable_nodes = new LATWAITR_Elementor_Translatable_Nodes();

        $updated_count = 0;
        $skipped_count = 0;

        // Process all translated strings
        foreach ($translated_strings as $string_key => $translated_value) {
            if (isset($string_objects[$string_key])) {
                $string_data = $string_objects[$string_key];

                // Skip empty translations
                if (empty($translated_value) || !is_string($translated_value)) {
                    $this->logger->debug('Skipping empty translation', array(
                        'string_key' => $string_key,
                    ));
                    $skipped_count++;
                    continue;
                }

                // Create LATWAITR_PB_String object
                $pb_string = new LATWAITR_PB_String(
                    $translated_value,
                    $string_key,
                    $string_data['title'],
                    $string_data['editor_type'],
                    isset($string_data['wrap_tag']) ? $string_data['wrap_tag'] : ''
                );

                $node_id = $string_data['node_id'];

                // Update elements recursively
                $this->update_elementor_elements_recursive($data, $translatable_nodes, $node_id, $pb_string);
                $updated_count++;
            } else {
                $this->logger->warning('String object not found for translation', array(
                    'string_key' => $string_key,
                ));
                $skipped_count++;
            }
        }

        $this->logger->info('Elementor translation summary', array(
            'updated_strings' => $updated_count,
            'skipped_strings' => $skipped_count,
            'total_strings' => count($translated_strings),
        ));

        // Note: We keep _latwaitr_elementor_string_objects and _latwaitr_elementor_original_data
        // in post meta to allow multiple languages to be translated simultaneously.
        // These will be overwritten on next translation.

        // Use json_encode with flags matching Elementor's format
        // UNESCAPED_UNICODE - prevents encoding unicode characters
        // UNESCAPED_SLASHES - prevents escaping slashes (crucial for URLs, CSS, JS)
        $encoded = json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

        if ($encoded === false) {
            $this->logger->error('Failed to encode Elementor data', array(
                'json_error' => json_last_error_msg(),
            ));
            return $original_data;
        }

        $this->logger->debug('Encoded Elementor data sample', array(
            'encoded_length' => strlen($encoded),
            'encoded_sample' => substr($encoded, 0, 500),
        ));

        return $encoded;
    }

    /**
     * Recursively update Elementor elements with translations
     */
    private function update_elementor_elements_recursive(&$elements, $translatable_nodes, $target_node_id, $pb_string) {
        foreach ($elements as $index => &$element) {
            // Check if this is the target element
            if (isset($element['id']) && $element['id'] === $target_node_id) {
                // Found the target element, update it
                $original_element = $element;
                $updated_element = $translatable_nodes->update($target_node_id, $element, $pb_string);

                // Only update if we got a valid response
                if (is_array($updated_element) && !empty($updated_element)) {
                    $element = $updated_element;

                    if ($element !== $original_element) {
                        $this->logger->debug('Updated element with translation', array(
                            'node_id' => $target_node_id,
                            'string_name' => $pb_string->get_name(),
                        ));
                    } else {
                        $this->logger->warning('Failed to apply translation to element', array(
                            'node_id' => $target_node_id,
                            'string_name' => $pb_string->get_name(),
                        ));
                    }
                }

                // Continue searching - there might be multiple instances
            }

            // Recursively process nested elements
            if (isset($element['elements']) && is_array($element['elements'])) {
                $this->update_elementor_elements_recursive($element['elements'], $translatable_nodes, $target_node_id, $pb_string);
            }
        }
    }
}
