<?php
/**
 * Security handler for LATW
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_Security {

    /**
     * Required capability
     */
    const REQUIRED_CAPABILITY = 'manage_options';

    /**
     * Nonce action
     */
    const NONCE_ACTION = 'latwaitraitr_action';

    /**
     * Nonce name
     */
    const NONCE_NAME = 'latwaitraitr_nonce';

    /**
     * Constructor
     */
    public function __construct() {
        // Hook into init
        add_action('init', array($this, 'init'));
    }

    /**
     * Initialize security
     */
    public function init() {
        // Add security headers
        add_action('send_headers', array($this, 'add_security_headers'));
    }

    /**
     * Check if user has permission
     */
    public function check_permission($capability = null) {
        if ($capability === null) {
            $capability = self::REQUIRED_CAPABILITY;
        }

        return current_user_can($capability);
    }

    /**
     * Verify nonce
     */
    public function verify_nonce($nonce = null, $action = null) {
        if ($nonce === null) {
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This is the nonce retrieval function itself
            $nonce = isset($_REQUEST[self::NONCE_NAME]) ? sanitize_text_field(wp_unslash($_REQUEST[self::NONCE_NAME])) : '';
        }

        if ($action === null) {
            $action = self::NONCE_ACTION;
        }

        return wp_verify_nonce($nonce, $action);
    }

    /**
     * Create nonce
     */
    public function create_nonce($action = null) {
        if ($action === null) {
            $action = self::NONCE_ACTION;
        }

        return wp_create_nonce($action);
    }

    /**
     * Get nonce field
     */
    public function get_nonce_field($action = null, $name = null, $referer = true, $echo = false) {
        if ($action === null) {
            $action = self::NONCE_ACTION;
        }

        if ($name === null) {
            $name = self::NONCE_NAME;
        }

        return wp_nonce_field($action, $name, $referer, $echo);
    }

    /**
     * Check AJAX referer
     */
    public function check_ajax_referer($action = null) {
        if ($action === null) {
            $action = self::NONCE_ACTION;
        }

        check_ajax_referer($action, self::NONCE_NAME);
    }

    /**
     * Sanitize API key
     */
    public function sanitize_api_key($key) {
        return sanitize_text_field(trim($key));
    }

    /**
     * Mask API key for display
     */
    public function mask_api_key($key) {
        if (empty($key)) {
            return '';
        }

        $length = strlen($key);
        if ($length <= 8) {
            return str_repeat('*', $length);
        }

        return substr($key, 0, 4) . str_repeat('*', $length - 8) . substr($key, -4);
    }

    /**
     * Sanitize language code
     */
    public function sanitize_language_code($code) {
        return sanitize_text_field(strtolower(trim($code)));
    }

    /**
     * Sanitize translation mode
     */
    public function sanitize_translation_mode($mode) {
        $allowed = array('on-demand', 'auto');
        return in_array($mode, $allowed) ? $mode : 'on-demand';
    }

    /**
     * Sanitize post types array
     */
    public function sanitize_post_types($post_types) {
        if (!is_array($post_types)) {
            return array();
        }

        return array_map('sanitize_text_field', $post_types);
    }

    /**
     * Sanitize yes/no option
     */
    public function sanitize_yes_no($value) {
        return ($value === 'yes' || $value === '1' || $value === 1 || $value === true) ? 'yes' : 'no';
    }

    /**
     * Add security headers
     */
    public function add_security_headers() {
        if (!is_admin()) {
            return;
        }

        // Only add headers on plugin pages
        $screen = get_current_screen();
        if ($screen && strpos($screen->id, 'latwaitr') !== false) {
            header('X-Content-Type-Options: nosniff');
            header('X-Frame-Options: SAMEORIGIN');
            header('X-XSS-Protection: 1; mode=block');
        }
    }

    /**
     * Check rate limit
     */
    public function check_rate_limit($action, $limit = 10, $period = 60) {
        $transient_key = 'latwaitraitr_rate_limit_' . $action . '_' . get_current_user_id();
        $attempts = get_transient($transient_key);

        if ($attempts === false) {
            set_transient($transient_key, 1, $period);
            return true;
        }

        if ($attempts >= $limit) {
            return false;
        }

        set_transient($transient_key, $attempts + 1, $period);
        return true;
    }

    /**
     * Sanitize translation content
     */
    public function sanitize_translation_content($content, $field_type = 'post_content') {
        switch ($field_type) {
            case 'post_title':
            case 'post_excerpt':
                return sanitize_text_field($content);

            case 'post_content':
                return wp_kses_post($content);

            case 'meta':
            case 'acf':
                // Allow more HTML for custom fields
                return wp_kses_post($content);

            default:
                return sanitize_textarea_field($content);
        }
    }

    /**
     * Validate OpenAI API key format
     */
    public function validate_openai_key($key) {
        // OpenAI keys start with 'sk-'
        if (substr($key, 0, 3) !== 'sk-') {
            return false;
        }

        // Should be at least 20 characters
        if (strlen($key) < 20) {
            return false;
        }

        return true;
    }
}
