<?php
/**
 * Logger for LATW
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_Logger {

    /**
     * Log levels
     */
    const LEVEL_DEBUG = 'debug';
    const LEVEL_INFO = 'info';
    const LEVEL_WARNING = 'warning';
    const LEVEL_ERROR = 'error';

    /**
     * Log file path
     */
    private $log_file;

    /**
     * Constructor
     */
    public function __construct() {
        $upload_dir = wp_upload_dir();
        $plugin_uploads_path = $upload_dir['basedir'] . '/latw-ai-translator-for-wpml/';
        $log_dir = $plugin_uploads_path . 'logs';

        // Create log directory if it doesn't exist
        if (!file_exists($log_dir)) {
            wp_mkdir_p($log_dir);

            // Add .htaccess to protect logs
            $htaccess = $log_dir . '/.htaccess';
            if (!file_exists($htaccess)) {
                file_put_contents($htaccess, 'Deny from all');
            }
        }

        $this->log_file = $log_dir . '/latwaitr-' . gmdate('Y-m-d') . '.log';
    }

    /**
     * Log a message
     */
    public function log($message, $level = self::LEVEL_INFO, $context = array()) {
        // Check if debug mode is enabled for debug level messages
        if ($level === self::LEVEL_DEBUG && get_option('latwaitr_debug_mode') !== 'yes') {
            return;
        }

        $timestamp = current_time('mysql');
        $formatted_message = $this->format_message($message, $level, $timestamp, $context);

        // Write to file
        error_log($formatted_message . PHP_EOL, 3, $this->log_file);

        // If error level, also log to PHP error log
        if ($level === self::LEVEL_ERROR) {
            error_log('LATW Error: ' . $message);
        }

        do_action('latwaitr_log', $message, $level, $context);
    }

    /**
     * Format log message
     */
    private function format_message($message, $level, $timestamp, $context) {
        $formatted = sprintf(
            '[%s] [%s] %s',
            $timestamp,
            strtoupper($level),
            $message
        );

        if (!empty($context)) {
            $formatted .= ' | Context: ' . json_encode($context);
        }

        return $formatted;
    }

    /**
     * Debug log
     */
    public function debug($message, $context = array()) {
        $this->log($message, self::LEVEL_DEBUG, $context);
    }

    /**
     * Info log
     */
    public function info($message, $context = array()) {
        $this->log($message, self::LEVEL_INFO, $context);
    }

    /**
     * Warning log
     */
    public function warning($message, $context = array()) {
        $this->log($message, self::LEVEL_WARNING, $context);
    }

    /**
     * Error log
     */
    public function error($message, $context = array()) {
        $this->log($message, self::LEVEL_ERROR, $context);
    }

    /**
     * Get recent logs
     */
    public function get_recent_logs($lines = 100) {
        if (!file_exists($this->log_file)) {
            return array();
        }

        $file = new SplFileObject($this->log_file, 'r');
        $file->seek(PHP_INT_MAX);
        $total_lines = $file->key() + 1;

        $start_line = max(0, $total_lines - $lines);
        $file->seek($start_line);

        $logs = array();
        while (!$file->eof()) {
            $line = trim($file->current());
            if (!empty($line)) {
                $logs[] = $line;
            }
            $file->next();
        }

        return array_reverse($logs);
    }

    /**
     * Clear old logs
     */
    public function clear_old_logs($days = 30) {
        $upload_dir = wp_upload_dir();
        $log_dir = $upload_dir['basedir'] . '/latw-ai-translator-for-wpml/logs';

        if (!file_exists($log_dir)) {
            return;
        }

        $files = glob($log_dir . '/latwaitr-*.log');
        $cutoff = strtotime("-{$days} days");

        foreach ($files as $file) {
            if (filemtime($file) < $cutoff) {
                wp_delete_file($file);
            }
        }
    }

    /**
     * Get log file path
     */
    public function get_log_file() {
        return $this->log_file;
    }
}
