<?php
/**
 * Database handler for LATW
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_Database {

    /**
     * Get translations table name
     *
     * @return string
     */
    public static function get_table_name() {
        global $wpdb;
        return $wpdb->prefix . 'latwaitr_translations';
    }

    /**
     * Create plugin tables
     */
    public static function create_tables() {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();
        $table_name = self::get_table_name();

        // Translations table - table name is safe as it uses $wpdb->prefix
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $sql = "CREATE TABLE IF NOT EXISTS {$table_name} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            post_id bigint(20) unsigned NOT NULL,
            source_lang varchar(10) NOT NULL,
            target_lang varchar(10) NOT NULL,
            source_post_id bigint(20) unsigned DEFAULT NULL,
            status varchar(20) NOT NULL DEFAULT 'pending',
            translation_type varchar(50) DEFAULT 'on-demand',
            element_type varchar(20) NOT NULL DEFAULT 'post',
            taxonomy varchar(50) DEFAULT NULL,
            fields_translated text DEFAULT NULL,
            input_tokens_used int(11) DEFAULT 0,
            output_tokens_used int(11) DEFAULT 0,
            actual_cost decimal(10,6) DEFAULT 0.000000,
            error_message text DEFAULT NULL,
            retry_count int(11) DEFAULT 0,
            response_id varchar(100) DEFAULT NULL,
            response_status varchar(50) DEFAULT NULL,
            response_submitted_at datetime DEFAULT NULL,
            response_completed_at datetime DEFAULT NULL,
            request_data longtext DEFAULT NULL,
            created_at datetime NOT NULL,
            updated_at datetime NOT NULL,
            completed_at datetime DEFAULT NULL,
            PRIMARY KEY  (id),
            KEY post_id (post_id),
            KEY source_post_id (source_post_id),
            KEY status (status),
            KEY target_lang (target_lang),
            KEY created_at (created_at),
            KEY response_id (response_id),
            KEY response_status (response_status)
        ) {$charset_collate};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);

        // Save database version
        update_option('latwaitr_db_version', LATWAITR_VERSION);
    }

    /**
     * Drop plugin tables (used on uninstall)
     */
    public static function drop_tables() {
        global $wpdb;

        $table_name = self::get_table_name();

        // Table name is safe as it uses $wpdb->prefix
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $wpdb->query(
            $wpdb->prepare( 'DROP TABLE IF EXISTS %i', $table_name )
        );

        delete_option('latwaitr_db_version');
    }

    /**
     * Log translation
     */
    public static function log_translation($data) {
        global $wpdb;

        $table_name = self::get_table_name();

        $defaults = array(
            'post_id' => 0,
            'source_lang' => '',
            'target_lang' => '',
            'source_post_id' => null,
            'status' => 'pending',
            'translation_type' => 'on-demand',
            'element_type' => 'post',
            'taxonomy' => null,
            'fields_translated' => '',
            'input_tokens_used' => 0,
            'output_tokens_used' => 0,
            'actual_cost' => 0,
            'error_message' => null,
            'retry_count' => 0,
            'created_at' => current_time('mysql', true),
            'updated_at' => current_time('mysql', true),
            'completed_at' => null,
        );

        $data = wp_parse_args($data, $defaults);

        // Serialize fields_translated if array
        if (is_array($data['fields_translated'])) {
            $data['fields_translated'] = wp_json_encode($data['fields_translated']);
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        $wpdb->insert($table_name, $data);

        return $wpdb->insert_id;
    }

    /**
     * Update translation log
     */
    public static function update_translation($id, $data) {
        global $wpdb;

        $table_name = self::get_table_name();

        $data['updated_at'] = current_time('mysql', true);

        // Serialize fields_translated if array
        if (isset($data['fields_translated']) && is_array($data['fields_translated'])) {
            $data['fields_translated'] = wp_json_encode($data['fields_translated']);
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
        return $wpdb->update(
            $table_name,
            $data,
            array('id' => $id),
            null,
            array('%d')
        );
    }

    /**
     * Get translation by ID
     */
    public static function get_translation($id) {
        global $wpdb;

        $table_name = self::get_table_name();

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $translation = $wpdb->get_row(
            $wpdb->prepare(
                'SELECT * FROM %i WHERE id = %d',
                $table_name,
                $id
            ),
            ARRAY_A
        );

        if ($translation && !empty($translation['fields_translated'])) {
            $translation['fields_translated'] = json_decode($translation['fields_translated'], true);
        }

        return $translation;
    }

    /**
     * Get monthly costs
     */
    public static function get_monthly_costs($year = null, $month = null) {
        global $wpdb;

        if (!$year) {
            $year = gmdate('Y');
        }
        if (!$month) {
            $month = gmdate('m');
        }

        $table_name = self::get_table_name();

        $start_date = "{$year}-{$month}-01 00:00:00";
        $end_date = gmdate('Y-m-t 23:59:59', strtotime($start_date));

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $result = $wpdb->get_var(
            $wpdb->prepare(
                'SELECT SUM(actual_cost) FROM %i WHERE status = %s AND created_at BETWEEN %s AND %s',
                $table_name,
                'completed',
                $start_date,
                $end_date
            )
        );

        return floatval($result);
    }

    /**
     * Get translation stats
     */
    public static function get_stats($days = 30) {
        global $wpdb;

        $table_name = self::get_table_name();
        $date_from = gmdate('Y-m-d H:i:s', strtotime("-{$days} days"));

        $stats = array(
            'total' => 0,
            'completed' => 0,
            'failed' => 0,
            'queued' => 0,
            'in-progress' => 0,
            'total_cost' => 0,
            'total_tokens' => 0,
        );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $results = $wpdb->get_results(
            $wpdb->prepare(
                'SELECT status, COUNT(*) as count, SUM(actual_cost) as cost, SUM(input_tokens_used) + SUM(output_tokens_used) as tokens FROM %i WHERE created_at >= %s GROUP BY status',
                $table_name,
                $date_from
            ),
            ARRAY_A
        );

        foreach ($results as $row) {
            $stats['total'] += $row['count'];
            $stats[$row['status']] = $row['count'];
            $stats['total_cost'] += floatval($row['cost']);
            $stats['total_tokens'] += intval($row['tokens']);
        }

        return $stats;
    }

    /**
     * Upgrade database schema if needed
     */
    public static function maybe_upgrade() {
        global $wpdb;
        $table_name = self::get_table_name();
        $current_version = get_option('latwaitr_db_version', '0');

        if (version_compare($current_version, '1.2.0', '<')) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $columns = $wpdb->get_col("DESCRIBE {$table_name}", 0);

            if (!in_array('element_type', $columns)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $wpdb->query("ALTER TABLE {$table_name} ADD COLUMN element_type varchar(20) NOT NULL DEFAULT 'post' AFTER translation_type");
            }
            if (!in_array('taxonomy', $columns)) {
                // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $wpdb->query("ALTER TABLE {$table_name} ADD COLUMN taxonomy varchar(50) DEFAULT NULL AFTER element_type");
            }

            update_option('latwaitr_db_version', LATWAITR_VERSION);
        }
    }

    /**
     * Get translations by response status (for response checker)
     */
    public static function get_translations_by_response_status($statuses, $limit = 50) {
        global $wpdb;

        $table_name = self::get_table_name();

        if (!is_array($statuses)) {
            $statuses = array($statuses);
        }

        $placeholders = implode(',', array_fill(0, count($statuses), '%s'));

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $translations = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM %i WHERE status IN ({$placeholders}) AND response_id IS NOT NULL ORDER BY response_submitted_at ASC LIMIT %d",
                array_merge(array($table_name), $statuses, array($limit))
            ),
            ARRAY_A
        );

        return $translations;
    }
}
