<?php
/**
 * AJAX Handler for LATW
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_AJAX_Handler {

    /**
     * Security
     */
    private $security;

    /**
     * Logger
     */
    private $logger;

    /**
     * Constructor
     */
    public function __construct() {
        $this->security = LATWAITR()->security;
        $this->logger = LATWAITR()->logger;

        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        add_action('wp_ajax_latwaitr_queue_translations', array($this, 'queue_translations'));
        add_action('wp_ajax_latwaitr_get_translation_status', array($this, 'get_translation_status'));
        add_action('wp_ajax_latwaitr_get_stats', array($this, 'get_stats'));
        add_action('wp_ajax_latwaitr_get_translated_posts', array($this, 'get_translated_posts'));
        add_action('wp_ajax_latwaitr_get_translated_terms', array($this, 'get_translated_terms'));
        add_action('wp_ajax_latwaitr_queue_term_translations', array($this, 'queue_term_translations'));
        add_action('wp_ajax_latwaitr_check_responses', array($this, 'check_responses'));
    }

    /**
     * Queue translations (asynchronous)
     */
    public function queue_translations() {
        // Wrap EVERYTHING in try-catch to prevent 500 errors
        try {
            $this->logger->info("queue_translations");

            // Verify nonce first
            if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'latwaitr_action')) {
                throw new Exception(__('Security check failed', 'latw-ai-translator-for-wpml'));
            }

            // Check permission
            if (!current_user_can('manage_options')) {
                throw new Exception(__('Permission denied', 'latw-ai-translator-for-wpml'));
            }

            // Check OpenAI API key
            $api_key = get_option('latwaitr_openai_api_key');
            if (empty($api_key)) {
                throw new Exception(__('OpenAI API Key is not configured. Please add your API key in the settings.', 'latw-ai-translator-for-wpml'));
            }

            // Get parameters - nonce already verified above
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above
            $post_ids = isset($_POST['post_ids']) ? array_map('intval', (array) wp_unslash($_POST['post_ids'])) : array();
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above
            $target_languages = ['en'];

            if (empty($post_ids) || empty($target_languages)) {
                throw new Exception(__('Invalid parameters', 'latw-ai-translator-for-wpml'));
            }

            $queue = LATWAITR()->queue;
            $wpml = LATWAITR()->wpml;
            $queued_count = 0;
            $queue_full = false;

            $this->logger->info('LATW AJAX - queue_translations called with post_ids: ' . print_r($post_ids, true));
            $this->logger->info('LATW AJAX - target_languages: ' . print_r($target_languages, true));

            foreach ($post_ids as $post_id) {
                if ($queue_full) {
                    break;
                }

                // Get available languages for this post
                $post_source_lang = $wpml->get_post_language($post_id);
                $this->logger->info('LATW AJAX - Post ' . $post_id . ' source language: ' . $post_source_lang);

                foreach ($target_languages as $target_lang) {
                    // Skip if target language is same as source
                    if ($target_lang === $post_source_lang) {
                        $this->logger->info('LATW AJAX - Skipping ' . $target_lang . ' (same as source)');
                        continue;
                    }

                    // Check if translation already exists
                    $translation_exists = $wpml->has_translation($post_id, $target_lang);
                    $this->logger->info('LATW AJAX - Translation exists for ' . $post_id . ' -> ' . $target_lang . ': ' . ($translation_exists ? 'yes' : 'no'));

                    // Only queue if translation doesn't exist or needs update
                    // (we'll let the handler decide if update is needed based on date)
                    $action_id = $queue->queue_translation($post_id, $target_lang, array(
                        'publish' => false,
                    ));

                    if (is_wp_error($action_id)) {
                        $error_message = $action_id->get_error_message();
                        $error_code = $action_id->get_error_code();
                        $this->logger->error('LATW AJAX - Error queueing ' . $post_id . ' -> ' . $target_lang . ': [' . $error_code . '] ' . $error_message);

                        // Queue full - stop trying, but don't lose what was already queued
                        if ($error_code === 'queue_full') {
                            $queue_full = true;
                            break;
                        }

                        // For other errors, continue but log
                        continue;
                    }

                    $this->logger->info('LATW AJAX - Queue result for ' . $post_id . ' -> ' . $target_lang . ': action_id=' . $action_id);
                    $queued_count++;
                    $this->logger->info('LATW AJAX - Successfully queued (total: ' . $queued_count . ')');
                }
            }

            $this->logger->info('LATW AJAX - Total queued: ' . $queued_count . ', queue_full: ' . ($queue_full ? 'yes' : 'no'));

            if ($queued_count === 0 && !$queue_full) {
                throw new Exception(__('No translations were queued. Please check the logs for details.', 'latw-ai-translator-for-wpml'));
            }

            if ($queued_count === 0 && $queue_full) {
                throw new Exception(__('Too many translations in progress(Max 50). Please wait.', 'latw-ai-translator-for-wpml'));
            }

            $message = sprintf(
                /* translators: %d: Number of translations submitted */
                __('%d background translations submitted. Check status in a few minutes.', 'latw-ai-translator-for-wpml'),
                $queued_count
            );

            if ($queue_full) {
                $message .= ' ' . __('Some translations were skipped because the queue is full(Max 50). Please wait.', 'latw-ai-translator-for-wpml');
            }

            wp_send_json_success(array(
                'message' => $message,
                'queued_count' => $queued_count,
                'queue_full' => $queue_full,
            ));

        } catch (Exception $e) {
            if ($this->logger) {
                $this->logger->error('LATW AJAX - Error: ' . $e->getMessage());
            }
            wp_send_json_error(array(
                'message' => $e->getMessage(),
            ));
        } catch (Throwable $e) {
            // Catch ALL errors including Fatal Errors (PHP 7+)
            if ($this->logger) {
                $this->logger->error('LATW AJAX - Fatal Error: ' . $e->getMessage());
            }
            wp_send_json_error(array(
                'message' => 'An unexpected error occurred: ' . $e->getMessage(),
            ));
        }
    }

    /**
     * Queue term translations (asynchronous)
     */
    public function queue_term_translations() {
        try {
            $this->logger->info("queue_term_translations");

            // Verify nonce
            if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'latwaitr_action')) {
                throw new Exception(__('Security check failed', 'latw-ai-translator-for-wpml'));
            }

            // Check permission
            if (!current_user_can('manage_options')) {
                throw new Exception(__('Permission denied', 'latw-ai-translator-for-wpml'));
            }

            // Check OpenAI API key
            $api_key = get_option('latwaitr_openai_api_key');
            if (empty($api_key)) {
                throw new Exception(__('OpenAI API Key is not configured. Please add your API key in the settings.', 'latw-ai-translator-for-wpml'));
            }

            // Get parameters - nonce already verified above
            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above
            $term_ids = isset($_POST['term_ids']) ? array_map('intval', (array) wp_unslash($_POST['term_ids'])) : array();
            // FREE version: always translate to English only
            $target_languages = array('en');

            if (empty($term_ids)) {
                throw new Exception(__('Invalid parameters', 'latw-ai-translator-for-wpml'));
            }

            $queue = LATWAITR()->queue;
            $queued_count = 0;
            $queue_full = false;

            $this->logger->info('LATW AJAX - queue_term_translations called with term_ids: ' . print_r($term_ids, true));
            $this->logger->info('LATW AJAX - target_languages: ' . print_r($target_languages, true));

            foreach ($term_ids as $term_id) {
                if ($queue_full) {
                    break;
                }

                foreach ($target_languages as $target_lang) {
                    $action_id = $queue->queue_term_translation($term_id, $target_lang);

                    if (is_wp_error($action_id)) {
                        $error_message = $action_id->get_error_message();
                        $error_code = $action_id->get_error_code();
                        $this->logger->error('LATW AJAX - Error queueing term ' . $term_id . ' -> ' . $target_lang . ': [' . $error_code . '] ' . $error_message);

                        if ($error_code === 'queue_full') {
                            $queue_full = true;
                            break;
                        }

                        continue;
                    }

                    $queued_count++;
                }
            }

            $this->logger->info('LATW AJAX - Total term translations queued: ' . $queued_count . ', queue_full: ' . ($queue_full ? 'yes' : 'no'));

            if ($queued_count === 0 && !$queue_full) {
                throw new Exception(__('No term translations were queued. Please check the logs for details.', 'latw-ai-translator-for-wpml'));
            }

            if ($queued_count === 0 && $queue_full) {
                throw new Exception(__('Too many translations in progress(Max 50). Please wait.', 'latw-ai-translator-for-wpml'));
            }

            $message = sprintf(
                /* translators: %d: Number of translations submitted */
                __('%d background term translations submitted. Check status in a few minutes.', 'latw-ai-translator-for-wpml'),
                $queued_count
            );

            if ($queue_full) {
                $message .= ' ' . __('Some translations were skipped because the queue is full(Max 50). Please wait.', 'latw-ai-translator-for-wpml');
            }

            wp_send_json_success(array(
                'message' => $message,
                'queued_count' => $queued_count,
                'queue_full' => $queue_full,
            ));

        } catch (Exception $e) {
            if ($this->logger) {
                $this->logger->error('LATW AJAX - Term translation error: ' . $e->getMessage());
            }
            wp_send_json_error(array(
                'message' => $e->getMessage(),
            ));
        } catch (Throwable $e) {
            if ($this->logger) {
                $this->logger->error('LATW AJAX - Term translation Fatal Error: ' . $e->getMessage());
            }
            wp_send_json_error(array(
                'message' => 'An unexpected error occurred: ' . $e->getMessage(),
            ));
        }
    }

    /**
     * Manually check pending responses (for servers without working cron)
     */
    public function check_responses() {
        try {
            // Verify nonce
            if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'latwaitr_action')) {
                throw new Exception(__('Security check failed', 'latw-ai-translator-for-wpml'));
            }

            // Check permission
            if (!current_user_can('manage_options')) {
                throw new Exception(__('Permission denied', 'latw-ai-translator-for-wpml'));
            }

            $this->logger->info('Manual check responses triggered by user');

            $response_checker = LATWAITR()->response_checker;
            $response_checker->check_all_pending_responses();

            // Get current pending count after check
            $pending_statuses = array('queued', 'in_progress');
            $still_pending = LATWAITR_Database::get_translations_by_response_status($pending_statuses, 1000);
            $pending_count = count($still_pending);

            $message = $pending_count > 0
                ? sprintf(
                    __('Check completed. %d response(s) still pending.', 'latw-ai-translator-for-wpml'),
                    $pending_count
                )
                : __('Check completed. No pending responses.', 'latw-ai-translator-for-wpml');

            wp_send_json_success(array(
                'message' => $message,
                'pending_count' => $pending_count,
            ));

        } catch (Exception $e) {
            if ($this->logger) {
                $this->logger->error('Manual check responses error: ' . $e->getMessage());
            }
            wp_send_json_error(array(
                'message' => $e->getMessage(),
            ));
        } catch (Throwable $e) {
            if ($this->logger) {
                $this->logger->error('Manual check responses fatal error: ' . $e->getMessage());
            }
            wp_send_json_error(array(
                'message' => 'An unexpected error occurred: ' . $e->getMessage(),
            ));
        }
    }

    /**
     * Get translation status
     */
    public function get_translation_status() {
        try {
            // Verify nonce
            if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'latwaitr_action')) {
                throw new Exception(__('Security check failed', 'latw-ai-translator-for-wpml'));
            }

            // Check permission
            if (!current_user_can('manage_options')) {
                throw new Exception(__('Permission denied', 'latw-ai-translator-for-wpml'));
            }

            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified above
            $translation_id = isset($_GET['translation_id']) ? intval(wp_unslash($_GET['translation_id'])) : 0;

            if (empty($translation_id)) {
                throw new Exception(__('Invalid translation ID', 'latw-ai-translator-for-wpml'));
            }

            $translation = LATWAITR_Database::get_translation($translation_id);

            if (!$translation) {
                throw new Exception(__('Translation not found', 'latw-ai-translator-for-wpml'));
            }

            wp_send_json_success($translation);

        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => $e->getMessage(),
            ));
        }
    }

    /**
     * Get statistics
     */
    public function get_stats() {
        try {
            // Verify nonce
            if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'latwaitr_action')) {
                throw new Exception(__('Security check failed', 'latw-ai-translator-for-wpml'));
            }

            // Check permission
            if (!current_user_can('manage_options')) {
                throw new Exception(__('Permission denied', 'latw-ai-translator-for-wpml'));
            }

            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce verified above
            $days = isset($_GET['days']) ? intval(wp_unslash($_GET['days'])) : 30;

            $stats = LATWAITR_Database::get_stats($days);

            wp_send_json_success($stats);

        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => $e->getMessage(),
            ));
        }
    }

    /**
     * Get posts with existing translations (for retranslation)
     */
    public function get_translated_terms() {
        try {
            if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'latwaitr_action')) {
                throw new Exception(__('Security check failed', 'latw-ai-translator-for-wpml'));
            }

            if (!current_user_can('manage_options')) {
                throw new Exception(__('Permission denied', 'latw-ai-translator-for-wpml'));
            }

            $admin = LATWAITR()->admin;
            $terms = $admin->get_terms_with_translations();
            $active_languages = LATWAITR()->wpml->get_active_languages();

            wp_send_json_success(array(
                'terms'            => $terms,
                'active_languages' => $active_languages,
            ));

        } catch (Exception $e) {
            $this->logger->error('LATW AJAX - get_translated_terms error: ' . $e->getMessage());
            wp_send_json_error(array('message' => $e->getMessage()));
        } catch (Throwable $e) {
            if ($this->logger) {
                $this->logger->error('LATW AJAX - get_translated_terms Fatal Error: ' . $e->getMessage());
            }
            wp_send_json_error(array('message' => 'An unexpected error occurred: ' . $e->getMessage()));
        }
    }

    public function get_translated_posts() {
        try {
            // Verify nonce
            if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'latwaitr_action')) {
                throw new Exception(__('Security check failed', 'latw-ai-translator-for-wpml'));
            }

            // Check permission
            if (!current_user_can('manage_options')) {
                throw new Exception(__('Permission denied', 'latw-ai-translator-for-wpml'));
            }

            // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above
            $post_type = isset($_POST['post_type']) ? sanitize_text_field(wp_unslash($_POST['post_type'])) : '';

            if (empty($post_type)) {
                throw new Exception(__('Invalid post type', 'latw-ai-translator-for-wpml'));
            }

            // Get posts with translations
            $admin = LATWAITR()->admin;
            $posts = $admin->get_posts_with_translations($post_type);

            // Get active languages for display
            $active_languages = LATWAITR()->wpml->get_active_languages();

            wp_send_json_success(array(
                'posts' => $posts,
                'active_languages' => $active_languages,
            ));

        } catch (Exception $e) {
            $this->logger->error('LATW AJAX - get_translated_posts error: ' . $e->getMessage());
            wp_send_json_error(array(
                'message' => $e->getMessage(),
            ));
        } catch (Throwable $e) {
            // Catch ALL errors including Fatal Errors (PHP 7+)
            if ($this->logger) {
                $this->logger->error('LATW AJAX - get_translated_posts Fatal Error: ' . $e->getMessage());
            }
            wp_send_json_error(array(
                'message' => 'An unexpected error occurred: ' . $e->getMessage(),
            ));
        }
    }

}
