<?php
/**
 * Settings Handler for LATW
 *
 * @package LATW_AI_Translator_for_WPML
 */

if (!defined('ABSPATH')) {
    exit;
}

class LATWAITR_Settings {

    /**
     * Security
     */
    private $security;

    /**
     * Constructor
     */
    public function __construct() {
        $this->security = LATWAITR()->security;

        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_post_latwaitr_save_settings', array($this, 'save_settings'));
    }

    /**
     * Register settings
     */
    public function register_settings() {
        register_setting('latwaitr_settings', 'latwaitr_openai_api_key', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('latwaitr_settings', 'latwaitr_openai_model', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('latwaitr_settings', 'latwaitr_website_description', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_textarea_field',
        ));
        register_setting('latwaitr_settings', 'latwaitr_target_languages', array(
            'type' => 'array',
            'sanitize_callback' => array($this, 'sanitize_target_languages'),
        ));
        register_setting('latwaitr_settings', 'latwaitr_translate_title', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('latwaitr_settings', 'latwaitr_translate_content', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('latwaitr_settings', 'latwaitr_translate_excerpt', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('latwaitr_settings', 'latwaitr_translate_custom_fields', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('latwaitr_settings', 'latwaitr_translate_seo', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('latwaitr_settings', 'latwaitr_translate_taxonomies', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('latwaitr_settings', 'latwaitr_translation_prompt', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_textarea_field',
        ));
        register_setting('latwaitr_settings', 'latwaitr_enable_telemetry', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('latwaitr_settings', 'latwaitr_debug_mode', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
        register_setting('latwaitr_settings', 'latwaitr_manual_check_responses', array(
            'type' => 'string',
            'sanitize_callback' => 'sanitize_text_field',
        ));
    }

    /**
     * Sanitize target languages array
     *
     * @param mixed $value The value to sanitize.
     * @return array Sanitized array of language codes.
     */
    public function sanitize_target_languages($value) {
        if (!is_array($value)) {
            return array();
        }
        return array_map('sanitize_text_field', $value);
    }

    /**
     * Save settings
     */
    public function save_settings() {
        check_admin_referer('latwaitr_settings');

        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Unauthorized', 'latw-ai-translator-for-wpml'));
        }

        // Get current tab to know which settings to save
        // phpcs:ignore WordPress.Security.NonceVerification.Missing -- Nonce verified above with check_admin_referer
        $current_tab = isset($_POST['current_tab']) ? sanitize_text_field(wp_unslash($_POST['current_tab'])) : 'general';

        // Save settings based on the current tab
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified above with check_admin_referer
        if ($current_tab === 'general') {
            // OpenAI API Key
            $latwaitr_api_key = isset($_POST['latwaitr_openai_api_key']) ? $this->security->sanitize_api_key(sanitize_text_field(wp_unslash($_POST['latwaitr_openai_api_key']))) : '';
            if (!empty($latwaitr_api_key)) {
                update_option('latwaitr_openai_api_key', $latwaitr_api_key);
            }

            // OpenAI Model
            $latwaitr_model = isset($_POST['latwaitr_openai_model']) ? sanitize_text_field(wp_unslash($_POST['latwaitr_openai_model'])) : '';
            $allowed_models = array('gpt-5-nano', 'gpt-5-mini', 'gpt-5');
            if (!empty($latwaitr_model) && in_array($latwaitr_model, $allowed_models, true)) {
                update_option('latwaitr_openai_model', $latwaitr_model);
            }

            // Website Description
            $latwaitr_description = isset($_POST['latwaitr_website_description']) ? sanitize_textarea_field(wp_unslash($_POST['latwaitr_website_description'])) : '';
            update_option('latwaitr_website_description', $latwaitr_description);

        } elseif ($current_tab === 'translation') {
            // Target languages
            $latwaitr_raw_languages = isset($_POST['latwaitr_target_languages']) ? array_map('sanitize_text_field', wp_unslash($_POST['latwaitr_target_languages'])) : array();
            $latwaitr_languages = is_array($latwaitr_raw_languages) ? $latwaitr_raw_languages : array();
            update_option('latwaitr_target_languages', $latwaitr_languages);

            // Translation options - these are checkboxes, just checking isset() for presence
            update_option('latwaitr_translate_title', isset($_POST['latwaitr_translate_title']) ? 'yes' : 'no');
            update_option('latwaitr_translate_content', isset($_POST['latwaitr_translate_content']) ? 'yes' : 'no');
            update_option('latwaitr_translate_excerpt', isset($_POST['latwaitr_translate_excerpt']) ? 'yes' : 'no');
            update_option('latwaitr_translate_custom_fields', isset($_POST['latwaitr_translate_custom_fields']) ? 'yes' : 'no');
            update_option('latwaitr_translate_seo', isset($_POST['latwaitr_translate_seo']) ? 'yes' : 'no');
            update_option('latwaitr_translate_taxonomies', isset($_POST['latwaitr_translate_taxonomies']) ? 'yes' : 'no');

            // Translation prompt
            $latwaitr_prompt = isset($_POST['latwaitr_translation_prompt']) ? sanitize_textarea_field(wp_unslash($_POST['latwaitr_translation_prompt'])) : '';
            update_option('latwaitr_translation_prompt', $latwaitr_prompt);
        } elseif ($current_tab === 'advanced') {
            update_option('latwaitr_debug_mode', isset($_POST['latwaitr_debug_mode']) ? 'yes' : 'no');
            update_option('latwaitr_manual_check_responses', isset($_POST['latwaitr_manual_check_responses']) ? 'yes' : 'no');
        }
        // phpcs:enable

        // Redirect back to the settings page with the current tab
        wp_safe_redirect(add_query_arg(array('updated' => 'true', 'tab' => $current_tab), admin_url('admin.php?page=latwaitr-settings')));
        exit;
    }

    /**
     * Render settings page
     */
    public function render() {
        $latwaitr_wpml = LATWAITR()->wpml;
        $latwaitr_languages = $latwaitr_wpml->get_active_languages();

        include LATWAITR_PLUGIN_DIR . 'admin/views/settings.php';
    }
}
