<?php

namespace LadderSEO;

/**
 * The file that defines the sitemap plugin logics
 *
 *
 * @link       https://digitaldreamstech.com
 * @since      1.0.0
 *
 * @package    Ladder_SEO
 * @subpackage Ladder_SEO/src
 */
 
 class Sitemap extends Core{
    private $sitemap_dir;

    public function __construct() {
        $upload_dir = wp_upload_dir();
        $this->sitemap_dir = $upload_dir['basedir'] . '/sitemaps/';

        add_action('init', [$this, 'initialize_sitemaps']);
        add_action('save_post', [$this, 'update_sitemaps']);
        add_action('edit_term', [$this, 'update_sitemaps']);
        register_activation_hook(__FILE__, [$this, 'activate_plugin']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate_plugin']);
        add_filter('robots_txt', [$this, 'robot_txt'], 10, 2);
        
    }

    public function robot_txt($output, $public) {
        if ($public) {
            $upload_dir = wp_upload_dir();
            $sitemap = $upload_dir['basedir'] . '/sitemaps/sitemap.xml';
            $output .= "Sitemap: " . $sitemap . "\n";
        }
        return $output;
    }

    public function activate_plugin() {
        if (!file_exists($this->sitemap_dir)) {
            mkdir($this->sitemap_dir, 0755, true);
        }
        $this->generate_all_sitemaps();
    }

    public function deactivate_plugin() {
        if (file_exists($this->sitemap_dir)) {
            array_map('unlink', glob($this->sitemap_dir . "*.xml"));
            rmdir($this->sitemap_dir);
        }
    }

    public function initialize_sitemaps() {
        if (!file_exists($this->sitemap_dir)) {
            mkdir($this->sitemap_dir, 0755, true);
        }
    
        add_rewrite_rule('^sitemap\.xml$', 'index.php?ladderseoeo_sitemap=index', 'top');
        add_rewrite_rule('^sitemaps/(.+)\.xml$', 'index.php?ladderseoeo_sitemap=$matches[1]', 'top');
    
        add_filter('query_vars', function ($vars) {
            $vars[] = 'ladderseoeo_sitemap';
            return $vars;
        });
    
        add_action('template_redirect', [$this, 'render_sitemap']);
    }
    
    

    public function render_sitemap() {
        $sitemap = get_query_var('ladderseoeo_sitemap');
        if (!$sitemap) {
            return;
        }

        $filepath = $this->sitemap_dir . sanitize_file_name($sitemap) . '.xml';
        if (file_exists($filepath)) {
            header('Content-Type: application/xml; charset=utf-8');
            readfile($filepath);
            exit;
        }
        wp_die('Sitemap not found.', 404);
    }

    public function update_sitemaps() {
        $this->generate_all_sitemaps();
        $this->generate_index_sitemap();
    }
 
     private function generate_all_sitemaps() {

        try{
            $this->generate_post_type_sitemaps();
            $this->generate_category_sitemap();
            $this->generate_image_sitemap();
            $this->generate_google_news_sitemap();
            $this->generate_multilingual_sitemap();
            $this->generate_index_sitemap();   
        }
        catch (Exception $e) {
            Ladder_SEO_Logger::log_error($e->getMessage());
        }
     }

     private function generate_index_sitemap() {
        $sitemaps = array_merge(
            array_map(fn($post_type) => "{$post_type}.xml", array_keys(get_post_types(['public' => true]))),
            ['categories.xml', 'images.xml', 'news.xml', 'multilingual.xml']
        );
    
        $home_url = home_url('/');
        $xml = '<?xml version="1.0" encoding="UTF-8"?>';
        $xml .= '<sitemapindex xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';
    
        foreach ($sitemaps as $sitemap) {
            $sitemap_path = $this->sitemap_dir . $sitemap;
            if (!file_exists($sitemap_path)) {
                continue;
            }
    
            $xml .= '<sitemap>';
            $xml .= '<loc>' . esc_url($home_url .'sitemaps/'. $sitemap) . '</loc>';
            $xml .= '<lastmod>' . date(DATE_W3C, filemtime($sitemap_path)) . '</lastmod>';
            $xml .= '</sitemap>';
        }
    
        $xml .= '</sitemapindex>';

        $upload_dir = wp_upload_dir();
        file_put_contents($upload_dir['basedir'] . '/sitemap.xml', $xml);
        
    }
    
    
 
     private function generate_post_type_sitemaps() {
         $post_types = get_post_types(['public' => true], 'objects');
         foreach ($post_types as $post_type) {
             $posts = get_posts(['post_type' => $post_type->name, 'numberposts' => -1]);
             $this->create_sitemap_file($post_type->name . '.xml', $posts, $post_type->name);
         }
     }
 
     private function generate_category_sitemap() {
         $categories = get_categories(['hide_empty' => false]);
         $this->create_category_sitemap_file('categories.xml', $categories);
     }
 
     private function generate_image_sitemap() {
         $posts = get_posts(['post_type' => 'any', 'numberposts' => -1]);
         $xml = '<?xml version="1.0" encoding="UTF-8"?>';
         $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:image="http://www.google.com/schemas/sitemap-image/1.1">';
         foreach ($posts as $post) {
             $images = $this->get_images_from_post($post->ID);
             if (!$images) {
                 continue;
             }
             $xml .= '<url>';
             $xml .= '<loc>' . get_permalink($post->ID) . '</loc>';
             foreach ($images as $image) {
                 $xml .= '<image:image>';
                 $xml .= '<image:loc>' . esc_url($image) . '</image:loc>';
                 $xml .= '</image:image>';
             }
             $xml .= '</url>';
         }
         $xml .= '</urlset>';
         file_put_contents($this->sitemap_dir . 'images.xml', $xml);
     }
 
     private function generate_google_news_sitemap() {
         $posts = get_posts([
             'post_type' => 'post',
             'date_query' => [
                 'after' => date('Y-m-d', strtotime('-48 hours'))
             ],
             'numberposts' => -1
         ]);
         $xml = '<?xml version="1.0" encoding="UTF-8"?>';
         $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:news="http://www.google.com/schemas/sitemap-news/0.9">';
         foreach ($posts as $post) {
             $xml .= '<url>';
             $xml .= '<loc>' . get_permalink($post->ID) . '</loc>';
             $xml .= '<news:news>';
             $xml .= '<news:publication>';
             $xml .= '<news:name>' . get_bloginfo('name') . '</news:name>';
             $xml .= '<news:language>' . get_locale() . '</news:language>';
             $xml .= '</news:publication>';
             $xml .= '<news:publication_date>' . get_the_date(DATE_W3C, $post->ID) . '</news:publication_date>';
             $xml .= '<news:title>' . esc_html($post->post_title) . '</news:title>';
             $xml .= '</news:news>';
             $xml .= '</url>';
         }
         $xml .= '</urlset>';
         file_put_contents($this->sitemap_dir . 'news.xml', $xml);
     }
 
     private function generate_multilingual_sitemap() {
        if (!function_exists('pll_get_post_translations')) {
            return; // Requires Polylang or WPML
        }
        
        $posts = get_posts(['post_type' => 'any', 'numberposts' => -1]);
        if (empty($posts)) {
            return;
        }
    
        $xml = '<?xml version="1.0" encoding="UTF-8"?>';
        $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9" xmlns:xhtml="http://www.w3.org/1999/xhtml">';
    
        foreach ($posts as $post) {
            $translations = pll_get_post_translations($post->ID);
            if (empty($translations)) continue;
    
            $xml .= '<url>';
            $xml .= '<loc>' . esc_url(get_permalink($post->ID)) . '</loc>';
            foreach ($translations as $lang => $translation_id) {
                $xml .= '<xhtml:link rel="alternate" hreflang="' . esc_attr($lang) . '" href="' . esc_url(get_permalink($translation_id)) . '" />';
            }
            $xml .= '</url>';
        }
    
        $xml .= '</urlset>';
        file_put_contents($this->sitemap_dir . 'multilingual.xml', $xml);
    }
    
 
 
     private function create_sitemap_file($filename, $items, $type) {
         $xml = '<?xml version="1.0" encoding="UTF-8"?>';
         $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';
         foreach ($items as $item) {
             $xml .= '<url>';
             $xml .= '<loc>' . get_permalink($item->ID) . '</loc>';
             $xml .= '<lastmod>' . get_post_modified_time(DATE_W3C, false, $item->ID) . '</lastmod>';
             $xml .= '<changefreq>' . $this->calculate_changefreq($item) . '</changefreq>';
             $xml .= '<priority>' . $this->calculate_priority($item, $type) . '</priority>';
             $xml .= '</url>';
         }
         $xml .= '</urlset>';
         file_put_contents($this->sitemap_dir . $filename, $xml);
     }
 
     private function create_category_sitemap_file($filename, $categories) {
         $xml = '<?xml version="1.0" encoding="UTF-8"?>';
         $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">';
         foreach ($categories as $category) {
             $xml .= '<url>';
             $xml .= '<loc>' . get_category_link($category->term_id) . '</loc>';
             $xml .= '<lastmod>' . date(DATE_W3C) . '</lastmod>';
             $xml .= '<changefreq>' . $this->calculate_changefreq($category) . '</changefreq>';
             $xml .= '<priority>0.5</priority>';
             $xml .= '</url>';
         }
         $xml .= '</urlset>';
         file_put_contents($this->sitemap_dir . $filename, $xml);
     }
 
     private function calculate_changefreq($item) {
        if ($item instanceof WP_Post) {
            $days_since_last_mod = (time() - strtotime($item->post_modified)) / 86400;
            if ($days_since_last_mod <= 7) return 'daily';
            if ($days_since_last_mod <= 30) return 'weekly';
            return 'monthly';
        } elseif ($item instanceof WP_Term) {
            return 'monthly'; // Categories change less frequently
        }
        return 'monthly';
     }
 
     private function calculate_priority($item, $type) {
         return ($type === 'page') ? '1.0' : '0.8';
     }
 
     private function get_images_from_post($post_id) {
         $images = [];
         $content = get_post_field('post_content', $post_id);
         preg_match_all('/<img[^>]+src=["\']([^"\']+)["\']/', $content, $matches);
         if (!empty($matches[1])) {
             $images = array_unique($matches[1]);
         }
         return $images;
     }
 } 