<?php

namespace LadderSEO;
class Schema extends Core
{
    private $context = "https://schema.org";

    public function __construct()
    {
        parent::__construct();
    }

    public function add_actions()
    {
        add_action('wp_head', [$this, 'generate_schema']);
    }

    public function generate_schema()
    {
        $schema = $this->get_schema_for_page();
        if ($schema) {
            echo '<script type="application/ld+json">' . esc_html(wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT)) . '</script>';
        }
    }

    private function get_schema_for_page()
    {
        if (is_front_page() || is_home()) {
            return $this->get_home_schema();
        }
        else if (is_single() && 'post' === get_post_type()) {
            return $this->get_article_schema();
        } elseif (is_page_template('page-about.php')) {
            return $this->get_about_schema();
        } elseif (function_exists('is_product') && is_product()) {
            return $this->get_product_schema();
        } elseif (is_page_template('page-contact.php')) {
            return $this->get_contact_schema();
        } elseif (is_page_template('page-faq.php')) {
            return $this->get_faq_schema();
        } elseif (is_page_template('page-qa.php')) {
            return $this->get_qa_schema();
        } elseif (is_page_template('page-real-estate.php')) {
            return $this->get_real_estate_schema();
        } elseif (is_page('checkout')) {
            return $this->get_checkout_schema();
        } elseif (is_page('cart')) {
            return $this->get_cart_schema();
        } elseif (is_page('sales')) {
            return $this->get_sales_schema();
        } elseif (is_page_template('page-profile.php')) {
            return $this->get_profile_schema();
        } else {
            return $this->get_webpage_schema();
        }

        // Always include Organization schema as part of the output
        if ($page_schema) {
            $organization_schema = $this->get_organization_schema();
            return array_merge($organization_schema, ["mainEntity" => $page_schema]);
        }

        return $this->get_organization_schema();
    }

    private function get_home_schema(){
        $settings = get_option(LADDER_SEO_SETTINGS);       
        $schema_type = $settings["global"]["knowledge_graph"]["person_or_organization"];
        $schema_type = get_option('ladderseop_schema_type', 'Organization');

        if ($schema_type === 'Person') {
            $schema_data = [
                '@context' => 'https://schema.org',
                '@type'    => 'Person',
                'name'     => $settings["global"]["knowledge_graph"]["person_or_organization"],
                'sameAs'   => $this->ladderseop_get_social_links()
            ];
        } else {
            $organization_name = $this->ladderseo_parse_data($settings["global"]["knowledge_graph"]["organization_name"]);
            $alternate_organization_name = $this->ladderseo_parse_data($settings["global"]["knowledge_graph"]["alternate_organization_name"]);
            $organization_description = $this->ladderseo_parse_data($settings["global"]["knowledge_graph"]["organization_description"]);

            $schema_data = [
                '@context'    => 'https://schema.org',
                '@type'       => 'Organization',
                'name'        => $organization_name,
                'alternateName' => $alternate_organization_name,
                'description' => $organization_description,
                'address'     => $settings["global"]["knowledge_graph"]["organization_address"],
                'contactPoint' => [
                    '@type'  => 'ContactPoint',
                    'contactType' => 'customer service',
                    'name'   => $settings["global"]["knowledge_graph"]["contact_person"],
                    'email'  => $settings["global"]["knowledge_graph"]["email"],
                    'telephone' => $settings["global"]["knowledge_graph"]["phone_number"]
                ],
                'openingHours' => $settings["global"]["knowledge_graph"]["opening_hours"],
                'closingHours' => $settings["global"]["knowledge_graph"]["closing_hours"],
                'foundingDate' => $settings["global"]["knowledge_graph"]["founding_date"],
                'numberOfEmployees' => isset($settings["global"]["knowledge_graph"]["no_of_employees"])? : 0,
                'logo'        => $settings["global"]["knowledge_graph"]["organization_logo"],
                'sameAs'      => $this->ladderseop_get_social_links($settings)
            ];
        }
    
        echo '<script type="application/ld+json">' . esc_html(wp_json_encode($schema_data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT)) . '</script>';
    }

    private function get_article_schema()
    {
        $post_id = get_the_ID();
        return [
            "@context" => $this->context,
            "@type" => "Article",
            "headline" => get_the_title($post_id),
            "author" => [
                "@type" => "Person",
                "name" => get_the_author_meta('display_name', get_post_field('post_author', $post_id))
            ],
            "datePublished" => get_the_date('c', $post_id),
            "dateModified" => get_the_modified_date('c', $post_id),
            "description" => get_the_excerpt($post_id),
            "mainEntityOfPage" => [
                "@type" => "WebPage",
                "@id" => get_permalink($post_id)
            ],
            "image" => [
                "@type" => "ImageObject",
                "url" => get_the_post_thumbnail_url($post_id, 'full') ?: 'https://example.com/default-image.jpg'
            ],
        ];
    }

    private function get_webpage_schema()
    {
        return [
            "@context" => $this->context,
            "@type" => "WebPage",
            "name" => get_bloginfo('name'),
            "description" => get_bloginfo('description'),
            "url" => home_url(),
        ];
    }

    private function get_product_schema()
    {
        global $product;
        if (!$product) {
            return null;
        }

        return [
            "@context" => $this->context,
            "@type" => "Product",
            "name" => $product->get_name(),
            "image" => wp_get_attachment_url($product->get_image_id()),
            "description" => $product->get_description(),
            "brand" => [
                "@type" => "Brand",
                "name" => $product->get_attribute('brand') ?: 'Generic'
            ],
            "offers" => [
                "@type" => "Offer",
                "price" => $product->get_price(),
                "priceCurrency" => get_woocommerce_currency(),
                "availability" => "https://schema.org/" . ($product->is_in_stock() ? "InStock" : "OutOfStock"),
                "url" => get_permalink($product->get_id())
            ],
        ];
    }

    private function get_contact_schema()
    {
        return [
            "@context" => $this->context,
            "@type" => "ContactPage",
            "name" => get_the_title(),
            "url" => get_permalink(),
        ];
    }

    private function get_faq_schema()
    {
        $faq_data = get_field('faq_items'); // Assuming ACF or custom field for FAQ
        if (!$faq_data) {
            return null;
        }

        $questions = array_map(function ($item) {
            return [
                "@type" => "Question",
                "name" => $item['question'],
                "acceptedAnswer" => [
                    "@type" => "Answer",
                    "text" => $item['answer']
                ]
            ];
        }, $faq_data);

        return [
            "@context" => $this->context,
            "@type" => "FAQPage",
            "mainEntity" => $questions
        ];
    }

    private function get_qa_schema()
    {
        // Similar to FAQ schema but uses "QAPage" type
        return [
            "@context" => $this->context,
            "@type" => "QAPage",
            "mainEntity" => [
                "@type" => "Question",
                "name" => get_the_title(),
                "acceptedAnswer" => [
                    "@type" => "Answer",
                    "text" => get_field('accepted_answer') ?: 'No answer yet.'
                ]
            ]
        ];
    }

    private function get_about_schema()
    {
        return [
            "@context" => $this->context,
            "@type" => "AboutPage",
            "name" => get_the_title(),
            "url" => get_permalink(),
        ];
    }

    private function get_real_estate_schema()
    {
        // Example real estate schema structure
        return [
            "@context" => $this->context,
            "@type" => "RealEstateListing",
            "name" => get_the_title(),
            "url" => get_permalink(),
            "description" => get_the_excerpt(),
        ];
    }

    private function get_checkout_schema()
    {
        return [
            "@context" => $this->context,
            "@type" => "CheckoutPage",
            "name" => "Checkout",
            "url" => get_permalink(),
        ];
    }

    private function get_cart_schema()
    {
        return [
            "@context" => $this->context,
            "@type" => "ShoppingCart",
            "name" => "Cart",
            "url" => get_permalink(),
        ];
    }

    private function get_sales_schema()
    {
        return [
            "@context" => $this->context,
            "@type" => "WebPage",
            "name" => "Sales Page",
            "url" => get_permalink(),
        ];
    }

    private function get_profile_schema()
    {
        return [
            "@context" => $this->context,
            "@type" => "ProfilePage",
            "name" => get_the_title(),
            "url" => get_permalink(),
        ];
    }

    private function get_organization_schema()
    {
        return [
            "@type" => "Organization",
            "name" => get_bloginfo('name'),
            "url" => home_url(),
            "logo" => get_theme_mod('custom_logo')
                ? wp_get_attachment_url(get_theme_mod('custom_logo'))
                : 'https://example.com/default-logo.png',
            "contactPoint" => [
                "@type" => "ContactPoint",
                "telephone" => "+1-800-555-5555", // Replace with your organization's phone number
                "contactType" => "Customer Service"
            ],
            "sameAs" => [
                "https://www.facebook.com/YourPage",
                "https://twitter.com/YourProfile",
                "https://www.instagram.com/YourProfile",
            ]
        ];
    }

}
