<?php

namespace LadderSEO;

/**
 * The file that defines the open_graph plugin logics
 *
 *
 * @link       https://digitaldreamstech.com
 * @since      1.0.0
 *
 * @package    Ladder_SEO
 * @subpackage Ladder_SEO/src
 */


class Open_Graph extends Core
{
    public function __construct()
    {
        parent::__construct();
    }

    public function get_og_properties()
    {
        $og_tags = [
            'og:title' => $this->get_title(),
            'og:type'  => $this->get_type(),
            'og:image' => $this->get_image(),
            'og:image:width'  => $this->get_image_width(),
            'og:image:height' => $this->get_image_height(),
            'og:image:alt'    => $this->get_image_alt(),
            'og:description'  => $this->get_description(),
            'og:url'          => $this->get_url(),
            'og:locale'       => $this->get_locale(),
            'og:site_name'    => $this->get_site_name(),
            'article:published_time' => $this->get_published_time(),
            'article:modified_time'  => $this->get_modified_time(),
            'og:updated_time'        => $this->get_updated_time(),
            'article:section'        => $this->get_article_section(),
            'article:tag'            => $this->get_article_tag()];
        return $og_tags;
    }

    public function add_actions()
    {
        add_action('wp_head', [ $this, 'open_graph_output' ]);
    }

    public function open_graph_output(){
        $og_tags = $this->get_og_properties();
        foreach ($og_tags as $property => $content) {
            if (!empty($content)) { // Ensure no empty tags are output
                echo '<meta property="' . esc_attr($property) . '" content="' . esc_attr($content) . '">' . "\n";
            }
        }
    }

    // Retrieves the title of the current page or post
    public function get_title()
    {
        return wp_get_document_title();
    }

    // Specifies the Open Graph type (e.g., website, article)
    public function get_type()
    {
        if (is_singular('post')) {
            return 'article';
        } elseif (is_front_page() || is_home()) {
            return 'website';
        }
        return 'website';
    }

    // Retrieves the featured image URL for the current post
    public function get_image()
    {
        if (has_post_thumbnail()) {
            return get_the_post_thumbnail_url(get_the_ID(), 'full');
        }
        return ''; // Default or fallback image URL can be set here
    }

    // Retrieves the width of the featured image
    public function get_image_width()
    {
        $image = wp_get_attachment_metadata(get_post_thumbnail_id());
        return $image ? $image['width'] : '';
    }

    // Retrieves the height of the featured image
    public function get_image_height()
    {
        $image = wp_get_attachment_metadata(get_post_thumbnail_id());
        return $image ? $image['height'] : '';
    }

    // Retrieves the alt text for the featured image
    public function get_image_alt()
    {
        $thumbnail_id = get_post_thumbnail_id();
        return $thumbnail_id ? get_post_meta($thumbnail_id, '_wp_attachment_image_alt', true) : '';
    }

    // Retrieves the description of the current page or post
    public function get_description()
    {
        if (is_singular()) {
            global $post;
            $description = get_the_excerpt($post);
            if (!$description) {
                $description = wp_trim_words(strip_shortcodes($post->post_content), 30, '...');
            }
            return $description;
        }
        return get_bloginfo('description'); // Fallback to site description
    }

    // Retrieves the canonical URL of the current page or post
    public function get_url()
    {
        return get_permalink();
    }

    // Retrieves the locale of the site
    public function get_locale()
    {
        return get_locale();
    }

    // Retrieves the site name
    public function get_site_name()
    {
        return get_bloginfo('name');
    }

    // Retrieves the published time of the current post
    public function get_published_time()
    {
        if (is_singular('post')) {
            return get_the_time('c');
        }
        return '';
    }

    // Retrieves the last modified time of the current post
    public function get_modified_time()
    {
        if (is_singular('post')) {
            return get_the_modified_time('c');
        }
        return '';
    }

    // Retrieves the last updated time (alias for modified time)
    public function get_updated_time()
    {
        return $this->get_modified_time();
    }

    // Retrieves the primary category or section of the current post
    public function get_article_section()
    {
        if (is_singular('post')) {
            $category = get_the_category();
            return $category ? $category[0]->name : '';
        }
        return '';
    }

    // Retrieves the tags for the current post
    public function get_article_tag()
    {
        if (is_singular('post')) {
            $tags = get_the_tags();
            if ($tags) {
                return implode(', ', wp_list_pluck($tags, 'name'));
            }
        }
        return '';
    }
}