<?php

namespace LadderSEO;

/**
 * The file that defines the meta_robots plugin logics
 *
 *
 * @link       https://digitaldreamstech.com
 * @since      1.0.0
 *
 * @package    Ladder_SEO
 * @subpackage Ladder_SEO/src
 */


class Meta_Robots extends Core
{
    public function __construct()
    {
        parent::__construct();
    }


    public function add_actions()
    {
        add_filter('robots_txt', [$this, 'update_robots_txt'], 10, 2);
        add_filter('wp_robots', [$this, 'modify_wp_robots']);
    }

    /**
     * Update robots.txt dynamically to prevent indexing of hidden posts/pages.
     */
    public function update_robots_txt($output, $public) {
        if ($public === '0') {
            return $output;
        }

        $disallow_rules = [];

        // Get all posts that should be hidden from search
        $args = [
            'post_type'      => 'any',
            'post_status'    => 'publish',
            'meta_key'       => 'ladderseo_hide_from_result',
            'meta_value'     => '1',
            'posts_per_page' => -1
        ];

        $hidden_posts = get_posts($args);

        foreach ($hidden_posts as $post) {
            $disallow_rules[] = "Disallow: " . wp_parse_url(get_permalink($post->ID), PHP_URL_PATH);
        }

        // Default disallow rules for search pages
        $disallow_rules[] = "Disallow: /?s=";
        $disallow_rules[] = "Disallow: /search/";

        // Allow everything else
        $output .= "\n" . implode("\n", $disallow_rules) . "\n";
        $output .= "Allow: /\n";

        return $output;
    }

    /**
     * Modify meta robots for individual pages.
     */
    public function modify_wp_robots($robots) {
        if ($this->should_noindex()) {
            $this->remove_canonical_link();
            return ['noindex' => true, 'follow' => true];
        }

        // Set default indexing rules
        $robots['max-snippet'] = '-1';
        $robots['max-video-preview'] = '-1';

        return $robots;
    }

    /**
     * Determine if the current page should be noindexed.
     */
    private function should_noindex() {
        if (is_search() || is_404()) {
            return true;
        }

        if (is_singular()) {
            $post_id = get_queried_object_id();
            $noindex = get_post_meta($post_id, 'ladderseo_hide_from_result', true);
            if ($noindex === '1') {
                return true;
            }
        }

        return false;
    }

    /**
     * Remove canonical link if noindex is applied.
     */
    private function remove_canonical_link() {
        remove_action('wp_head', 'rel_canonical');
    }
}