<?php

namespace LadderSEO;

/**
 * The file that defines the core plugin functions
 *
 *
 * @link       https://digitaldreamstech.com
 * @since      1.0.0
 *
 * @package    Ladder_SEO
 * @subpackage Ladder_SEO/src
 */

class Core
{
    /**
     * Define the core functionality of the plugin.
     *
     *
     * @since    1.0.0
     */
    public function __construct()
    {
    }

    /**
     * Normalize a string by removing unnecessary characters and formatting it.
     *
     * @param string $alt The string to be normalized.
     * @return string The normalized string.
     */
    public function ladderseo_normalize($alt)
    {
        // Remove hyphens, underscores, and extra spaces, and normalize spacing.
        $alt = preg_replace('/\s*[-_\s]+\s*/', ' ', $alt);

        // Trim leading and trailing whitespace.
        $alt = trim($alt);

        // Capitalize the first letter of every word (other letters lower case).
        $alt = ucwords(strtolower($alt));

        return $alt;
    }

    /**
     * Retrieve available placeholders for title format parsing.
     *
     * @return array Associative array of placeholders with their callback functions.
     */
    /**
    * Retrieve available placeholders for title format parsing.
    *
    * @return array Associative array of placeholders with their callback functions.
    */
    public function ladderseo_get_placeholders()
    {
        $placeholders = [
            'site_title' => function () {
                return get_bloginfo('name');
            },
            'post_title' => function ($context) {
                return isset($context['post']) ? get_the_title($context['post']->ID) : '';
            },
            'post_description' => function ($context) {
                if (!isset($context['post'])) {
                    return '';
                }

                $content = $context['post']->post_content;
                $content = strip_shortcodes($content);
                $content = wp_strip_all_tags($content);
                $content = html_entity_decode($content, ENT_QUOTES, 'UTF-8');
                $content = preg_replace('/\s+/', ' ', $content);
                if (!empty($content)) {
                    return wp_trim_words($content, 30, '...');
                }
                // Fallback to site tagline
                return get_bloginfo('description');
            },
            'post_excerpt' => function ($context) {
                return isset($context['post']) ? wp_trim_words($context['post']->post_excerpt, 20) : '';
            },
            'author_name' => function ($context) {
                return isset($context['post']) ? get_the_author_meta('display_name', $context['post']->post_author) : '';
            },
            'author_first_name' => function ($context) {
                return isset($context['post']) ? get_the_author_meta('first_name', $context['post']->post_author) : '';
            },
            'author_last_name' => function ($context) {
                return isset($context['post']) ? get_the_author_meta('last_name', $context['post']->post_author) : '';
            },
            'categories' => function ($context) {
                if (!isset($context['post'])) {
                    return '';
                }
                $categories = get_the_category($context['post']->ID);
                return $categories ? implode(', ', wp_list_pluck($categories, 'name')) : '';
            },
            'current_month' => function () {
                return date_i18n('F');
            },
            'current_year' => function () {
                return date_i18n('Y');
            },
            'post_date' => function ($context) {
                return isset($context['post']) ? get_the_date('F j, Y', $context['post']->ID) : '';
            },
            'current_date' => function () {
                return date_i18n('F j, Y');
            },
            'current_day' => function () {
                return date_i18n('l');
            },
            'separator' => function () {
                return '|';
            },
            'tagline' => function () {
                return get_bloginfo('description');
            },
            'term_name' => function ($context) {
                return isset($context['term']) ? $context['term']->name : '';
            },
        ];

        // Allow developers to add custom placeholders via filter
        return apply_filters('ladderseo_placeholders', $placeholders);
    }


    /**
     * Parse the title format and replace placeholders with actual values.
     *
     * @param string $format The title format string, e.g., "{{post_title}} {{separator}} {{tagline}}".
     * @param array $context Contextual data to dynamically resolve placeholders.
     * @return string The processed title.
     */
    public function ladderseo_parse_data($format, $context = [])
    {
        $placeholders = $this->ladderseo_get_placeholders();

        // Find all placeholders in the format
        preg_match_all('/{{(.*?)}}/', $format, $matches);

        if (empty($matches[1])) {
            // No placeholders found, return the raw format
            return $format;
        }

        $replacements = [];

        foreach ($matches[1] as $placeholder) {
            if (isset($placeholders[$placeholder])) {
                // Call the callback to resolve the placeholder
                $replacements["{{{$placeholder}}}"] = $placeholders[$placeholder]($context);
            } else {
                // For undefined placeholders, replace with an empty string
                $replacements["{{{$placeholder}}}"] = '';
            }
        }

        // Replace placeholders with their resolved values
        return strtr($format, $replacements);
    }

    /**
     * Dynamically generate the meta title, meta description based on the page type.
     * @param string $type define the type of data to be generated, values are 'meta_title', 'meta_desc'
     *
     * @return string The generated meta title, meta description etc.
     */
    public function ladderseo_generate_dynamic_data($type)
    {
        global $post;

        $output = ''; // Default title

        // Check if it's a term archive page
        if (is_category() || is_tag() || is_tax()) {
            $term = get_queried_object();
            $format = $this->get_term_meta_data($term, $type);

            $output = $this->ladderseo_parse_data($format, [
                'term' => $term,
            ]);

            // Check if it's a single post or page
        } elseif (is_singular()) {
            $format = $this->get_post_meta_data($post, $type);
            $output = $this->ladderseo_parse_data($format, [
                'post' => $post,
            ]);
            // Check if it's a author page
        } elseif (is_author()) {

        }


        return $output;
    }


    private function get_post_meta_data($post, $field)
    {
        $settings = get_option(LADDER_SEO_SETTINGS);
        $post_type = $post->post_type;
        $id = $post->ID;
        $format = "";

        if (is_front_page() || is_home()) {
            if($field === "meta_title")
                return $settings["global"]["home"]["title"];
            else if($field === "meta_description")
                return $settings["global"]["home"]["description"];             
        }

        if (!empty(get_post_meta($id, "ladderseo_".$post_type."-".$field, true))) {
            $format = get_post_meta($id, "ladderseo_".$post_type."-".$field, true);
        } elseif (isset($settings["post_types"][$post_type][$field])) {
            $format = $settings["post_types"][$post_type][$field];
        } elseif (isset($settings["content_types"]["title_and_description"][$field])) {
            $format = $settings["content_types"]["title_and_description"][$field];
        } elseif (isset($settings["content_types"]["social"][$field])) {
            $format = $settings["content_types"]["social"][$field];
        }

        return $format;

    }

    private function get_term_meta_data($term, $field)
    {
        $term_type  = $term->taxonomy;
        $term_id = $term->term_id;
        $format = "";
        $settings = get_option(LADDER_SEO_SETTINGS);

        if (!empty(get_term_meta($id, "ladderseo_".$term_type."-".$field, true))) {
            $format = get_term_meta($id, "ladderseo_".$term_type."-".$field, true);
        } elseif (isset($settings["term_types"][$term_type][$field])) {
            $format = $settings["term_types"][$term_type][$field];
        } elseif (isset($settings["content_types"]["title_and_description"][$field])) {
            $format = $settings["content_types"]["title_and_description"][$field];
        } elseif (isset($settings["content_types"]["social"][$field])) {
            $format = $settings["content_types"]["social"][$field];
        }

        return $format;
    }

    public function ladderseop_get_social_links($settings) {
        
        $social_profiles = [
            'facebook'  => $settings["social"]["facebook_page_url"],
            'twitter'   => $settings["social"]["twitter_page_url"],
            'instagram' => $settings["social"]["instagram_url"],
            'linkedin'  => $settings["social"]["linkedIn_url"],
            'youtube'   => $settings["social"]["youtube_url"],
            'pinterest' => $settings["social"]["pinterest_url"],
            'tumblr' => $settings["social"]["tumblr_url"],
            'tiktok' => $settings["social"]["tiktok_url"],
            'yelp' => $settings["social"]["yelp_url"],
            'soundcloud' => $settings["social"]["soundcloud_url"],
            'wordpress' => $settings["social"]["wordpress_url"],
            'myspace' => $settings["social"]["myspace_url"],
            'wikipedia' => $settings["social"]["wikipedia_url"],
        ];
        return array_values(array_filter($social_profiles));
    }

}
