<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://digitaldreamstech.com
 * @since      1.0.0
 *
 * @package    Ladder_SEO
 * @subpackage Ladder_SEO/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Ladder_SEO
 * @subpackage Ladder_SEO/includes
 * @author     Pritesh Gupta <g.pritesh@gmail.com>
 */
class Ladder_SEO
{
    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      Ladder_SEO_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct()
    {
        if (defined('LADDER_SEO_VERSION')) {
            $this->version = LADDER_SEO_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'ladder-seo';

        $this->load_dependencies();
        $this->define_admin_hooks();
        $this->define_public_hooks();

    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies()
    {

        spl_autoload_register(array( $this, 'autoload' ));
        $this->loader = new Ladder_SEO_Loader();
    }

    /**
	 * Autoload classes for both namespaced and non-namespaced files.
	 *
	 * @param string $class Fully qualified class name or non-namespaced class name.
	 * @return void
	 */
    public function autoload($class) {
		$class = ltrim($class, '\\'); // Remove leading backslash for namespaced classes
		$file = '';
		$base_dir = LADDER_SEO_PLUGIN_DIR; // Base directory for the plugin
	
		if (strpos($class, 'LadderSEO\\') === 0) {
			// Handle namespaced classes
			$relative_class = substr($class, strlen('LadderSEO\\')); // Strip the namespace prefix
			$relative_class = str_replace('\\', '/', $relative_class);  // Convert namespace separators to directory structure
			$relative_class = str_replace('_', '-', $relative_class);  // Convert underscores to hyphens for file naming
			$file = $base_dir . '/src/' . 'class-' . strtolower($relative_class) . '.php'; // Include subfolder path
		} else {
			// Handle non-namespaced classes
			$class = strtolower($class); // Convert class name to lowercase
			$file = 'class-' . str_replace('_', '-', $class) . '.php';
	
			// Possible paths for non-namespaced classes
			$possible_paths = array(
				$base_dir . '/admin/',
				$base_dir . '/helper/',
				$base_dir . '/helper/html-elements/',
				$base_dir . '/includes/',
				$base_dir . '/public/',
			);
	
			// Search in possible paths for non-namespaced class
			foreach ($possible_paths as $path) {
				if (is_readable($path . $file)) {
					require_once $path . $file;
					return;
				}
			}
		}
	
		// Load file for namespaced classes
		if (is_readable($file)) {
			require_once $file;
		}
	}
	

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks()
    {

        $plugin_admin = new Ladder_SEO_Admin($this->get_plugin_name(), $this->get_version());

        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');
        $this->loader->add_action('admin_menu', $plugin_admin, 'admin_menu');
        $this->loader->add_action('wp_ajax_ladder_seo_admin_ajax', $plugin_admin, 'ladder_seo_admin_ajax');
        $this->loader->add_action('add_meta_boxes', $plugin_admin, 'ladder_seo_meta_box_add');
        $this->loader->add_action('save_post', $plugin_admin, 'ladder_seo_save_post');
        $this->loader->add_action('ladderseo_generate_content', $plugin_admin, 'ladder_seo_generate_content');
        $this->loader->add_filter('plugin_action_links_'.LADDER_SEO_PLUGIN_BASENAME_PATH,  $plugin_admin, 'ladder_seo_add_plugins_link');

    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks()
    {

        $plugin_public = new Ladder_SEO_Public($this->get_plugin_name(), $this->get_version());
        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_scripts');        
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run()
    {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name()
    {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    Ladder_SEO_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader()
    {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version()
    {
        return $this->version;
    }

}
