<?php

/**
 * Utility functions
 *
 * @link       https://ladder-seo.com
 * @since      1.0.0
 *
 *
 * @package    Ladder_SEO
 * @subpackage Ladder_SEO/includes
 */
class Ladder_SEO_Util
{
    /**
    * Get default collate
    *
    *
    * @since    1.0.0
    */
    public static function get_collate()
    {
        global $wpdb;
        return $wpdb->get_charset_collate();
    }

    /**
     * Get redirection document table name
     *
     *
     * @since    1.0.0
     */
    public static function get_redirection_table_name()
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'ladderseo_redirection';
        return $table_name;
    }

    /**
     * Get 404 logs able name
     *
     *
     * @since    1.0.0
     */
    public static function get_404_logs_table_name()
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'ladderseo_404_logs';
        return $table_name;
    }
    /**
    * Get field help text by the name of field
    *
    *
    * @since    1.0.0
    */

    public static function get_field_help_text($field_name)
    {
        $text = array();
        $text["meta_tags"] = "Automatically generate meta title tag. You don't need to configure meta title tag. We activate this feature by adding theme support for title-tag.";
        $text["meta_description"] = "Generate meta descriptions dynamically for your content. This helps improve click-through rates by providing relevant summaries for search results.";
        $text["meta_robots"] = "Control the meta robots tag settings to guide search engines on which pages to index or avoid. This ensures better search engine visibility management.";
        $text["open_graph"] = "Enable automatic Open Graph metadata generation for your pages, optimizing content for social media sharing and improving link previews.";
        $text["twitter"] = "Integrate Twitter Card metadata to enhance how your content appears when shared on Twitter, driving more engagement and visibility.";
        $text["canonical_url"] = "It helps search engines understand the preferred version of a page when duplicate or similar content exists. Set canonical URLs automatically to prevent duplicate content issues and consolidate ranking signals for better SEO results. You can set ";
        $text["rel"] = "It helps search engines and browsers understand how to treat the link. Add 'rel' attributes such as 'nofollow' or 'noopener' where necessary, improving link control and security for outbound links.";
        $text["sitemaps"] = "Automatically generate XML sitemaps for your site, ensuring search engines can crawl and index your pages effectively.  It's essential for improving site visibility and SEO performance";
        $text["image_alt"] = "Automatically add 'alt' text to images to boost SEO and improve accessibility, making your content more search-friendly. Our plugin will generate the most effective alt tags for image based on the image name.";
        $text["image_title"] = "Generate image titles to enhance the SEO of visual content, supporting better image indexing by search engines.  Our plugin will generate the most effective title for image based on the image name.";
        $text["breadcrumbs"] = "Add breadcrumb navigation to improve site usability and help search engines understand your site structure more clearly. Breadcrumbs are navigational links that show users their current location within your site’s hierarchy";
        $text["rss_feed"] = "Ensure your RSS feed is optimized for search engines to keep your subscribers updated with fresh content and boost engagement.";
        $text["redirection"] = "Manage and automate 301 redirects to ensure users and search engines are directed to the correct pages, reducing 404 errors. 301 and 400 handling is important in SEO and site management:";
        $text["webmaster"] = "Webmaster tools allows you to add meta tags to verify the ownership of your website while submitting on various search engine. It's for verification purpose.";
        $text["analytics"] = "Analytics tools allows you to add various search engine analytics code for systematic computational analysis of data or statistics. You can integrate various analytics tools using analytics page.";
        $text["schema"] = "Schema markup helps search engines understand the content of your pages better and display relevant information in an enhanced way in search results.";
        $text["w3c"] = "W3C validation is crucial for SEO because it ensures that your website adheres to web standards, which helps search engines properly crawl and index your site";
        $text["optimization"] = "Optimize your site’s efficiency and prevent unnecessary search engine crawls and eliminating unused WordPress features.";
        $text["bulk_AI"] = "Enable AI-powered content generation settings to optimize your SEO data effortlessly. Adjust preferences to enhance meta title, meta descriptions, and generate high-quality seo content with a single click.";

        $text["title"] = "Set SEO title for home page. Click on the tags or press # to insert variables into your website title. Recommended length: ≤ 60 characters";
        $text["description"] = "Set SEO meta description for home page. Click on the tags or press # to insert variables into your meta description. Recommended length: 50-160 characters.";

        $text["image"] = "Default image for the website. This image is used as a fallback for posts/pages that don't have any images set.";
        $text["facebook_image"] = "Default image for the facebook. Should have 1.91:1 aspect ratio with width ≥ 600 px.";
        $text["twitter_image"] = "Default image for the twitter. Should have 2:1 aspect ratio with width ≥ 600 px.";
        $text["title_seperator"] = "Default separator to be used to separate title to tagline etc.";
        $text["website_name"] = "A name that Google may use for your homepage in mobile search results. This will default to the website site title if left blank.";
        $text["alternate_title"] = "An alternate name for your site. This could be an acronym or shorter version of your website name.";
        $text["person_or_organization"] = "Choose whether your site represents an organization or a person";
        $text["chose_a_person"] = "Choose person from WordPress user list that represents your website";
        $text["organization_name"] = "Please tell us more about your organization. This information will help Google to understand your website, and improve your chance of getting rich results.";
        $text["alternate_organization_name"] = "Use the alternate organization name for acronyms, or a shorter version of your organization's name.";
        $text["organization_description"] = "Enrich your organization's profile by providing more in-depth information. The more details you share, the better Google understands your website.";
        $text["organization_address"] = "This information will be used for local SEO";
        $text["contact_person"] = "Details about the organization's contact person";
        $text["opening_hours"] = "Set organization's opening hours";
        $text["closing_hours"] = "Set organization's closing hours";
        $text["working_days"] = "Weekly working days of your organization";
        $text["email"] = "Organization email address";
        $text["phone_number"] = "Organization phone number";
        $text["founding_date"] = "Organization founding date";
        $text["no_of_employees"] = "Set number of employees working in your organization.";
        $text["organization_logo"] = "Organization logo";

        $text["google"] = "Add your Google verification code.";
        $text["bing"] = "Add your Bing verification code.";
        $text["yandex"] = "Add your Yandex verification code.";
        $text["baidu"] = "Add your Baidu verification code.";
        $text["pinterest"] = "Add your Pinterest verification code.";

        $text["google_app_id"] = "Add your Google analytics app id. No need to enter complete analytics code, only app id is required.";
        $text["bing_app_id"] = "Add your Bing analytics app id. No need to enter complete analytics code, only app id is required.";
        $text["yandex_app_id"] = "Add your Yandex analytics app id. No need to enter complete analytics code, only app id is required.";
        $text["mixpanel_app_id"] = "Add your Baidu analytics app id. No need to enter complete analytics code, only app id is required.";
        $text["baidu_app_id"] = "Add your Pinterest analytics app id. No need to enter complete analytics code, only app id is required.";

        $text["type_attributes"] = "Remove unnecessary type attributes from script and style tags generated by WordPress, thereby addressing potential validation issues highlighted by the W3C validator tool.";
        $text["aria_var"] = "This enables the auto generation of aria variables. The aria-label attribute provides a way to place a descriptive text label on an object, such as a link,";
        $text["duplicate_id"] = "If there is an duplicate id of some element, it will automatically correct by adding some unique prefix in the id of element";
        $text["doctype"] = "By ensuring the <!DOCTYPE> declaration is consistently present, your plugin will adhere to web standards, preventing quirks mode and improving compatibility with browsers";

        $text["shortlinks"] = "If you enable this, it will remove links to internal 'shortlink' URLs for posts. e.g. <link rel='shortlink' href='https://www.website.com/?p=1' />";
        $text["rest_api"] = "Remove links to the location of your websites' REST API endpoints. <link rel='https://api.w.org/' href='https://www.website.com/wp-json/' />";
        $text["external_system"] = "Remove links used by external systems for publishing content to your blog.<link rel='EditURI' type='application/rsd+xml' title='RSD' href='https://www.website.com/xmlrpc.php?rsd' /> ";
        $text["generator_tag"] = "Remove information about the plugins and software used by your site.<meta name='generator' content='WordPress 6.0.1' />";
        $text["ping_httpheader"] = "Remove links which allow others sites to ‘ping’ yours when they link to you. E.g.,X-Pingback: https://www.website.com/xmlrpc.php";
        $text["powered_httpheader"] = "Remove information about the plugins and software used by your website. E.g.,X-Powered-By: PHP/7.4.1 will be removed from output.";

        $text["generate_content"] = "Whether to generate the bulk content using AI or not. AI needs to be activated as content will be generated using AI"; //whether to generate bulk content or not
		$text["topic"] = "Write brief description about the article, posts, page you want to be generated"; //topic to be used for content generation
        $text["keywords"] = "Write list of keywords in comma seperated format. AI will use these keywords in article to be generated"; //topic to be used for content generation
		$text["categories"] = "Select category to be used for article generation. This category will be sent to AI along the topic to generate content."; //list of categories
		$text["post_type"] = "Select post type e.g post, page, custom posts etc to be generated using this feature. You can use this to generate blog with specific post type"; //post, page etc.
        $text["post_status"] = "Set post status of generated post. You can set 'publish' to immediately publish the posts or you can use 'draft' to check the posts before publishing."; //post, page etc.
		$text["total_articles"] = "Total number of article to be generated. Max no. article that can be generated must be less than 10."; //total articles to be generated
		$text["duration"] = "Select duration after which you want to generate the articles. If you want to generate articles immediately, select now else you can also schedule the generation of articles."; //duration, now, each_day, each_week, each_month...default 'now'

        if (isset($text[$field_name])) {
            return $text[$field_name];
        }
        return "";
    }
    /**
    * Get schema types
    *
    *
    * @since    1.0.0
    */
    public static function get_schema_types()
    {
        $schemas = ["web" => esc_html__('Web Page', 'ladder-seo'),
                   "item" => esc_html__('Item Page', 'ladder-seo'),
                   "faq" => esc_html__('FAQ Page', 'ladder-seo'),
                   "about" => esc_html__('About Page', 'ladder-seo'),
                   "profile" => esc_html__('Profile Page', 'ladder-seo'),
                   "contact" => esc_html__('Contact Page', 'ladder-seo'),
                   "sales" => esc_html__('Sales Page', 'ladder-seo'),
                   "real_estate" => esc_html__('Real Estate Listing Page', 'ladder-seo'),
                   "qa" => esc_html__('QA Page', 'ladder-seo'),
                   "checkout" => esc_html__('Checkout Page', 'ladder-seo'),
                    ];
        return $schemas;
    }
    /**
    * Get field title by the name of field
    *
    *
    * @since    1.0.0
    */
    public static function get_field_title($field_name)
    {
        return ucwords(str_replace('_', ' ', $field_name));
    }

    /**
 * Truncate string to a specified length and append ellipses if needed.
 *
 * @param string $str The original string.
 * @param int $length The maximum allowed length (default: 30 characters).
 * @return string The truncated title.
 */
    public static function ladderseo_truncate_string($str, $length = 30)
    {
        // Ensure the string is not empty
        if (empty($str)) {
            return '';
        }

        // Trim the string if it exceeds the allowed length
        if (mb_strlen($str) > $length) {
            return mb_substr($str, 0, $length) . '...';
        }

        return $str;
    }

    /**
    * Get field title by the name of field
    *
    *
    * @since    1.0.0
    */
    public static function get_field_type($field_name)
    {
        $type = array();
        $type["title"] = "editor";
        $type["tagline"] = "editor";
        $type["description"] = "editor";
        $type["meta_title"] = "editor";
        $type["show_in_search_results"] = "radio";
        $type["meta_description"] = "editor";
        $type["title_seperator"] = "radio";
        $type["image"] = "image";
        $type["icon"] = "image";
        $type["facebook_image"] = "image";
        $type["twitter_image"] = "image";
        $type["website_name"] = "editor"; //website name
        $type["alternate_title"] = "input"; //website alternate title
        $type["person_or_organization"] = "radio"; //person or Organization
        $type["chose_a_person"] = "select"; //person id
        $type["organization_name"] = "input"; // Organization name
        $type["organization_description"] = "textarea"; // Organization description
        $type["organization_address"] = "textarea"; // Organization address
        $type["contact_person"] = "input"; // Organization contact person
        $type["opening_hours"] = "select"; // Organization opening hours
        $type["closing_hours"] = "select"; // Organization closing hours
        $type["email"] = "input"; // email
        $type["phone_number"] = "input"; // phone number
        $type["founding_date"] = "date"; // founding date
        $type["organization_logo"] = "image"; // founding date
        $type["default_schema"] = "select"; //schema type

        $type["generate_content"] = "toggle"; //whether to generate bulk content or not
		$type["topic"] = "textarea"; //topic to be used for content generation
		$type["categories"] = "select"; //list of categories
		$type["post_type"] = "select"; //post, page etc.
        $type["post_status"] = "select"; //publish, draft etc
		$type["total_articles"] = "input"; //total articles to be generated
        $type["keywords"] = "input"; //total articles to be generated
		$type["duration"] = "select"; //duration, now, each_day, each_week, each_month...default 'now'

        if (isset($type[$field_name])) {
            return $type[$field_name];
        }
        return "input";
    }
    /**
    * Get field data by the name of field
    *
    *
    * @since    1.0.0
    */
    public static function get_field_data($field_name)
    {
        $option0 = array("person",esc_html__('Person', 'ladder-seo'));
        $option1 = array("organization",esc_html__('Organization', 'ladder-seo'));
        $field_data_person_org = array($option0, $option1);

        $data["person_or_organization"] =  $field_data_person_org;
        $data["opening_hours"] =  self::generate_hours("00:00:00", "24:00:00");
        $data["closing_hours"] =  self::generate_hours("00:00:00", "24:00:00");
        $data["chose_a_person"] =  self::get_all_users();
        $data["title_seperator"] =  self::get_title_seperator();       

        $option0 = array("1",esc_html__('Yes', 'ladder-seo'));
        $option1 = array("0",esc_html__('No', 'ladder-seo'));
        $field_data_search_result = array($option0, $option1);

        $data["show_in_search_results"] = $field_data_search_result;

        $helper = new Ladder_SEO_Helper();
        $terms = $helper->get_all_terms(); //list of categories
        $options = array();
        foreach($terms as $term){
            $options[] = array($term->term_id , $term->name);
        }
        $data["categories"] =$options; //list of categories
        $options = array();
		$post_types = $helper->get_post_types(); //post, page etc.
        foreach($post_types as $post_type){
            $options[] = array($post_type, $post_type);
        }
        $data["post_type"] = $options;

        $option0 = array("now",esc_html__('Now', 'ladder-seo'));
        $option1 = array("hourly",esc_html__('Every hour', 'ladder-seo'));
        $option2 = array("daily",esc_html__('Every day', 'ladder-seo'));
        $option3 = array("weekly",esc_html__('Every week', 'ladder-seo'));
        $option4 = array("monthly",esc_html__('Every month', 'ladder-seo'));
        $durations = array($option0, $option1, $option2, $option3, $option4);
        $data["duration"] = $durations;

        $post_status = get_post_statuses();
        $options = array();
        foreach($post_status as $status => $val){
            $options[] = array($status, $val);
        }
        $data["post_status"] = $options;

        $schemas  = self::get_schema_types();
        $options = array();
        foreach($schemas as $type => $schema){
            $options[] = array($type, $schema);
        }

        $data["default_schema"] = $options;

        if (isset($data[$field_name])) {
            return $data[$field_name];
        }
        return array();
    }

    /**
    * Get field attribute by the name of field
    *
    *
    * @since    1.0.0
    */
    public static function get_field_attr($field_name)
    {
        $attr["title"] =  array("data-tags" => "site_title,post_title,separator");
        $attr["tagline"] =  array("data-tags" => "site_title,separator");
        $attr["description"] =  array("data-tags" => "site_title,separator");
        $attr["website_name"] =  array("data-tags" => "site_title,author_name,separator");
        if (isset($attr[$field_name])) {
            return $attr[$field_name];
        }
        return  array("data-tags" => "site_title,post_title,separator");
    }

    /**
    * Get sanitized output
    *
    *
    * @since    1.0.0
    */
    public static function sanitize_input($input)
    {
        if (is_array($input)) {
            return array_map(array( __CLASS__,'sanitize_input'), $input);
        }

        return sanitize_text_field($input);
    }

    /**
    * Get custom post meta box fields
    *
    *
    * @since    1.0.0
    */
    public static function get_post_meta_fields()
    {
        $fields = ["meta_title", "meta_description", "show_in_search_results", "facebook_image", "twitter_image", "schema" ,"canonical"];
        return $fields;
    }

    public static function get_allowed_html()
    {
        $allowed_html = array(
             // Form Elements
                'form' => [
                    'action' => true,
                    'method' => true,
                    'class' => true,
                    'id' => true,
                    'style' => true,
                    'enctype' => true,
                ],
            // Allow <input> tag with common attributes
            'input' => array(
                'type' => array(),
                'name' => array(),
                'value' => array(),
                'class' => array(),
                'id' => array(),
                'placeholder' => array(),
                'maxlength' => array(),
                'size' => array(),
                'autocomplete' => array(),
                'readonly' => array(),
                'disabled' => array(),
                'checked' => array(),
                'required' => array(),
                'width' =>  array(),
                'height' =>  array(),
            ),

            // Allow <label> tag with attributes
            'label' => array(
                'for' => array(),
                'class' => array(),
                'id' => array(),
                'style' => array(),
            ),

            // Allow <select> and <option> tags with attributes
            'select' => array(
                'name' => array(),
                'class' => array(),
                'id' => array(),
                'size' => array(),
                'multiple' => array(),
                'disabled' => array(),
                'required' => array(),
            ),
            'option' => array(
                'value' => array(),
                'selected' => array(),
                'disabled' => array(),
                'label' => array(),
            ),
             // Allow <textarea> with common attributes
            'textarea' => array(
                'name' => array(),
                'class' => array(),
                'id' => array(),
                'rows' => array(),
                'cols' => array(),
                'placeholder' => array(),
                'maxlength' => array(),
                'readonly' => array(),
                'disabled' => array(),
                'required' => array(),
            ),

            // Allow <div> tag with common attributes
            'div' => array(
                'class' => array(),
                'id' => array(),
                'style' => array(),
                'data-*' => array(),  // Allows custom data attributes
            ),
             // Allow <span> tag with common attributes
             'span' => array(
                'class' => array(),
                'id' => array(),
                'style' => array(),
                'data-*' => array(),  // Allows custom data attributes
            ),
            // Allow <span> tag with common attributes
            'small' => array(
                'class' => array(),
                'id' => array(),
                'style' => array(),
                'data-*' => array(),  // Allows custom data attributes
            ),
            'button' => array(
                'type' => array(),
                'name' => array(),
                'value' => array(),
                'class' => array(),
                'id' => array(),
                'disabled' => array(),
                'onclick' => array(),  // Allows inline JavaScript event handling
                'title' => array(),
            ),
            'canvas' => array(
                'class' => array(),
                'id' => array(),
                'width' =>  array(),
                'height' =>  array(),
                'style' => array(),
                'data-*' => array(),  // Allows custom data attributes
            ),
             // SVG Elements and Attributes
                'svg' => [
                    'class' =>  array(),
                    'id' =>  array(),
                    'style' =>  array(),
                    'width' =>  array(),
                    'height' =>  array(),
                    'xmlns' =>  array(),
                    'viewBox' =>  array(),
                    'viewbox' =>  array(),
                    'preserveAspectRatio' =>  array(),
                    'fill' =>  array(),
                    'stroke' => array(),
                    'stroke-width' => array(),
                    'stroke-linecap' => array(),
                    'stroke-linejoin' => array(),
                ],
                'path' => [
                    'd' => true,
                    'fill' => true,
                    'stroke' => true,
                    'stroke-width' => true,
                ],
                'circle' => [
                    'cx' => true,
                    'cy' => true,
                    'r' => true,
                    'fill' => true,
                    'stroke' => true,
                    'stroke-width' => true,
                ],
                'rect' => [
                    'width' => true,
                    'height' => true,
                    'x' => true,
                    'y' => true,
                    'rx' => true,
                    'ry' => true,
                    'fill' => true,
                    'stroke' => true,
                    'stroke-width' => true,
                ],
                'line' => [
                    'x1' => true,
                    'y1' => true,
                    'x2' => true,
                    'y2' => true,
                    'stroke' => true,
                    'stroke-width' => true,
                ],
                'polygon' => [
                    'points' => true,
                    'fill' => true,
                    'stroke' => true,
                    'stroke-width' => true,
                ],
                'polyline' => [
                    'points' => true,
                    'fill' => true,
                    'stroke' => true,
                    'stroke-width' => true,
                ],
                'text' => [
                    'x' => true,
                    'y' => true,
                    'dx' => true,
                    'dy' => true,
                    'fill' => true,
                    'stroke' => true,
                    'font-family' => true,
                    'font-size' => true,
                    'text-anchor' => true,
                ],
                'lineargradient' => array(             // linearGradient
                    'gradientunits'     => true,   // gradientUnits
                    'gradienttransform' => true,   // gradientTransform
                    'spreadmethod'      => true,   // spreadMethod
                ),
        );

        $allowed_html_post = wp_kses_allowed_html('post');
        $allowed_html = array_merge_recursive($allowed_html, $allowed_html_post);
        return $allowed_html;
    }



    /**
     * Generates opening and closing hours with half-hour intervals.
     *
     * @param string $start_time Start time in "H:i:s" format.
     * @param string $end_time   End time in "H:i:s" format.
     * @return array            Array of time slots in the specified format.
     */
    private static function generate_hours(string $start_time, string $end_time)
    {
        $start = strtotime($start_time);
        $end = strtotime($end_time);
        $intervals = [];

        while ($start <= $end) {
            $formattedTime = gmdate("H:i:s", $start);
            $intervals[] = [$formattedTime, $formattedTime];
            $start = strtotime('+30 minutes', $start);
        }

        return $intervals;
    }

    private static function get_all_users()
    {
        $users = get_users();
        $user_data = [];
        foreach ($users as $user) {
            $user_data[]  = [$user->ID, $user->display_name];
        }
        return $user_data;
    }

    private static function get_title_seperator()
    {
        $separator = array(array("-","-"), array("_","_"), array("|","|"));
        return $separator;
    }

}