<?php
// File: includes/class-ladder-seo-logger.php
/*
* @link       https://ladder-seo.com
* @since      1.0.0
*
* @package    Ladder_SEO
* @subpackage Ladder_SEO/includes
*/
if (! defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class Ladder_SEO_Logger
{
    private static $max_file_size = 5 * 1024 * 1024;

    private static function get_log_dir()
    {
        $upload_dir = wp_upload_dir();
        $log_dir = $upload_dir['basedir'] . '/ladderseo-logs/';
        if (! file_exists($log_dir)) {
            mkdir($log_dir, 0755, true);
        }
        return $log_dir;
    }

    private static function get_log_file()
    {
        return self::get_log_dir() . 'ladderseo_log.txt';
    }

    private static function rotate_log()
    {
        $log_file = self::get_log_file();

        if (file_exists($log_file) && filesize($log_file) >= self::$max_file_size) {
            $timestamp = gmdate('Ymd_His');
            $new_log_file = self::get_log_dir() . "ladderseo_log_$timestamp.txt";
            rename($log_file, $new_log_file);
        }
    }

    private static function write_log($level, $message)
    {
        self::rotate_log();

        $log_file = self::get_log_file();
        $timestamp = gmdate('Y-m-d H:i:s');
        $log_entry = "[$timestamp] " . strtoupper($level) . ": $message" . PHP_EOL;

        file_put_contents($log_file, $log_entry, FILE_APPEND);
    }

    public static function log_emergency($message)
    {
        self::write_log('emergency', $message);
    }

    public static function log_alert($message)
    {
        self::write_log('alert', $message);
    }

    public static function log_critical($message)
    {
        self::write_log('critical', $message);
    }

    public static function log_error($message)
    {
        self::write_log('error', $message);
    }

    public static function log_warning($message)
    {
        self::write_log('warning', $message);
    }

    public static function log_notice($message)
    {
        self::write_log('notice', $message);
    }

    public static function log_info($message)
    {
        self::write_log('info', $message);
    }

    public static function log_debug($message)
    {
        self::write_log('debug', $message);
    }

    // Retrieve and filter logs based on search criteria
    public static function get_logs($limit = 100, $search = '', $level = '')
    {
        $log_file = self::get_log_file();

        if (! file_exists($log_file)) {
            return [];
        }

        $lines = file($log_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $lines = array_reverse($lines);

        if ($search) {
            $lines = array_filter($lines, function ($line) use ($search) {
                return stripos($line, $search) !== false;
            });
        }

        if ($level) {
            $lines = array_filter($lines, function ($line) use ($level) {
                return stripos($line, strtoupper($level)) !== false;
            });
        }

        return array_slice($lines, 0, $limit);
    }

    // Display logs in admin with search and filter options
    public static function display_logs_in_admin()
    {
        $search = isset($_GET['log_search']) ? sanitize_text_field($_GET['log_search']) : '';
        $level = isset($_GET['log_level']) ? sanitize_text_field($_GET['log_level']) : '';
        $logs = self::get_logs(100, $search, $level);

        ?>
            <div class="wrap">
                <ul>
                    <?php
                        if (empty($logs)) {
                            echo '<li>'. esc_html__("No logs available.","ladder-seo").'</li>';
                        } else {
                            foreach ($logs as $log) {
                                echo '<li>' . esc_html($log) . '</li>';
                            }
                        }
                    ?>
                </ul>
            </div>
        <?php
    }
}

?>