let editors = {}; // Store all editors

document.addEventListener('DOMContentLoaded', function () {
  // Tags with display names for buttons and dropdown
  const tagDisplayMap = {
    site_title: '+ Site Title',
    post_title: '+ Post Title',
    author_name: '+ Author Name',
    tagline: '+ Tagline',
    page_description: '+ Page Description',
    post_description: '+ Post Description',
    term_name: '+ Term name',
    author_first_name: '+ Author first name',
    author_last_name: '+ Author last name',
    current_month: '+ Current month',
    current_year: '+ Current year',
    current_date: '+ Current date',
    current_day: '+ Current day',
    post_date: '+ Post date',
    separator: '+ Text Separator'
  };

  // Initialize Quill editors dynamically
  document.querySelectorAll('.editor-container').forEach((editorContainer) => {
    const editorId = editorContainer.id;

    // Initialize Quill editor
    editors[editorId] = new Quill(`#${editorId}`, {
      theme: 'snow',
      modules: { toolbar: false }
    });

    // Find corresponding quick-buttons container (optional)
    const buttonsContainer = document.getElementById(`quick-buttons-${editorId}`);
    if (buttonsContainer) {
      createQuickButtons(buttonsContainer, editorId);
    }

    // Listen for text changes and trigger dropdown
    editors[editorId].on('text-change', function (delta, oldDelta, source) {
      if (source === 'user') {
        const range = editors[editorId].getSelection();
        if (range) {
          const cursorPosition = range.index;
          const textBeforeCursor = editors[editorId].getText(0, cursorPosition);

          if (textBeforeCursor.endsWith('#')) {
            showDropdown(editorId, range);
          }
        }
      }
    });
  });

  // Function to create quick-access buttons dynamically
  function createQuickButtons(container, editorId) {
    const tags = container.getAttribute('data-tags').split(',');

    tags.forEach((tag) => {
      if (tagDisplayMap[tag]) {
        const button = document.createElement('button');
        button.setAttribute('type', 'button'); // Prevent form submission
        button.classList.add('ladderseo-buttons'); // Add multiple classes
        button.classList.add('quick-tag-button');
        button.textContent = tagDisplayMap[tag]; // Display user-friendly name
        button.addEventListener('click', () => insertTag(editorId, tag));
        container.appendChild(button);
      }
    });
  }

  // Create dropdown element
  const dropdown = document.createElement('div');
  dropdown.classList.add('tag-dropdown');

  // Add search input inside dropdown
  const searchInput = document.createElement('input');
  searchInput.setAttribute('type', 'text');
  searchInput.setAttribute('placeholder', 'Search tags...');
  dropdown.appendChild(searchInput);

  const listContainer = document.createElement('ul');
  dropdown.appendChild(listContainer);

  document.body.appendChild(dropdown);

  let activeEditorId = null;

  // Show dropdown near caret in the active editor
  function showDropdown(editorId, range) {
    activeEditorId = editorId;
    const editor = editors[editorId];
    const bounds = editor.getBounds(range.index);

    const editorContainer = document.querySelector(`#${editorId}`);
    const editorRect = editorContainer.getBoundingClientRect();

    dropdown.style.left = `${editorRect.left + bounds.left + window.pageXOffset}px`;
    dropdown.style.top = `${editorRect.top + bounds.bottom + window.pageYOffset}px`;
    dropdown.style.display = 'block';

    populateDropdown('');
    searchInput.focus();
  }

  // Populate dropdown with tag items
  function populateDropdown(filterText) {
    listContainer.innerHTML = ''; // Clear previous list items
    Object.keys(tagDisplayMap)
      .filter((tag) => tagDisplayMap[tag].toLowerCase().includes(filterText.toLowerCase()))
      .forEach((tag) => {
        const listItem = document.createElement('li');
        listItem.textContent = tagDisplayMap[tag];
        listItem.addEventListener('click', () => insertTag(activeEditorId, tag));
        listContainer.appendChild(listItem);
      });
  }

  // Filter dropdown as user types in the search box
  searchInput.addEventListener('input', function () {
    populateDropdown(this.value);
  });

  // Insert the selected tag into the specified editor
  function insertTag(editorId, tag) {
    const range = editors[editorId].getSelection(true);
    editors[editorId].insertText(range.index, `{{${tag}}} `);
    editors[editorId].setSelection(range.index + tag.length + 4);
    dropdown.style.display = 'none';
  }

  // Hide the dropdown when clicking outside
  document.addEventListener('click', function (event) {
    if (!dropdown.contains(event.target)) {
      dropdown.style.display = 'none';
    }
  });
});
