<?php  if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly ?>
<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://digitaldreamstech.com
 * @since      1.0.0
 *
 * @package    Ladder_SEO
 * @subpackage Ladder_SEO/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 *
 * @package    Ladder_SEO
 * @subpackage Ladder_SEO/admin
 * @author     Pritesh Gupta <g.pritesh@gmail.com>
 */
class Ladder_SEO_Admin
{
    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;
    /**
    * The object of view class of admin(backend)
    *
    * @since    1.0.0
    * @access   private
    * @var      string    $view    Object of view class
    */
    private $view;
    /**
     * The object of helper class
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $view    Object of view class
     */
    private $helper;

    /**
     * Different pages hooks
     *
     * @since    1.0.0
     * @access   private
     */
    private $hooks;
    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      string    $plugin_name       The name of this plugin.
     * @param      string    $version    The version of this plugin.
     */
    public function __construct($plugin_name, $version)
    {
        $this->view = new Ladder_SEO_Admin_View();
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->helper = new Ladder_SEO_Helper_Elements();
        $this->ladder_seo_taxonomy_meta_box_add();
        $this->ladder_seo_schedule_event();
        $this->hooks = [
            "ladder-seo_page_ladder_seo_settings",
            "toplevel_page_ladder_seo_admin",
            "ladder-seo_page_ladder_seo_global",
            "ladder-seo_page_ladder_seo_content_generation",
            "ladder-seo_page_ladder_seo_bulk_entry",
            "ladder-seo_page_ladder_seo_w3c",
            "ladder-seo_page_ladder_seo_webmaster_analytics",
            "ladder-seo_page_ladder_seo_keyword_research",
            "ladder-seo_page_ladder_seo_social",
            "ladder-seo_page_ladder_seo_optimizations",
            "ladder-seo_page_ladder_seo_redirection"
        ];
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles($hook)
    {
        if (in_array($hook, $this->hooks)) {
            wp_enqueue_style("bootstrap", plugin_dir_url(__FILE__) . 'css/bootstrap/bootstrap.min.css', array(), $this->version, 'all');
            wp_enqueue_style("datatables", plugin_dir_url(__FILE__) . 'css/datatables/datatables.min.css', array("bootstrap"), $this->version, 'all');
            wp_enqueue_style("select2", plugin_dir_url(__FILE__) . 'css/select2.min.css', array(), $this->version, 'all');
            wp_enqueue_style('font-awesome', plugin_dir_url(__FILE__) .'css/all.min.css', array(), '6.5.1');
            wp_enqueue_style("quill", plugin_dir_url(__FILE__) . 'css/quill.snow.css', array(), $this->version, 'all');
            wp_enqueue_style($this->plugin_name, plugin_dir_url(__FILE__) . 'css/ladder-seo-admin.css', array('bootstrap'), $this->version, 'all');
            wp_enqueue_style($this->plugin_name."-style", plugin_dir_url(__FILE__) . 'css/ladder-seo-admin-style.css', array('bootstrap'), $this->version, 'all');
        } elseif (in_array($hook, ['post.php', 'post-new.php', 'edit-tags.php', 'term.php'])) {
            wp_enqueue_style('ladderseo-meta-box-css', plugin_dir_url(__FILE__) . 'css/ladder-seop-admin-meta.css', array(), $this->version, false);
        }
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts($hook)
    {
        if (in_array($hook, $this->hooks)) {
            wp_enqueue_script('jquery');
            wp_enqueue_media();
            wp_enqueue_script("bootstrap", plugin_dir_url(__FILE__) . 'js/bootstrap/bootstrap.bundle.min.js', array(), $this->version, false);
            wp_enqueue_script("select2", plugin_dir_url(__FILE__) . 'js/select2.min.js', array(), $this->version, false);
            wp_enqueue_script("datatables", plugin_dir_url(__FILE__) . 'js/datatables/datatables.min.js', array('jquery','bootstrap'), $this->version, false);
            wp_enqueue_script($this->plugin_name, plugin_dir_url(__FILE__) . 'js/ladder-seo-admin.js', array( 'jquery', 'bootstrap' ), $this->version, false);
            wp_enqueue_script("quill", plugin_dir_url(__FILE__) . 'js/quill.js', array( ), $this->version, false);
            wp_enqueue_script($this->plugin_name."-quill", plugin_dir_url(__FILE__) . 'js/ladder-seo-admin-quill.js', array('quill' ), $this->version, false);
        } elseif (in_array($hook, ['post.php', 'post-new.php'])) {
            wp_enqueue_script('ladderseo-meta-box-js', plugin_dir_url(__FILE__) . 'js/ladder-seo-admin-meta.js', array(), $this->version, true);
            wp_enqueue_script('ladderseo-custom-quill-js', plugin_dir_url(__FILE__) . 'js/ladder-seo-admin-custom-quill.js', array(), $this->version, true);
            wp_enqueue_media();
        } elseif (in_array($hook, [ 'edit-tags.php', 'term.php'])) {
            wp_enqueue_script('ladderseo-meta-box-js', plugin_dir_url(__FILE__) . 'js/ladder-seo-admin-meta-tax.js', array(), $this->version, true);
            wp_enqueue_script('ladderseo-custom-quill-js', plugin_dir_url(__FILE__) . 'js/ladder-seo-admin-custom-quill.js', array(), $this->version, true);
            wp_enqueue_media();
        }


    }

    /**
     * Register the menu for the admin area.
     *
     * @since    1.0.0
     */
    public function admin_menu()
    {
        $menu_slug = "ladder_seo_admin";

        add_menu_page(esc_html__('Ladder SEO', 'ladder-seo'), esc_html__("Ladder SEO", "ladder-seo"), 'manage_options', $menu_slug, false, plugins_url('admin/images/logo.png', LADDER_SEO_FILE));
        add_submenu_page($menu_slug, esc_html__('General Settings Page', 'ladder-seo'), esc_html__('General', 'ladder-seo'), 'manage_options', $menu_slug, array($this,'ladder_seo_general' ));
        add_submenu_page($menu_slug, esc_html__('Global Settings Page', 'ladder-seo'), esc_html__('Global', 'ladder-seo'), 'manage_options', 'ladder_seo_global', array($this,'ladder_seo_global' ));
        add_submenu_page($menu_slug, esc_html__('Content Generation Page', 'ladder-seo'), esc_html__('Content Generation', 'ladder-seo'), 'manage_options', 'ladder_seo_content_generation', array($this,'ladder_seo_content_generation' ));
        add_submenu_page($menu_slug, esc_html__('Bulk Entry Page', 'ladder-seo'), esc_html__('Bulk Entry', 'ladder-seo'), 'manage_options', 'ladder_seo_bulk_entry', array($this,'ladder_seo_bulk_entry' ));
        add_submenu_page($menu_slug, esc_html__('W3C Settings Page', 'ladder-seo'), esc_html__('W3C', 'ladder-seo'), 'manage_options', 'ladder_seo_w3c', array($this,'ladder_seo_w3c' ));
        add_submenu_page($menu_slug, esc_html__('Redirection Page', 'ladder-seo'), esc_html__('Redirection', 'ladder-seo'), 'manage_options', 'ladder_seo_redirection', array($this,'ladder_seo_redirection' ));
        add_submenu_page($menu_slug, esc_html__('Webmaster & Analytics Page', 'ladder-seo'), esc_html__('Webmaster & Analytics', 'ladder-seo'), 'manage_options', 'ladder_seo_webmaster_analytics', array($this,'ladder_seo_webmaster_analytics' ));
        add_submenu_page($menu_slug, esc_html__('Keyword Research Page', 'ladder-seo'), esc_html__('Keyword Research', 'ladder-seo'), 'manage_options', 'ladder_seo_keyword_research', array($this,'ladder_seo_keyword_research' ));
        add_submenu_page($menu_slug, esc_html__('Social Media Settings Page', 'ladder-seo'), esc_html__('Social', 'ladder-seo'), 'manage_options', 'ladder_seo_social', array($this,'ladder_seo_social' ));
        add_submenu_page($menu_slug, esc_html__('Optimizations Page', 'ladder-seo'), esc_html__('Optimizations', 'ladder-seo'), 'manage_options', 'ladder_seo_optimizations', array($this,'ladder_seo_optimizations' ));
        add_submenu_page($menu_slug, esc_html__('Settings Page', 'ladder-seo'), esc_html__('Settings', 'ladder-seo'), 'manage_options', 'ladder_seo_settings', array($this,'ladder_seo_settings' ));
    }

    /**
    * Add custom meta box
    *
    * @since    1.0.0
    */
    public function ladder_seo_meta_box_add()
    {
        $post_types = $this->helper->get_post_types();
        add_meta_box(
            'ladderseo_meta_box', // ID
            esc_html__('Ladder SEO', 'ladder-seo'), // Title
            array($this,'ladder_seo_post_meta_box' ),
            $post_types, // Post types
            'normal', // Context
            'high' // Priority
        );
    }

    /**
    * Add custom meta box for taxonomies
    *
    * @since    1.0.0
    */
    public function ladder_seo_taxonomy_meta_box_add()
    {
        $taxonomies = $this->helper->get_taxonomies();
        foreach ($taxonomies as $taxonomy) {
            add_action("{$taxonomy}_edit_form", [ $this, 'ladder_seo_taxonomy_meta_box' ]);
            add_action("edited_$taxonomy", [ $this, 'ladder_seo_save_taxonomy' ]);
        }
    }

    /**
     * Add links in plugin section
     *
     * @since    1.0.0
     */
    public function ladder_seo_add_plugins_link($links){
        $how_to_use_url = admin_url('admin.php?page=ladder_seo_admin#ladderseo_how_to_use');
        $how_to_use_link = '<a href="' . esc_url($how_to_use_url) . '">'.esc_html__("How to use?", "ladder-seo").'</a>';
        array_unshift($links, $how_to_use_link);

        $settings_url = admin_url('admin.php?page=ladder_seo_admin');
        $settings_link = '<a href="' . esc_url($settings_url) . '">'.esc_html__("Settings", "ladder-seo").'</a>';
        array_unshift($links, $settings_link); 

        return $links;
    }

    /**
     * General page
     *
     * @since    1.0.0
     */
    public function ladder_seo_general()
    {
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        echo wp_kses($this->ladder_seo_page_data("general"), $allowed_html);
    }

    /**
     * Global page
     *
     * @since    1.0.0
     */
    public function ladder_seo_global()
    {
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        echo wp_kses( $this->ladder_seo_page_data("global"), $allowed_html);
    }

    /**
     * Content generation page
     *
     * @since    1.0.0
     */
    public function ladder_seo_content_generation()
    {
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        echo wp_kses( $this->ladder_seo_page_data("content"), $allowed_html);
    }

    /**
     * Bulk entry page
     *
     * @since    1.0.0
     */
    public function ladder_seo_bulk_entry()
    {
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        echo wp_kses(  $this->ladder_seo_page_data("bulk-entry"), $allowed_html);
    }

    /**
     * W3C page
     *
     * @since    1.0.0
     */
    public function ladder_seo_w3c()
    {
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        echo wp_kses(  $this->ladder_seo_page_data("w3c"), $allowed_html);
    }

    /**
     * Redirection page
     *
     * @since    1.0.0
     */
    public function ladder_seo_redirection()
    {
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        echo wp_kses(  $this->ladder_seo_page_data("redirection"), $allowed_html);
    }

    /**
     * Keyword-research page
     *
     * @since    1.0.0
     */
    public function ladder_seo_keyword_research()
    {
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        echo wp_kses(  $this->ladder_seo_page_data("keyword-research"), $allowed_html);
    }

    /**
     * webmaster and analytics
     *
     * @since    1.0.0
     */
    public function ladder_seo_webmaster_analytics()
    {
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        echo wp_kses(  $this->ladder_seo_page_data("webmaster-analytics"), $allowed_html);
    }
    
     /**
     * social page
     *
     * @since    1.0.0
     */
    public function ladder_seo_social()
    {
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        echo wp_kses(  $this->ladder_seo_page_data("social"), $allowed_html);
    }

     /**
     * optimizations page
     *
     * @since    1.0.0
     */
    public function ladder_seo_optimizations()
    {
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        echo wp_kses(  $this->ladder_seo_page_data("optimizations"), $allowed_html);
    }



    /**
     * Settings page
     *
     * @since    1.0.0
     */
    public function ladder_seo_settings()
    {
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        echo wp_kses(  $this->ladder_seo_page_data("settings"), $allowed_html);       
    }

    private function ladder_seo_page_data($page_type ){

        $settings = get_option(LADDER_SEO_SETTINGS);
        $post_types = $this->helper->get_post_types();
        $taxonomies = $this->helper->get_taxonomies();
        $posts = $this->helper->get_all_posts();
        $terms = $this->helper->get_all_terms();
        $post_data = array();
        $term_data = array();
        $data = array();

        $fields = ["meta_title","meta_description"];
        foreach ($posts as $post) {
            foreach ($fields as $field) {
                $post_type = $post->post_type;
                $id = $post->ID;
                if (!empty(get_post_meta($id, "ladderseo_".$post_type."-".$field, true))) {
                    $post_data[$id][$post_type][$field] = get_post_meta($id, "ladderseo_".$post_type."-".$field, true);
                } elseif (isset($settings["post_types"][$post_type][$field])) {
                    $post_data[$id][$post_type][$field] = $settings["post_types"][$post_type][$field];
                } elseif (isset($settings["content_types"]["title_and_description"][$field])) {
                    $post_data[$id][$post_type][$field] = $settings["content_types"]["title_and_description"][$field];
                } else {
                    $post_data[$id][$post_type][$field] = "";
                }
            }
        }

        foreach ($terms as $term) {
            foreach ($fields as $field) {
                $term_type  = $term->taxonomy;
                $term_id = $term->term_id;
                if (!empty(get_post_meta($term_id, "ladderseo_".$term_type."-".$field, true))) {
                    $term_data[$term_id][$term_type][$field] = get_term_meta($term_id, "ladderseo_".$term_type."-".$field, true);
                } elseif (isset($settings["term_types"][$term_type][$field])) {
                    $term_data[$term_id][$term_type][$field] = $settings["term_types"][$term_type][$field];
                } elseif (isset($settings["content_types"]["title_and_description"][$field])) {
                    $term_data[$term_id][$term_type][$field] = $settings["content_types"]["title_and_description"][$field];
                } else {
                    $term_data[$term_id][$term_type][$field] = "";
                }
            }
        }

		$db_manager = new \LadderSEO\DB_Manager();		
        $data["redirection"] = $db_manager->get_data_redirection();
        $data["404_logs"] = $db_manager->get_data_404_logs();
        $data["chart_data"] = $this->get_chart_data("stats");
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        $settings_form =  $this->view->get_settings_form($settings, $this->helper, $post_types, $taxonomies, $posts, $terms, $post_data, $term_data, $data, $page_type);
        return wp_kses($settings_form,  $allowed_html);
    }


    /**
     * Display custom post meta box
     *
     * @since    1.0.0
     */

    public function ladder_seo_post_meta_box($post)
    {
        $settings = get_option(LADDER_SEO_SETTINGS);
        $fields = Ladder_SEO_Util::get_post_meta_fields();
        $schemas  = Ladder_SEO_Util::get_schema_types();
        $data = array();
        $post_type = $post->post_type;

        foreach ($fields as $field) {
            if (!empty(get_post_meta($post->ID, "ladderseo_".$post_type."-".$field, true))) {
                $data[$post_type][$field] = get_post_meta($post->ID, "ladderseo_".$post_type."-".$field, true);
            } elseif (isset($settings["post_types"][$post_type][$field])) {
                $data[$post_type][$field] = $settings["post_types"][$post_type][$field];
            } elseif (isset($settings["content_types"]["title_and_description"][$field])) {
                $data[$post_type][$field] = $settings["content_types"]["title_and_description"][$field];
            } elseif (isset($settings["content_types"]["social"][$field])) {
                $data[$post_type][$field] = $settings["content_types"]["social"][$field];
            } else {
                $data[$post_type][$field] = "";
            }
        }
        $data[$post_type]['facebook_image'] = !empty($data[$post_type]['facebook_image'])? $data[$post_type]['facebook_image'] : LADDER_SEO_PLUGIN_URL."/admin/images/placeholder.jpeg";
        $data[$post_type]['twitter_image'] = !empty($data[$post_type]['twitter_image'])? $data[$post_type]['twitter_image'] : LADDER_SEO_PLUGIN_URL."/admin/images/placeholder.jpeg";
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        $post_meta_form = $this->view->get_post_meta_form($settings, $this->helper, $post, $data, $schemas);
        echo wp_kses($post_meta_form,  $allowed_html);
    }

    /**
     * Display custom meta box for taxonomy
     *
     * @since    1.0.0
     */

    public function ladder_seo_taxonomy_meta_box($term)
    {
        $settings = get_option(LADDER_SEO_SETTINGS);
        $fields = Ladder_SEO_Util::get_post_meta_fields();
        $schemas  = Ladder_SEO_Util::get_schema_types();
        $term_data = array();
        $term_type  = $term->taxonomy;
        $term_id = $term->term_id;

        foreach ($fields as $field) {            
            if (!empty(get_term_meta($term_id, "ladderseo_".$term_type."-".$field, true))) {
                $term_data[$term_type][$field] = get_term_meta($term_id, "ladderseo_".$term_type."-".$field, true);
            } elseif (isset($settings["term_types"][$term_type][$field])) {
                $term_data[$term_type][$field] = $settings["term_types"][$term_type][$field];
            } elseif (isset($settings["content_types"]["title_and_description"][$field])) {
                $term_data[$term_type][$field] = $settings["content_types"]["title_and_description"][$field];
            } else {
                $term_data[$term_type][$field] = "";
            }
        }

        $term_data[$term_type]['facebook_image'] = !empty($term_data[$term_type]['facebook_image'])? $term_data[$term_type]['facebook_image'] : LADDER_SEO_PLUGIN_URL."/admin/images/placeholder.jpeg";
        $term_data[$term_type]['twitter_image'] = !empty($term_data[$term_type]['twitter_image'])? $term_data[$term_type]['twitter_image'] : LADDER_SEO_PLUGIN_URL."/admin/images/placeholder.jpeg";
        $allowed_html = Ladder_SEO_Util::get_allowed_html();
        $tax_form = $this->view->get_taxonomy_meta_form($settings, $this->helper, $term, $term_data, $schemas);
        echo wp_kses($tax_form,  $allowed_html);
    }

    /**
     * save custom post data
     *
     * @since    1.0.0
     */
    public function ladder_seo_save_post($post_id)
    {
        if (!isset($_POST['ladder_seo_nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash ($_POST['ladder_seo_nonce'])), 'ladder_seo')) {
            return;
        }

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        $post_type = get_post_type($post_id);
        $fields = Ladder_SEO_Util::get_post_meta_fields();
        
        foreach ($fields as $field) {
            $meta_key = "ladderseo_".$post_type."-".$field;
            $meta_val = isset($_POST[$meta_key]) ? sanitize_text_field($_POST[$meta_key]) : "";
            update_post_meta($post_id, "ladderseo_".$post_type."-".$field, sanitize_text_field($_POST["ladderseo_".$post_type."-".$field]));
        }

        if(isset($_POST["ladderseo_hide_from_result"])){
            $meta_val = sanitize_text_field($_POST['ladderseo_hide_from_result']);
            update_post_meta($post_id, "ladderseo_hide_from_result", $meta_val);
        } else{
            update_post_meta($post_id, "ladderseo_hide_from_result", 0);
        }

    }

    /**
     * save custom taxonomy meta data
     *
     * @since    1.0.0
     */
    public function ladder_seo_save_taxonomy($term_id)
    {
        if (!isset($_POST['ladder_seo_nonce']) || !wp_verify_nonce(sanitize_text_field( wp_unslash ($_POST['ladder_seo_nonce'])), 'ladder_seo')) {
            return;
        }

        $term = get_term($term_id);
        $term_type  = $term->taxonomy;
        $fields = Ladder_SEO_Util::get_post_meta_fields();

        foreach ($fields as $field) {
            $meta_key = "ladderseo_".$term_type."-".$field;
            $meta_val = isset($_POST[$meta_key]) ? sanitize_text_field($_POST[$meta_key]) : "";
            update_term_meta($term_id, "ladderseo_".$term_type."-".$field, sanitize_text_field($_POST["ladderseo_".$term_type."-".$field]));
        }

    }

    /**
     * handles all ajax requests
     *
     * @since    1.0.0
     */
    public function ladder_seo_admin_ajax()
    {
        $ajax_nonce = isset($_POST['ladder_seo_nonce']) ? sanitize_text_field($_POST['ladder_seo_nonce']) : "";
        if (! wp_verify_nonce($ajax_nonce, 'ladder_seo')) {
            esc_html_e('Unauthorized request', 'ladder-seo');
            wp_die();
        }

        $action = sanitize_text_field($_POST["action_type"]);
        $data = Ladder_SEO_Util::sanitize_input($_POST);
        switch ($action) {
            case "save_ai_settings": return $this->ladderseo_save_ai_settings($data);
            case "save_global_settings": return $this->ladderseo_save_global_settings($data);
            case "save_content_settings": return $this->ladderseo_save_content_settings($data);
            case "save_license_settings": return $this->ladderseo_save_license_settings($data);
            case "save_general_settings": return $this->ladderseo_save_general_settings($data);
            case "save_w3c_settings": return $this->ladderseo_save_w3c_settings($data);
            case "save_optimization_settings": return $this->ladderseo_save_optimization_settings($data);
            case "save_social_settings": return $this->ladderseo_save_social_settings($data);
            case "save_post_types": return $this->ladderseo_save_post_types($data);
            case "save_taxonomy_types": return $this->ladderseo_save_tax_types($data);
            case "save_bulk_data_settings": return $this->ladderseo_save_bulk_data($data);
            case "save_redirection": return $this->ladderseo_save_redirection($data);
            case "save_webmaster_analytics": return $this->ladderseo_save_webmaster_analytics($data);
            case "ladderseo_get_meta_title": return $this->ladderseo_get_meta_data($data);
            case "ladderseo_get_meta_description": return $this->ladderseo_get_meta_data($data);
            case "delete_redirection": return $this->ladderseo_delete_redirection($data);
            case "delete_404_logs": return $this->ladderseo_delete_404_logs($data);            
            default: return esc_html_e('Invalid request', 'ladder-seo');
        }
        wp_die();
    }

    private function ladderseo_save_ai_settings($data)
    {
        $settings = get_option(LADDER_SEO_SETTINGS);
        $ai_fields = isset($settings["ai"]) ? $settings["ai"] : array();
        foreach ($ai_fields as $ai_field_name => $ai_field_val) {
            $field_name = "ladderseo_ai_".$ai_field_name;
            $settings["ai"][$ai_field_name] = isset($data[$field_name]) ? $data[$field_name] : "";
        }
        return update_option(LADDER_SEO_SETTINGS, $settings);
    }

    private function ladderseo_save_license_settings($data)
    {
        $settings = get_option(LADDER_SEO_SETTINGS);
        $license_fields = isset($settings["license"]) ? $settings["license"] : array();
        foreach ($license_fields as $license_field_name => $license_field_val) {
            $field_name = "ladderseo_license_".$license_field_name;
            $settings["license"][$license_field_name] = isset($data[$field_name]) ? $data[$field_name] : "";
        }
        return update_option(LADDER_SEO_SETTINGS, $settings);
    }

    private function ladderseo_save_general_settings($data)
    {
        $settings = get_option(LADDER_SEO_SETTINGS);
        $general_fields = isset($settings["general"]) ? $settings["general"] : array();
        foreach ($general_fields as $general_field_name => $general_field_val) {
            $field_name = "ladderseo_general_".$general_field_name;
            $settings["general"][$general_field_name] = isset($data[$field_name]) ? $data[$field_name] : 0;
        }
        return update_option(LADDER_SEO_SETTINGS, $settings);
    }

    private function ladderseo_save_global_settings($data)
    {
        $settings = get_option(LADDER_SEO_SETTINGS);
        $global_fields = isset($settings["global"]) ? $settings["global"] : array();
        foreach ($global_fields as $global_field_name => $global_field_val) {
            foreach($global_field_val as $key => $val ){
                $field_name = "ladderseo_global_".$global_field_name."_".$key;
                $settings["global"][$global_field_name][$key] = isset($data[$field_name]) ? $data[$field_name] : 0;
            }            
        }
        return update_option(LADDER_SEO_SETTINGS, $settings);
    }

    private function ladderseo_save_content_settings($data)
    {
        wp_clear_scheduled_hook('ladderseo_generate_content');
        $settings = get_option(LADDER_SEO_SETTINGS);
        $content_fields = isset($settings["content"]) ? $settings["content"] : array();
        foreach ($content_fields as $content_field_name => $content_field_val) {
            $field_name = "ladderseo_content_".$content_field_name;
            $settings["content"][$content_field_name] = isset($data[$field_name]) ? $data[$field_name] : 0;
        }
        $data = $settings["content"];
        $duration = strtolower($data["duration"]) ;
        if($duration  === "now")
        {            
	        $this->ladder_seo_generate_content();
        }
        return update_option(LADDER_SEO_SETTINGS, $settings);
    }

    private function ladderseo_save_w3c_settings($data)
    {
        $settings = get_option(LADDER_SEO_SETTINGS);
        $w3c_fields = isset($settings["w3c"]) ? $settings["w3c"] : array();
        foreach ($w3c_fields as $w3c_field_name => $w3c_field_val) {
            $field_name = "ladderseo_w3c_".$w3c_field_name;
            $settings["w3c"][$w3c_field_name] = isset($data[$field_name]) ? $data[$field_name] : 0;
        }
        return update_option(LADDER_SEO_SETTINGS, $settings);
    }

    private function ladderseo_save_optimization_settings($data)
    {
        $settings = get_option(LADDER_SEO_SETTINGS);
        $optimization_fields = isset($settings["optimization"]) ? $settings["optimization"] : array();
        foreach ($optimization_fields as $optimization_field_name => $optimization_field_val) {
            $field_name = "ladderseo_optimization_".$optimization_field_name;
            $settings["optimization"][$optimization_field_name] = isset($data[$field_name]) ? $data[$field_name] : 0;
        }
        return update_option(LADDER_SEO_SETTINGS, $settings);
    }

    private function ladderseo_save_social_settings($data)
    {
        $settings = get_option(LADDER_SEO_SETTINGS);
        $social_fields = isset($settings["social"]) ? $settings["social"] : array();
        foreach ($social_fields as $social_field_name => $social_field_val) {
            $field_name = "ladderseo_social_".$social_field_name;
            $settings["social"][$social_field_name] = isset($data[$field_name]) ? $data[$field_name] : 0;
        }
        return update_option(LADDER_SEO_SETTINGS, $settings);
    }

    private function ladderseo_save_post_types($data)
    {
        $settings = get_option(LADDER_SEO_SETTINGS);
        $content_type = [];
        foreach ($data as $field_name => $val) {
            $post_type_data = $this->parse_content_type_field_name($field_name);
            if (count($post_type_data)) {
                $content_type[$post_type_data["post_type"]][$post_type_data["field_name"]] = $val;
            }
        }
        $settings["post_types"] = $content_type;
        return update_option(LADDER_SEO_SETTINGS, $settings);
    }

    private function ladderseo_save_tax_types($data)
    {
        $settings = get_option(LADDER_SEO_SETTINGS);
        $content_type = [];
        foreach ($data as $field_name => $val) {
            $tax_type_data = $this->parse_content_type_field_name($field_name);
            if (count($tax_type_data)) {
                $content_type[$tax_type_data["post_type"]][$tax_type_data["field_name"]] = $val;
            }
        }
        $settings["term_types"] = $content_type;
        return update_option(LADDER_SEO_SETTINGS, $settings);
    }

    private function ladderseo_save_redirection($data)
    {
        $insert_data = [];
        $insert_data["source_url"] = isset($data["ladderseo_redirection_source_url"]) ? $data["ladderseo_redirection_source_url"] : "" ;
        $insert_data["target_url"] = isset($data["ladderseo_redirection_target_url"]) ? $data["ladderseo_redirection_target_url"] : "" ;
        $insert_data["redirect_type"] = isset($data["ladderseo_redirection_type"]) ? $data["ladderseo_redirection_type"] : "" ;
        $db_manager = new \LadderSEO\DB_Manager();
		return $db_manager->insert_data_redirection($insert_data);
    }

    private function ladderseo_save_webmaster_analytics($data){
        $settings = get_option(LADDER_SEO_SETTINGS);
        $webmaster_fields = isset($settings["webmaster"]) ? $settings["webmaster"] : array();
        foreach ($webmaster_fields as $webmaster_field_name => $webmaster_field_val) {
            $field_name = "ladderseo_webmaster_".$webmaster_field_name;
            $settings["webmaster"][$webmaster_field_name] = isset($data[$field_name]) ? $data[$field_name] : 0;
        }

        $analytics_fields = isset($settings["analytics"]) ? $settings["analytics"] : array();
        foreach ($analytics_fields as $analytics_field_name => $analytics_field_val) {
            $field_name = "ladderseo_analytics_".$analytics_field_name;
            $settings["analytics"][$analytics_field_name] = isset($data[$field_name]) ? $data[$field_name] : 0;
        }

        return update_option(LADDER_SEO_SETTINGS, $settings);
    }

    private function ladderseo_save_bulk_data($data)
    {
        if (isset($data["form_type"]) && $data["form_type"] === "bulk_post") {
            return $this->ladderseo_save_bulk_post_data($data);
        } elseif (isset($data["form_type"]) && $data["form_type"] === "bulk_taxonomy") {
            return $this->ladderseo_save_bulk_taxonomy_data($data);
        }
    }

    private function ladderseo_get_meta_data($data)
    {
        if (isset($data["type"]) && $data["type"] === "ai") {
                $response = array(
                    'message' => esc_html__("Please use PRO version to use this feature", "ladder-seo")
                );
                wp_send_json_error($response);
            }
    }
    

    private function ladderseo_get_title_and_content($data){        
        if (isset($data["type"]) && $data["type"] === "ai") {            
                $response = array(
                    'message' => esc_html__("Please use PRO version to use this feature", "ladder-seo")
                );
            
        }
    }

    

    private function ladderseo_get_high_traffic_keywords($data){
        return $this->ladderseo_get_keyword_data($data);
    }

    private function ladderseo_get_long_tail_keywords($data) {
        return $this->ladderseo_get_keyword_data($data);
    }

    

    private function ladderseo_delete_redirection($data){
        $id = isset($data["id"]) ? $data["id"] : 0;
        $db_manager = new \LadderSEO\DB_Manager();
		return $db_manager->delete_data_redirection($id);
    }

    private function ladderseo_delete_404_logs($data){
        $db_manager = new \LadderSEO\DB_Manager();
		return $db_manager->delete_data_404_logs();
    }

    private function get_prompt($action_type){
        $query = "";
        if($action_type === "ladderseo_get_meta_title"){
            $query = "You are an expert in creating SEO-optimized meta titles. Generate a concise, engaging, and keyword-rich meta title that is under 60 characters, ";
        }
        else if($action_type === "ladderseo_get_meta_description"){ 
            $query = "You are an expert in creating SEO-optimized meta descriptions. Based on the given content, generate a concise and compelling meta description that is under 160 characters. Ensure it includes relevant keywords and motivates users to click on the link. Content: ";
        }
        else if($action_type === "ladderseo_ai_action_generate_title_content"){             

            $query =  "You are an expert content creator and copywriter for WordPress blogs. "
            . "Given a brief description and keywords, generate a compelling and SEO-friendly response in JSON format.\n\n"
            . "- Title Requirements:\n"
            . "  - Catchy and attention-grabbing.\n"
            . "  - Incorporate keywords naturally.\n"
            . "  - Limit title to 60 characters for optimal SEO.\n\n"
            . "- Content Requirements:\n"
            . "  - Well-structured and engaging.\n"
            . "  - Include an introduction, informative body, and conclusion with a call to action.\n"
            . "  - Use natural language and incorporate the provided keywords seamlessly.\n\n"
            . "- Output Format:\n"
            . "  Return the response in the following JSON format:\n"
            . "  {\n"
            . "    \"ladderseo_title\": \"Generated Title\",\n"
            . "    \"ladderseo_content\": \"Generated Content\"\n"
            . "  }\n\n"
            . "- Input:\n";
        }
        else if($action_type === "ladderseo_ai_action_spin"){             

            $query =  "You are an expert copywriter and content creator. Your task is to regenerate (spin) the title and content of a WordPress post while keeping the original meaning intact. Use engaging, fresh, and SEO-friendly language to make the article appealing to a new audience. Follow these guidelines: "
            . "- Title Requirements:\n"
            . "  - Rewrite the title to be attention-grabbing and concise.\n"
            . "  - Incorporate relevant keywords naturally.\n"
            . "  - Limit title to 60 characters for optimal SEO.\n\n"
            . "- Content Requirements:\n"
            . "  - Rewrite the content to maintain the original structure and intent.\n"
            . "  - Use engaging language, avoid repetition, and make the text sound fresh.\n"
            . "  - Incorporate SEO-friendly keywords seamlessly.\n\n"
            . "- Output Format:\n"
            . "  Return the response in the following JSON format:\n"
            . "  {\n"
            . "    \"ladderseo_title\": \"Generated Title\",\n"
            . "    \"ladderseo_content\": \"Generated Content\"\n"
            . "  }\n\n"
            . "- Input:\n";
        }
        else if($action_type === "ladderseo_get_high_traffic_keywords"){ 
            $query = "You are an expert in creating SEO-optimized keywords. Generate a list of top 10 high-traffic SEO keywords related to: ";
        }
        else if($action_type === "ladderseo_get_long_tail_keywords"){ 
            $query = "You are an expert in creating SEO-optimized keywords. Generate a list of top 10 long-tail keywords related to: ";
        }
        
        return $query;
    }

    private function get_prompt_content($data)
    {
        $type = isset($data["content_type"]) ? $data["content_type"] : "post";
        $content  = "";

        if ($type === "post") {
            $post_id = isset($data["post_id"]) ? $data["post_id"] : 0;
            $post = get_post($post_id);

            if ($post) {

                if($data["action_type"] === "ladderseo_ai_action_generate_title_content"){
                    $description = wp_strip_all_tags($data["brief_description"]);
                    $keywords = $data["keywords"];
                    $content = "Brief Description: \"$description\"\n"
                                . "  Keywords: \"$keywords\"";
                    return $content;
                }
                else if($data["action_type"] === "ladderseo_ai_action_spin"){
                    $description = wp_strip_all_tags($data["description"]);
                    $title  = wp_strip_all_tags($data["title"]);
                    $keywords = $data["keywords"];
                    $content = "title: \"$title\"\n"
                                . "  description: \"$description\"\n"
                                . "  Keywords: \"$keywords\"";
                    return $content;
                    
                } 
                
                $local  = $this->helper->is_local_environment();

                if ($local) {//get content
                    $post_title = $post->post_title;
                    $post_content = $post->post_content;
                    $content = "for the following post, 'Post Title': ".$post_title."\n 'Post Content': ".$post_content;
                    return $content;
                } else {//get url
                    $url = get_permalink($post_id);
                    $content = "for the following post at url: " .$url;
                    return $content;
                }
            }
            if(isset($data["ladderseo_keyword"])){
                $content = wp_strip_all_tags($data["ladderseo_keyword"]);
                return $content;
            }
        }
        return $content;
    }

    private function ladderseo_save_bulk_post_data($data)
    {
        foreach ($data as $key => $val) {
            if (strpos($key, 'ladderseo_bulk_') === 0) {
                $field = substr($key, strlen('ladderseo_bulk_'));
                $field_data = explode('-', $field);
                $field_name = isset($field_data[0]) ? $field_data[0] : "";
                $post_type = isset($field_data[1]) ? $field_data[1] : "post";
                $post_id = isset($field_data[2]) ? $field_data[2] : 0;
                update_post_meta($post_id, "ladderseo_".$post_type."-".$field_name, $val);
            }
        }
    }

    private function ladderseo_save_bulk_taxonomy_data($data)
    {
        foreach ($data as $key => $val) {
            if (strpos($key, 'ladderseo_bulk_') === 0) {
                $field = substr($key, strlen('ladderseo_bulk_'));
                $field_data = explode('-', $field);
                $field_name = isset($field_data[0]) ? $field_data[0] : "";
                $post_type = isset($field_data[1]) ? $field_data[1] : "post";
                $post_id = isset($field_data[2]) ? $field_data[2] : 0;
                //this needs to be changed to save data to taxonomy field
                update_post_meta($post_id, "ladderseo_".$post_type."-".$field_name, $val);
            }
        }
    }

    private function parse_content_type_field_name($field_name)
    {
        $pattern = '/^ladderseo_([a-zA-Z0-9_]+)-(.+)$/';

        if (preg_match($pattern, $field_name, $matches)) {
            return [
                'post_type' => $matches[1], // Captures the post type
                'field_name' => $matches[2], // Captures the field name
            ];
        }

        // Return null if the field name doesn't match the expected format
        return [];
    }

    private function get_chart_data($type){
        $tasks_done = 70;
        $task_pending = 100 - $tasks_done;

        $chart_data = [
            "labels" => ["Tasks Done", "Tasks Pending"],
            "datasets" => [
                [
                    "data" => [$tasks_done, $task_pending],
                    "backgroundColor" => ["#3498db", "#e74c3c"]
                ]
            ]
        ];
        return $chart_data;
    }

    /**
     * schedule event
     *
     * @since    1.0.0
     */
    public function ladder_seo_schedule_event()
    {
        $settings = get_option(LADDER_SEO_SETTINGS);

        if (isset($settings["content"])) {
            $data = $settings["content"];
            $generate_content = $data["generate_content"];
            $duration = strtolower($data["duration"]) ;
			if ($generate_content) {
                
				if (! wp_next_scheduled ( 'ladderseo_generate_content' )) {
                    if($duration === "hourly")
                	    wp_schedule_event(time(), 'hourly', 'ladderseo_generate_content');
                    else if($duration === "daily")
                        wp_schedule_event(time(), 'daily', 'ladderseo_generate_content');
                    else if($duration === "weekly")
                        wp_schedule_event(time(), 'weekly', 'ladderseo_generate_content');
                    else if($duration === "monthly")
                        wp_schedule_event(time(), 'monthly', 'ladderseo_generate_content');
                        
				}
            }
        }

    }

    /**
     * Generate content
     *
     * @since    1.0.0
     */

     public function ladder_seo_generate_content()
     {
         $settings = get_option(LADDER_SEO_SETTINGS);         
         $api_key = isset($settings["ai"]["key"]) ? $settings["ai"]["key"] : "";
 
         if (empty($api_key)) {
             $response = array(
                 'message' => esc_html__("No api key found. Please enter api key", "ladder-seo")
             );
             wp_send_json_error($response);
         }
 
     }

}
