<?php
namespace LACI_InternalLinks\Controllers;

use LACI_InternalLinks\Utils\SingletonTrait;

/**
 * @method static ExternalLinksTableController get_instance()
 */
class ExternalLinksTableController extends CreatePostListTableController {

    use SingletonTrait;

    protected $stats;

    public function __construct() {
        parent::__construct();
        $this->stats = WPILCustomTableManager::get_latest_statistics();
    }

    public function get_columns() {
        $columns = [
            'domain'      => $this->get_title( __( 'Domain', 'laci-link-cluster' ) ),
            'total_links' => $this->get_title( __( 'Total Links', 'laci-link-cluster' ) ),
            'posts'       => $this->get_title( __( 'Posts', 'laci-link-cluster' ) ),
            'percentage'  => $this->get_title( __( 'Percentage', 'laci-link-cluster' ) ),
        ];
        return $columns;
    }

    public function get_sortable_columns() {
        $sortable_columns = [
            'domain'      => [ 'domain', false ],
            'total_links' => [ 'total_links', false ],
            'posts'       => [ 'posts', false ],
            'percentage'  => [ 'percentage', false ],
        ];
        return $sortable_columns;
    }

    public function prepare_items() {
        $columns               = $this->get_columns();
        $hidden                = [];
        $sortable              = $this->get_sortable_columns();
        $this->_column_headers = [ $columns, $hidden, $sortable ];

        $data = [];
        if ( ! empty( $this->stats['external_links_data'] ) ) {
            foreach ( $this->stats['external_links_data'] as $domain => $domain_data ) {
                $percentage = $this->stats['total_external_links'] > 0
                    ? ( $domain_data['count'] / $this->stats['total_external_links'] ) * 100
                    : 0;

                $data[] = [
                    'domain'      => $domain,
                    'total_links' => $domain_data['count'],
                    'posts'       => count( $domain_data['posts'] ),
                    'percentage'  => round( $percentage, 1 ),
                ];
            }

            // Xử lý search
            $search = isset( $_REQUEST['s'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['s'] ) ) : '';
            if ( ! empty( $search ) ) {
                $data = array_filter(
                    $data,
                    function ( $item ) use ( $search ) {
                        return stripos( $item['domain'], $search ) !== false;
                    }
                );
            }

            // Xử lý sắp xếp
            $orderby = ( ! empty( $_REQUEST['orderby'] ) ) ? $_REQUEST['orderby'] : 'percentage';
            $order   = ( ! empty( $_REQUEST['order'] ) ) ? $_REQUEST['order'] : 'asc';
            usort( $data, [ $this, 'usort_reorder' ] );

            // Xử lý phân trang
            $per_page     = $this->get_items_per_page( 'external_links_per_page', 10 );
            $current_page = $this->get_pagenum();
            $total_items  = count( $data );

            $this->set_pagination_args(
                [
                    'total_items' => $total_items,
                    'per_page'    => $per_page,
                    'total_pages' => ceil( $total_items / $per_page ),
                ]
            );

            $data = array_slice( $data, ( ( $current_page - 1 ) * $per_page ), $per_page );
        }

        $this->items = $data;
    }

    private function usort_reorder( $a, $b ) {
        // Lấy orderby và order từ request
        $orderby = ( ! empty( $_REQUEST['orderby'] ) ) ? $_REQUEST['orderby'] : 'percentage';
        $order   = ( ! empty( $_REQUEST['order'] ) ) ? $_REQUEST['order'] : 'desc';

        // So sánh dựa trên loại dữ liệu
        switch ( $orderby ) {
            case 'domain':
                $result = strcasecmp( $a[ $orderby ], $b[ $orderby ] );
                break;
            case 'total_links':
            case 'posts':
            case 'percentage':
                $result = $a[ $orderby ] - $b[ $orderby ];
                break;
            default:
                $result = 0;
                break;
        }

        // Đảo ngược kết quả nếu order là desc
        return ( $order === 'asc' ) ? $result : -$result;
    }

    public function column_default( $item, $column_name ) {
        switch ( $column_name ) {
            case 'domain':
                return sprintf(
                    '<span class="laci-domain-cell" data-domain="%s" data-column="%s">%s</span>',
                    esc_attr( $item['domain'] ),
                    'domain',
                    esc_html( $item['domain'] )
                );
            case 'total_links':
                return sprintf(
                    '<a href="%s" class="laci-total-links-cell" data-domain="%s" data-column="%s">%s</a>',
                    '#',
                    esc_attr( $item['domain'] ),
                    'total_links',
                    esc_html( $item['total_links'] )
                );
            case 'posts':
                return sprintf(
                    '<a href="%s" class="laci-posts-cell" data-domain="%s" data-column="%s">%s</a>',
                    '#',
                    esc_attr( $item['domain'] ),
                    'posts',
                    esc_html( $item['posts'] )
                );
            case 'percentage':
                return esc_html( $item['percentage'] . '%' );
            default:
                return print_r( $item, true );
        }
    }

    public function no_items() {
        esc_html_e( 'No external links found.', 'laci-link-cluster' );
    }

    public function get_table_classes() {
        return [ 'widefat', 'fixed', 'striped', $this->_args['plural'], 'laci-external-links-table' ];
    }
}
