<?php
namespace LACI_InternalLinks\Controllers;

use LACI_InternalLinks\Utils\SingletonTrait;

/**
 * @method static ExternalLinksController get_instance()
 */
class ExternalLinksController {
    use SingletonTrait;

    public function __construct() {
        add_action( 'wp_ajax_laci_get_total_links', [ $this, 'get_total_links' ] );
        add_action( 'wp_ajax_laci_get_domain_posts', [ $this, 'get_domain_posts' ] );
        add_action( 'wp_ajax_laci_update_url', [ $this, 'update_url' ] );
    }

    public function get_total_links() {
        check_ajax_referer( 'laci-internal-links-nonce', 'nonce' );

        $domain = isset( $_POST['domain'] ) ? sanitize_text_field( $_POST['domain'] ) : '';
        if ( empty( $domain ) ) {
            wp_send_json_error();
        }

        $links_data = $this->get_total_links_data( $domain );
        $html       = '<table><thead><tr><th>URL</th><th>Count</th></tr></thead><tbody>';
        foreach ( $links_data as $link ) {
            $html .= '<tr><td>' . $link['url'] . '</td><td>' . $link['count'] . '</td></tr>';
        }
        $html .= '</tbody></table>';
        wp_send_json_success(
            [
                'html'  => $html,
                'title' => $domain,
            ]
        );
    }

    public function get_domain_posts() {
        check_ajax_referer( 'laci-internal-links-nonce', 'nonce' );

        $domain = isset( $_POST['domain'] ) ? sanitize_text_field( $_POST['domain'] ) : '';
        if ( empty( $domain ) ) {
            wp_send_json_error();
        }

        $posts_data = $this->get_domain_posts_data( $domain );
        $html       = '<table><thead><tr><th>Title</th><th>Links Count</th><th>Edit</th></tr></thead><tbody>';
        foreach ( $posts_data as $post ) {
            $html .= '<tr><td>' . $post['title'] . '</td><td>' . $post['links_count'] . '</td><td><a target="_blank" class="button" href="' . $post['edit_url'] . '">Edit</a></td></tr>';
        }
        $html .= '</tbody></table>';
        wp_send_json_success(
            [
                'html'  => $html,
                'title' => $domain,
            ]
        );
    }

    public function update_url() {
        check_ajax_referer( 'laci-internal-links-nonce', 'nonce' );

        $old_url = isset( $_POST['old_url'] ) ? esc_url_raw( $_POST['old_url'] ) : '';
        $new_url = isset( $_POST['new_url'] ) ? esc_url_raw( $_POST['new_url'] ) : '';

        if ( empty( $old_url ) || empty( $new_url ) ) {
            wp_send_json_error();
        }

        $this->update_url_in_posts( $old_url, $new_url );
        wp_send_json_success();
    }

    private function get_total_links_data( $domain ) {
        $stats = WPILCustomTableManager::get_latest_statistics();
        if ( empty( $stats['external_links_data'][ $domain ] ) ) {
            return [];
        }

        $domain_data = $stats['external_links_data'][ $domain ];
        $links_data  = [];

        foreach ( $domain_data['urls'] as $url ) {
            $links_data[] = [
                'url'   => $url,
                'count' => $this->count_url_occurrences( $url ),
            ];
        }

        return $links_data;
    }

    private function get_domain_posts_data( $domain ) {
        $stats = WPILCustomTableManager::get_latest_statistics();
        if ( empty( $stats['external_links_data'][ $domain ] ) ) {
            return [];
        }

        $domain_data = $stats['external_links_data'][ $domain ];
        $posts_data  = [];

        foreach ( $domain_data['posts'] as $post_id ) {
            $post         = get_post( $post_id );
            $posts_data[] = [
                'title'       => $post->post_title,
                'links_count' => $this->count_domain_links_in_post( $domain, $post->ID ),
                'edit_url'    => get_edit_post_link( $post->ID, 'raw' ),
            ];
        }

        return $posts_data;
    }

    private function count_url_occurrences( $url ) {
        global $wpdb;
        $count = 0;

        $posts = get_posts(
            [
                'post_type'      => 'post',
                'post_status'    => 'publish',
                'posts_per_page' => -1,
            ]
        );

        foreach ( $posts as $post ) {
            $content = $post->post_content;
            $count  += substr_count( $content, $url );
        }

        return $count;
    }

    private function count_domain_links_in_post( $domain, $post_id ) {
        $post    = get_post( $post_id );
        $content = $post->post_content;
        $count   = 0;

        preg_match_all( '/<a\s[^>]*href=["\']([^"\']+)["\'][^>]*>/i', $content, $matches );

        if ( ! empty( $matches[1] ) ) {
            foreach ( $matches[1] as $url ) {
                $url_domain = parse_url( $url, PHP_URL_HOST );
                if ( $url_domain === $domain ) {
                    ++$count;
                }
            }
        }

        return $count;
    }

    private function update_url_in_posts( $old_url, $new_url ) {
        $posts = get_posts(
            [
                'post_type'      => 'post',
                'post_status'    => 'publish',
                'posts_per_page' => -1,
            ]
        );

        foreach ( $posts as $post ) {
            $content = $post->post_content;

            // Thay thế URL trong content
            $updated_content = str_replace( $old_url, $new_url, $content );

            if ( $content !== $updated_content ) {
                wp_update_post(
                    [
                        'ID'           => $post->ID,
                        'post_content' => $updated_content,
                    ]
                );
            }
        }
    }
}
