<?php
namespace LACI_InternalLinks\Controllers;

use LACI_InternalLinks\Utils\SingletonTrait;

/**
 * @method static BrokenLinksTableController get_instance()
 */
class BrokenLinksTableController extends CreatePostListTableController {

    use SingletonTrait;

    protected $stats;

    public function __construct() {
        parent::__construct();
        $this->stats = WPILCustomTableManager::get_latest_statistics();
    }

    public function get_columns() {
        $columns = [
            'url'         => $this->get_title( __( 'URL', 'laci-link-cluster' ) ),
            'total_links' => $this->get_title( __( 'Total Links', 'laci-link-cluster' ) ),
            'posts'       => $this->get_title( __( 'Posts', 'laci-link-cluster' ) ),
            'status'      => $this->get_title( __( 'Status', 'laci-link-cluster' ) ),
        ];
        return $columns;
    }

    public function get_sortable_columns() {
        $sortable_columns = [
            'url'         => [ 'url', true ],
            'total_links' => [ 'total_links', true ],
            'posts'       => [ 'posts', true ],
            'status'      => [ 'status', true ],
        ];
        return $sortable_columns;
    }

    public function prepare_items() {
        $columns               = $this->get_columns();
        $hidden                = [];
        $sortable              = $this->get_sortable_columns();
        $this->_column_headers = [ $columns, $hidden, $sortable ];

        $data = [];
        if ( ! empty( $this->stats['broken_links_data'] ) ) {
            foreach ( $this->stats['broken_links_data'] as $url => $url_data ) {
                $data[] = [
                    'url'         => $url,
                    'total_links' => $url_data['count'],
                    'posts'       => count( $url_data['posts'] ),
                    'status'      => $url_data['status'],
                ];
            }

            // Xử lý search
            $search = isset( $_REQUEST['s'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['s'] ) ) : '';
            if ( ! empty( $search ) ) {
                $data = array_filter(
                    $data,
                    function ( $item ) use ( $search ) {
                        return stripos( $item['url'], $search ) !== false;
                    }
                );
            }

            // Sắp xếp theo số lượng link giảm dần
            usort(
                $data,
                function ( $a, $b ) {
                    return $b['total_links'] - $a['total_links'];
                }
            );

            // Xử lý phân trang
            $per_page     = $this->get_items_per_page( 'broken_links_per_page', 10 );
            $current_page = $this->get_pagenum();
            $total_items  = count( $data );

            $this->set_pagination_args(
                [
                    'total_items' => $total_items,
                    'per_page'    => $per_page,
                    'total_pages' => ceil( $total_items / $per_page ),
                ]
            );

            $data = array_slice( $data, ( ( $current_page - 1 ) * $per_page ), $per_page );
        }

        $this->items = $data;
    }

    public function column_default( $item, $column_name ) {
        switch ( $column_name ) {
            case 'url':
                return sprintf(
                    '<span class="broken-url-cell">%s</span>',
                    esc_html( $item['url'] )
                );
            case 'total_links':
                return sprintf(
                    '<span class="total-links-cell">%s</span>',
                    esc_html( $item['total_links'] )
                );
            case 'posts':
                return sprintf(
                    '<span class="posts-cell">%s</span>',
                    esc_html( $item['posts'] )
                );
            case 'status':
                $status_class = $this->get_status_class( $item['status'] );
                return sprintf(
                    '<span class="status-cell %s">%s</span>',
                    esc_attr( $status_class ),
                    esc_html( $item['status'] )
                );
            default:
                return print_r( $item, true );
        }
    }

    private function get_status_class( $status ) {
        switch ( $status ) {
            case '404':
                return 'status-404';
            case '500':
                return 'status-500';
            case 'timeout':
                return 'status-timeout';
            default:
                return 'status-unknown';
        }
    }

    public function no_items() {
        esc_html_e( 'No broken links found.', 'laci-link-cluster' );
    }

    public function get_table_classes() {
        return [ 'widefat', 'fixed', 'striped', $this->_args['plural'], 'laci-broken-links-table' ];
    }
}
