<?php
/**
 * Plugin Name: Kotia Content Integration
 * Plugin URI: https://github.com/Serg25001/kotia-content-integration
 * Description: Securely import posts, media, categories, tags, and featured images into WordPress via a REST API using token-based authentication.
 * Version: 2.2.0
 * Author: Serg25001
 * Author URI: https://github.com/Serg25001
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: kotia-content-integration
 * Domain Path: /languages
 */

defined( 'ABSPATH' ) || exit;

final class Kotia_Content_Integration {

    const OPTION_TOKEN_HASH = 'kotia_token_hash';

    /* ================= INIT ================= */
    public static function init() {
        add_action( 'rest_api_init', [ self::class, 'register_routes' ] );
        add_action( 'admin_menu', [ self::class, 'admin_menu' ] );
    }

    /* ================= ACTIVATION ================= */
    public static function activate() {
        if ( ! get_option( self::OPTION_TOKEN_HASH ) ) {
            self::generate_token();
        }
    }

    /* ================= TOKEN ================= */
    private static function generate_token(): string {
        $token = bin2hex( random_bytes( 32 ) );
        update_option( self::OPTION_TOKEN_HASH, hash( 'sha256', $token ), false );
        return $token;
    }

    private static function validate_token( string $token ): bool {
        $stored = get_option( self::OPTION_TOKEN_HASH );
        return $stored && hash_equals( $stored, hash( 'sha256', $token ) );
    }


    /* ================= AUTH ================= */
    public static function permission_callback( \WP_REST_Request $request ) {
        $header = $request->get_header( 'authorization' );

        if ( ! $header || strpos( $header, 'Bearer ' ) !== 0 ) {
            return new \WP_Error(
                'kotia_unauthorized',
                __( 'Missing token.', 'kotia-content-integration' ),
                [ 'status' => 401 ]
            );
        }

        if ( ! self::validate_token( substr( $header, 7 ) ) ) {
            return new \WP_Error(
                'kotia_forbidden',
                __( 'Invalid token.', 'kotia-content-integration' ),
                [ 'status' => 403 ]
            );
        }

        return true;
    }


    /* ================= ROUTES ================= */
    public static function register_routes() {

        $routes = [
            '/post'          => [ 'create_post' ],
            '/media'         => [ 'upload_media' ],
            '/category'      => [ 'create_category' ],
            '/tag'           => [ 'create_tag' ],
            '/set-thumbnail' => [ 'set_thumbnail' ],
        ];

        foreach ( $routes as $route => $callback ) {
            register_rest_route(
                'kotia-content-integration/v1',
                $route,
                [
                    'methods'             => \WP_REST_Server::CREATABLE,
                    'callback'            => [ self::class, $callback[0] ],
                    'permission_callback' => [ self::class, 'permission_callback' ],
                ]
            );
        }
    }

    /* ================= POST ================= */
    public static function create_post( \WP_REST_Request $r ) {

        $data = $r->get_json_params();

        if ( empty( $data['title'] ) || empty( $data['content'] ) ) {
            return new \WP_Error(
                'kotia_missing_required_fields',
                __( 'Both title and content are required.', 'kotia-content-integration' ),
                [ 'status' => 400 ]
            );
        }

        $post_status = sanitize_key( $data['status'] ?? 'publish' );

        $post_id = wp_insert_post(
            [
                'post_title'   => wp_strip_all_tags( $data['title'] ),
                'post_content' => $data['content'],
                'post_status'  => $post_status,
                'post_type'    => 'post',
                'post_date'    => $data['date'] ?? current_time( 'mysql' ),
            ],
            true
        );

        if ( is_wp_error( $post_id ) ) {
            return $post_id;
        }

        if ( ! empty( $data['categories'] ) ) {
            wp_set_post_terms( $post_id, array_map( 'intval', $data['categories'] ), 'category' );
        }

        if ( ! empty( $data['tags'] ) ) {
            wp_set_post_terms( $post_id, array_map( 'intval', $data['tags'] ), 'post_tag' );
        }

        return [ 'post_id' => (int) $post_id ];
    }

    /* ================= MEDIA ================= */
    public static function upload_media( \WP_REST_Request $request ) {

        $file = $request->get_file_params()['file'] ?? null;

        if ( ! $file ) {
            return new \WP_Error(
                'kotia_no_file_provided',
                __( 'No file was provided.', 'kotia-content-integration' ),
                [ 'status' => 400 ]
            );
        }

        require_once ABSPATH . 'wp-admin/includes/file.php';
        require_once ABSPATH . 'wp-admin/includes/media.php';
        require_once ABSPATH . 'wp-admin/includes/image.php';

        $upload = wp_handle_upload( $file, [ 'test_form' => false ] );

        if ( ! is_array( $upload ) || isset( $upload['error'] ) ) {
            return new \WP_Error(
                'kotia_upload_error',
                __( 'Upload failed.', 'kotia-content-integration' ),
                [ 'status' => 400 ]
            );
        }

        $allowed_types = [ 'image/jpeg', 'image/png', 'image/webp', 'image/gif' ];

        if ( empty( $upload['type'] ) || ! in_array( $upload['type'], $allowed_types, true ) ) {
            if ( ! empty( $upload['file'] ) ) {
    wp_delete_file( $upload['file'] );
}

            return new \WP_Error(
                'kotia_invalid_file_type',
                __( 'Invalid file type.', 'kotia-content-integration' ),
                [ 'status' => 400 ]
            );
        }

        $id = wp_insert_attachment(
            [
                'post_mime_type' => $upload['type'],
                'post_title'     => basename( $upload['file'] ),
                'post_status'    => 'inherit',
            ],
            $upload['file']
        );

        wp_update_attachment_metadata( $id, wp_generate_attachment_metadata( $id, $upload['file'] ) );

        return [
            'attachment_id' => (int) $id,
            'url'           => wp_get_attachment_url( $id ),
        ];
    }

    /* ================= TAXONOMY ================= */
    public static function create_category( \WP_REST_Request $r ) {
       

        $name = sanitize_text_field( $r['name'] ?? '' );

        if ( empty( $name ) ) {
            return new \WP_Error(
                'kotia_missing_name',
                __( 'Name is required.', 'kotia-content-integration' ),
                [ 'status' => 400 ]
            );
        }

        $term = wp_insert_term( $name, 'category' );
        return is_wp_error( $term ) ? $term : [ 'id' => $term['term_id'] ];
    }

    public static function create_tag( \WP_REST_Request $r ) {
  

        $name = sanitize_text_field( $r['name'] ?? '' );

        if ( empty( $name ) ) {
            return new \WP_Error(
                'kotia_missing_name',
                __( 'Name is required.', 'kotia-content-integration' ),
                [ 'status' => 400 ]
            );
        }

        $term = wp_insert_term( $name, 'post_tag' );
        return is_wp_error( $term ) ? $term : [ 'id' => $term['term_id'] ];
    }

    /* ================= THUMBNAIL ================= */
    public static function set_thumbnail( \WP_REST_Request $r ) {
      
        set_post_thumbnail( (int) $r['post_id'], (int) $r['attachment_id'] );
        return [ 'ok' => true ];
    }

    /* ================= ADMIN ================= */
    public static function admin_menu() {
        add_options_page(
            __( 'Kotia Content Integration', 'kotia-content-integration' ),
            __( 'Kotia Content Integration', 'kotia-content-integration' ),
            'manage_options',
            'kotia-content-integration',
            [ self::class, 'admin_page' ]
        );
    }

    public static function admin_page() {

        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }

        if ( isset( $_POST['regenerate'] ) ) {

            $nonce = isset( $_POST['_wpnonce'] )
                ? sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) )
                : '';

            if ( ! wp_verify_nonce( $nonce, 'kotia_regenerate' ) ) {
                wp_die( esc_html__( 'Nonce verification failed.', 'kotia-content-integration' ) );
            }

            $token = self::generate_token();

            echo '<div class="notice notice-success">';
            echo '<p><strong>' . esc_html__( 'Save token now:', 'kotia-content-integration' ) . '</strong></p>';
            echo '<code>' . esc_html( $token ) . '</code>';
            echo '</div>';
        }

        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'Kotia Content Integration', 'kotia-content-integration' ); ?></h1>
            <form method="post">
                <?php wp_nonce_field( 'kotia_regenerate' ); ?>
                <button type="submit" name="regenerate" class="button button-primary">
                    <?php esc_html_e( 'Generate new token', 'kotia-content-integration' ); ?>
                </button>
            </form>
            <p><code><?php echo esc_url( rest_url( 'kotia-content-integration/v1/' ) ); ?></code></p>
        </div>
        <?php
    }
}

/* ================= BOOTSTRAP ================= */
register_activation_hook( __FILE__, [ 'Kotia_Content_Integration', 'activate' ] );
Kotia_Content_Integration::init();