<?php
/*
Plugin Name: kombat-optimizer
Plugin URI: https://webdesign-profesional.com/kombat-optimizer
Description: Optimizare automată WebP, preload inteligent și monitorizare TTFB. Include cron săptămânal și scanare pe foldere.
Version: 1.2.3
Author: Alin
Author URI: https://webdesign-profesional.com
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: kombat-optimizer
*/

defined('ABSPATH') or die('Nu ai voie aici.');

// 🔧 Include module externe
require_once plugin_dir_path(__FILE__) . 'inc/generate-webp.php';
require_once plugin_dir_path(__FILE__) . 'inc/webp-cron.php';
require_once plugin_dir_path(__FILE__) . 'inc/optimize-webp.php';
require_once plugin_dir_path(__FILE__) . 'inc/scan-webp.php';
require_once plugin_dir_path(__FILE__) . 'inc/monitor-ttfb.php';
require_once plugin_dir_path(__FILE__) . 'inc/log.php';

// 🖼️ Include înlocuire imagini doar dacă opțiunea e activă
if (get_option('kombatoptimizer_replace_images')) {
    require_once plugin_dir_path(__FILE__) . 'inc/replace-webp.php';
}

// 🔧 Adaugă meniu în admin
add_action('admin_menu', function() {
    add_menu_page(
        'KombatOptimizer',
        'KombatOptimizer',
        'manage_options',
        'kombatoptimizer',
        'kombatoptimizer_panel',
        'dashicons-admin-generic'
    );
});

// 🔧 Panou administrare


function kombatoptimizer_panel() {
    echo '<div class="wrap"><h2>KombatOptimizer</h2>';

    echo '<div class="kombat-tabs">';
    echo '<div class="kombat-tab-buttons">';
    echo '<button data-target="tab-generator">🔧 Generator WebP</button>';
    echo '<button data-target="tab-replace">🖼️ Înlocuire imagini</button>';
    echo '<button data-target="tab-scan">🔍 Scanare folder</button>';
    echo '<button data-target="tab-ttfb">📊 TTFB</button>';
    echo '<button data-target="tab-cron">🕒 Cron WebP</button>';
    echo '</div>';

// 🔍 Pentru scanare: listăm toate folderele din uploads/, excluzând cele sensibile
$upload_dir = wp_upload_dir();
$base_dir = trailingslashit($upload_dir['basedir']);

$folders = array_filter(glob($base_dir . '*'), function($folder) {
    $name = basename($folder);
    return is_dir($folder) && !preg_match('/(cache|tmp|woocommerce)/i', $name);
});

// 💾 Pentru salvare: ne asigurăm că folderul kombatoptimizer există
$plugin_dir = $base_dir . 'kombatoptimizer/';
if (!file_exists($plugin_dir)) {
    wp_mkdir_p($plugin_dir);
}

$folders = array_filter(glob($base_dir . '*'), 'is_dir');

// 🔧 Tab 1: Generator WebP

echo '<div id="tab-generator" class="kombat-tab-content">';
echo '<h3>🔧 Generator & Optimizare WebP</h3>';
echo '<form method="post">';
wp_nonce_field('kombatoptimizer_webp_action'); // ✅ nonce pentru securitate

echo '<p><label for="webp_folder"><strong>Selectează folderul:</strong></label><br>';
echo '<select name="webp_folder" id="webp_folder">';
foreach (get_year_folders() as $folder) {
    $name = str_replace($base_dir, '', $folder);
    echo '<option value="' . esc_attr($name) . '">' . esc_html($name) . '</option>';
}
echo '</select> </p>';

echo '<p>';
echo '<button type="submit" name="generate_webp" class="button button-primary">Generează WebP</button> ';
echo '<button type="submit" name="optimize_webp" class="kombat-btn">Optimizează imaginile WebP</button>';
echo '</p>';

echo '<div id="kombat-loader">';
echo '<div class="spinner"></div>';
echo '<p>Se procesează imaginile... te rugăm să aștepți.</p>';
echo '</div>';


echo '</form>';


// ℹ️ Mesaj explicativ permanent sub butoane
echo '<div id="kombat-info-box" class="notice notice-info" style="margin-top:20px; position:relative;">';
echo '<button type="button" id="close-info-box" style="position:absolute; top:8px; right:10px; background:none; border:none; font-size:16px; cursor:pointer;">✖</button>';
echo '<p><strong>ℹ️ Cum funcționează:</strong></p>';
echo '<ul style="margin-left:20px;">';
echo '<li>Se afișează doar folderele de tip <code>uploads/2025</code>, <code>uploads/2024</code> etc. — cele care conțin imagini media.</li>';
echo '<li>Folderele interne ale pluginurilor (ex: <code>rank_math</code>, <code>kombatoptimizer</code>) sunt automat excluse.</li>';
echo '<li>Se procesează maxim <strong>150 imagini</strong> per folder, pentru a evita blocarea serverului.</li>';
echo '<li>Imaginile deja convertite în WebP sunt ignorate automat.</li>';
echo '</ul>';
echo '<p>Pentru rezultate optime, selectează folderul corespunzător anului în care au fost încărcate imaginile.</p>';
echo '</div>';
echo '<p id="kombat-info-restore" style="margin-top:10px;">';
echo '<a href="#" style="text-decoration:none;" onclick="return false;">🔄 Afișează din nou explicația</a>';
echo '</p>';

// 🔁 Procesare acțiuni POST
if (
    isset($_SERVER['REQUEST_METHOD']) &&
    $_SERVER['REQUEST_METHOD'] === 'POST' &&
    isset($_POST['webp_folder']) &&
    check_admin_referer('kombatoptimizer_webp_action')
) {
    $selected_folder = sanitize_text_field(wp_unslash($_POST['webp_folder'])); // ✅ unslash + sanitize

    // ✅ Generare WebP
if (isset($_POST['generate_webp']) && function_exists('generate_webp_from_folder')) {
    $report = generate_webp_from_folder($selected_folder);

    echo '<div class="updated">';
    echo '<h4>✅ Generare WebP finalizată</h4>';
    echo '<p><strong>Folder:</strong> ' . esc_html($selected_folder) . '</p>';
    echo '<p><strong>Imagini convertite:</strong> ' . esc_html($report['converted']) . '</p>';
    echo '<p><strong>Imagini ignorate:</strong> ' . esc_html($report['skipped']) . '</p>';
  echo '<p><strong>Generat la:</strong> ' . esc_html( current_time('H:i') ) . '</p>';

    echo '</div>';
}


    // ✅ Optimizare WebP
    if (isset($_POST['optimize_webp']) && function_exists('optimize_webp_images')) {
        $report = optimize_webp_images($selected_folder);

        if (is_array($report)) {
            echo '<div class="updated">';
            echo '<h4>✅ Optimizare WebP completă</h4>';
            echo '<p><strong>Folder:</strong> ' . esc_html($selected_folder) . '</p>';
            echo '<p><strong>Imagini optimizate:</strong> ' . esc_html($report['optimized']) . '</p>';
            echo '<p><strong>Imagini ignorate:</strong> ' . esc_html($report['skipped']) . '</p>';
            echo '<p><strong>Finalizat la:</strong> ' . esc_html( current_time('H:i') ) . '</p>';

            echo '</div>';
        } else {
            echo '<div class="error">';
            echo '<p>❌ Eroare la optimizarea imaginilor WebP în folderul <strong>' . esc_html($selected_folder) . '</strong>.</p>';
            echo '</div>';
        }
    }
}

echo '</div>';


 // 🖼️ Tab 2: Înlocuire imagini
echo '<div id="tab-replace" class="kombat-tab-content">';

if (
    isset($_POST['kombat_enable_webp_replace']) &&
    check_admin_referer('kombatoptimizer_replace_action')
) {
    update_option('kombatoptimizer_replace_images', true);
    echo '<div class="updated"><p>✅ Înlocuirea imaginilor a fost <strong>activată</strong>.</p></div>';
}

if (
    isset($_POST['kombat_disable_webp_replace']) &&
    check_admin_referer('kombatoptimizer_replace_action')
) {
    update_option('kombatoptimizer_replace_images', false);
    echo '<div class="updated"><p>🛑 Înlocuirea imaginilor a fost <strong>dezactivată</strong>.</p></div>';
}

$replace_status = get_option('kombatoptimizer_replace_images');

echo '<h3>🖼️ Înlocuire imagini cu WebP</h3>';
echo $replace_status
    ? '<p style="color:green;"><strong>✅ Înlocuirea imaginilor este activă.</strong></p>'
    : '<p style="color:red;"><strong>🛑 Înlocuirea imaginilor este dezactivată.</strong></p>';

echo '<form method="post" style="display:inline-block;">';
wp_nonce_field('kombatoptimizer_replace_action');
echo '<input type="hidden" name="kombat_enable_webp_replace" value="1">';
echo '<button type="submit" class="kombat-btn">Activează</button>';
echo '</form>';

echo '<form method="post" style="display:inline-block; margin-left:10px;">';
wp_nonce_field('kombatoptimizer_replace_action');
echo '<input type="hidden" name="kombat_disable_webp_replace" value="1">';
echo '<button type="submit" class="kombat-btn">Dezactivează</button>';
echo '</form>';

echo '</div>';

   // 🔍 Tab 3: Scanare folder
   
echo '<div id="tab-scan" class="kombat-tab-content">';
echo '<h3>🔍 Scanare detaliată folder</h3>';
echo '<form method="post">';
wp_nonce_field('kombatoptimizer_scan_action'); // ✅ nonce pentru securitate

echo '<select name="scan_folder">';
foreach (get_year_folders() as $folder) {
    $name = str_replace($base_dir, '', $folder);
    echo '<option value="' . esc_attr($name) . '">' . esc_html($name) . '</option>';
}
echo '</select> ';
echo '<button type="submit" name="scan_webp" class="kombat-btn">Scanează</button>';
echo '</form>';

if (
    isset($_POST['scan_webp']) &&
    isset($_POST['scan_folder']) &&
    check_admin_referer('kombatoptimizer_scan_action') // ✅ verificare nonce
) {
    $folder = sanitize_text_field(wp_unslash($_POST['scan_folder'])); // ✅ unslash + sanitize
    $scan = kombatoptimizer_scan_webp_folder($folder);

    echo '<div class="updated"><p><strong>📁 ' . esc_html($folder) . '</strong></p>';
    echo '<p>🖼️ Total imagini: ' . esc_html($scan['total']) . '</p>';
    echo '<p>✅ WebP valide: ' . esc_html($scan['webp']) . '</p>';
    echo '<p>❌ WebP lipsă: ' . esc_html($scan['missing']) . '</p>';
    echo '<p>⚠️ WebP corupte: ' . esc_html($scan['corrupt']) . '</p>';
    echo '<p>📊 Acoperire: ' . esc_html($scan['percent']) . '%</p></div>';
}

echo '</div>';



    // 📊 Tab 4: TTFB
echo '<div id="tab-ttfb" class="kombat-tab-content">';
echo '<h3>📊 Monitorizare TTFB</h3>';

$ttfb = kombatoptimizer_measure_ttfb(home_url());

if (
    isset($_POST['kombat_refresh_ttfb']) &&
    check_admin_referer('kombatoptimizer_ttfb_action') // ✅ verificare nonce
) {
    $ttfb = kombatoptimizer_measure_ttfb(home_url());
    echo '<div class="kombat-success"><p>✅ TTFB reîmprospătat: <strong>' . esc_html(round($ttfb, 2)) . ' ms</strong></p></div>';
}

echo '<p><strong>TTFB homepage:</strong> ' . esc_html(round($ttfb, 2)) . ' ms</p>';

echo '<form method="post">';
wp_nonce_field('kombatoptimizer_ttfb_action'); // ✅ nonce pentru protecție
echo '<input type="submit" name="kombat_refresh_ttfb" class="kombat-btn" value="Reîmprospătează">';
echo '</form>';

echo '</div>';

   // 🕒 Tab 5: Cron WebP
echo '<div id="tab-cron" class="kombat-tab-content">';

if (
    isset($_POST['kombat_enable_webp_cron']) &&
    check_admin_referer('kombatoptimizer_cron_action')
) {
    update_option('kombatoptimizer_webp_cron_enabled', true);
    wp_clear_scheduled_hook('kombatoptimizer_weekly_webp_cron');
    wp_schedule_event(time(), 'weekly', 'kombatoptimizer_weekly_webp_cron');
    echo '<div class="updated"><p>✅ Cronul săptămânal a fost <strong>activat</strong>.</p></div>';
}

if (
    isset($_POST['kombat_disable_webp_cron']) &&
    check_admin_referer('kombatoptimizer_cron_action')
) {
    update_option('kombatoptimizer_webp_cron_enabled', false);
    wp_clear_scheduled_hook('kombatoptimizer_weekly_webp_cron');
    echo '<div class="updated"><p>🛑 Cronul săptămânal a fost <strong>dezactivat</strong>.</p></div>';
}

$cron_status = get_option('kombatoptimizer_webp_cron_enabled');
$last_run    = get_option('kombatoptimizer_last_cron_run');

echo '<h3>🕒 Generare automată WebP (cron săptămânal)</h3>';
echo $cron_status
    ? '<p style="color:green;"><strong>✅ Cronul este activ.</strong></p>'
    : '<p style="color:red;"><strong>🛑 Cronul este dezactivat.</strong></p>';

if ($last_run) {
    echo '<p><strong>Ultima rulare:</strong> ' . esc_html($last_run) . '</p>';
}

echo '<form method="post" style="display:inline-block;">';
wp_nonce_field('kombatoptimizer_cron_action');
echo '<input type="hidden" name="kombat_enable_webp_cron" value="1">';
echo '<button type="submit" class="kombat-btn">Activează cronul</button></form>';

echo '<form method="post" style="display:inline-block; margin-left:10px;">';
wp_nonce_field('kombatoptimizer_cron_action');
echo '<input type="hidden" name="kombat_disable_webp_cron" value="1">';
echo '<button type="submit" class="kombat-btn">Dezactivează cronul</button></form>';

echo '</div>'; // end tab-cron
echo '</div>'; // end .kombat-tabs
echo '</div>'; // end .wrap

    
    
}
add_action('admin_enqueue_scripts', 'kombatoptimizer_enqueue_admin_assets');

function kombatoptimizer_enqueue_admin_assets($hook) {
    // ✅ Încarcă doar pe pagina pluginului
 $page = isset($_GET['page']) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';

if (strpos($page, 'kombatoptimizer') === false) {
    return;
}


    $plugin_url = plugin_dir_url(__FILE__);

    wp_enqueue_style('kombatoptimizer-admin-style', $plugin_url . 'assets/admin.css', [], '1.2.1');
    wp_enqueue_script('kombatoptimizer-admin-script', $plugin_url . 'assets/admin.js', ['jquery'], '1.2.1', true);
}


