<?php
// 🔒 Protecție directă
if (!defined('ABSPATH')) exit;

/**
 * ⚙️ Optimizează imaginile WebP din folderul kombatoptimizer/{subfolder}
 *
 * @param string $subfolder Folder relativ din uploads/ (ex: "2025/10")
 * @return array ['optimized' => X, 'skipped' => Y]
 */
function optimize_webp_images($subfolder) {
    // 📁 Obține calea completă către uploads/kombatoptimizer/{subfolder}
    $upload_dir = wp_upload_dir();
    $base_dir   = trailingslashit($upload_dir['basedir']) . 'kombatoptimizer/';
    $target_dir = trailingslashit($base_dir) . $subfolder;

    // ❌ Verifică dacă folderul există
    if (!is_dir($target_dir)) {
        if (function_exists('kombatoptimizer_log')) {
            kombatoptimizer_log("Folder invalid pentru optimizare: " . $subfolder);
        }
        return ['optimized' => 0, 'skipped' => 0];
    }

    // 📊 Inițializare contorizare
    $optimized = 0;
    $skipped   = 0;

    // 🔁 Scanare recursivă în folderul target
    $rii = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($target_dir));
    foreach ($rii as $file) {
        if ($file->isDir()) continue;

        // 🎯 Verifică extensia .webp
        if (!preg_match('/\.webp$/i', $file->getFilename())) continue;

        $webp_path = $file->getPathname();

        // 🧪 Încearcă să încarce imaginea
        $image = @imagecreatefromwebp($webp_path);
        if (!$image) {
            $skipped++;
            continue;
        }

        // ⚙️ Re-scrie imaginea cu compresie 80%
        imagewebp($image, $webp_path, 80);
        imagedestroy($image);
        $optimized++;
    }

    // 📝 Log optional
    if (function_exists('kombatoptimizer_log')) {
        kombatoptimizer_log("WebP optimizat în $subfolder: $optimized imagini, $skipped ignorate");
    }

    // 📦 Returnează raportul
    return ['optimized' => $optimized, 'skipped' => $skipped];
}