<?php
if (!defined('ABSPATH')) exit;

// 🔧 Generează WebP din folderul selectat, cu limită de procesare
function generate_webp_from_folder($subfolder, $batch_size = 150, $max_seconds = 10) {
    $upload_dir = wp_upload_dir();
    $source_dir = trailingslashit($upload_dir['basedir']) . $subfolder;
    $target_dir = trailingslashit($upload_dir['basedir']) . 'kombatoptimizer/' . $subfolder;

    if (!is_dir($source_dir)) {
        kombatoptimizer_log("Folder sursă invalid: $source_dir");
        return ['converted' => 0, 'skipped' => 0];
    }

    if (!file_exists($target_dir)) {
        wp_mkdir_p($target_dir);
    }

    $rii = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($source_dir));
    $all_images = [];

    foreach ($rii as $file) {
        if ($file->isDir()) continue;
        $path = $file->getPathname();
        if (preg_match('/\.(jpg|jpeg|png)$/i', $path)) {
            $all_images[] = $path;
        }
    }

    $converted = 0;
    $skipped   = 0;
    $total     = count($all_images);
    $offset    = 0;
    $start_time = time();

    while ($offset < $total) {
        $batch = array_slice($all_images, $offset, $batch_size);

        foreach ($batch as $image_path) {
            // ⏱️ Verifică dacă am depășit timpul maxim
            if ((time() - $start_time) >= $max_seconds) {
                kombatoptimizer_log("Timpul maxim de $max_seconds secunde a fost atins.");
                break 2; // iese din ambele bucle
            }

            $relative_path = str_replace($source_dir, '', $image_path);
            $webp_path = $target_dir . $relative_path;
            $webp_path = preg_replace('/\.(jpg|jpeg|png)$/i', '.webp', $webp_path);

            $webp_folder = dirname($webp_path);
            if (!file_exists($webp_folder)) {
                wp_mkdir_p($webp_folder);
            }

            if (file_exists($webp_path)) {
                $skipped++;
                continue;
            }

            if (preg_match('/\.png$/i', $image_path)) {
                $image = @imagecreatefrompng($image_path);
                if (!$image) { $skipped++; continue; }
                if (!imageistruecolor($image)) imagepalettetotruecolor($image);
                imagealphablending($image, false);
                imagesavealpha($image, true);
            } else {
                $image = @imagecreatefromstring(file_get_contents($image_path));
                if (!$image) { $skipped++; continue; }
            }

            imagewebp($image, $webp_path, 85);

            if (filesize($webp_path) < 100) {
                wp_delete_file($webp_path);
                $skipped++;
                kombatoptimizer_log("WebP corupt șters: $webp_path");
            } else {
                $converted++;
            }

            imagedestroy($image);
        }

        $offset += $batch_size;
    }

    kombatoptimizer_log("Generare WebP din '$subfolder': convertite=$converted, ignorate=$skipped");
    return ['converted' => $converted, 'skipped' => $skipped];
}


/**
 * 🔍 Returnează doar folderele de tip an (ex: 2025, 2024) din wp-content/uploads/
 * Ignoră folderele non-media (kombatoptimizer, rank_math, etc.)
 */
function get_year_folders() {
    $upload_dir = wp_upload_dir()['basedir'];
    $excluded = ['kombatoptimizer', 'rank_math', 'elementor', 'wpforms', 'woocommerce', 'siteorigin'];

    // 📁 Obține toate subfolderele din uploads/
    $folders = glob($upload_dir . '/*', GLOB_ONLYDIR);

    // 🧹 Filtrare: doar cele care sunt ani și nu conțin nume excluse
    $year_folders = array_filter($folders, function($folder) use ($excluded) {
        $basename = basename($folder);

        // ✅ Trebuie să fie format de an (ex: 2025)
        if (!preg_match('/^\d{4}$/', $basename)) return false;

        // ❌ Excludem folderele non-media
        foreach ($excluded as $ex) {
            if (stripos($folder, $ex) !== false) return false;
        }

        return true;
    });

    return $year_folders; // 🔙 Returnează array cu căi complete
}