<?php
/**
 * Invite Service class for Kiyoh Reviews.
 *
 * @package Converzo\KiyohReviews
 */

namespace Converzo\KiyohReviews\Service;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Converzo\KiyohReviews\Admin\Settings;
use Converzo\KiyohReviews\Api\Client;
use Converzo\KiyohReviews\Logger;

/**
 * Handles sending review invitations.
 */
class Invite {

	/**
	 * Meta key for tracking if invite was sent.
	 *
	 * @var string
	 */
	const META_INVITE_SENT = '_converzo_kiyoh_invite_sent';

	/**
	 * Initialize invite hooks.
	 *
	 * @return void
	 */
	public static function init(): void {
		add_action( 'woocommerce_order_status_changed', array( static::class, 'on_order_status_changed' ), 10, 3 );
		add_filter( 'woocommerce_order_actions', array( static::class, 'add_order_actions' ) );
		add_action( 'woocommerce_order_action_converzo_kiyoh_send_invite', array( static::class, 'handle_manual_invite' ) );
	}

	/**
	 * Add Kiyoh actions to order actions dropdown.
	 *
	 * @param array $actions Existing order actions.
	 * @return array Modified order actions.
	 */
	public static function add_order_actions( array $actions ): array {
		if ( 'yes' === Settings::get_option( 'reviews_enabled' ) ) {
			$actions['converzo_kiyoh_send_invite'] = __( 'Send Kiyoh review invitation', 'kiyoh-reviews' );
		}
		return $actions;
	}

	/**
	 * Handle manual invite action from order actions dropdown.
	 *
	 * @param \WC_Order $order The order.
	 * @return void
	 */
	public static function handle_manual_invite( \WC_Order $order ): void {
		self::send_invite( $order->get_id(), true );
	}

	/**
	 * Send review invitation after order status change.
	 *
	 * @param int    $order_id    The order ID.
	 * @param string $prev_status The previous order status.
	 * @param string $status      The new order status.
	 * @return bool True if invitation was sent.
	 */
	public static function on_order_status_changed( int $order_id, string $prev_status, string $status ): bool {
		if ( 'yes' !== Settings::get_option( 'reviews_enabled' ) ) {
			return false;
		}

		if ( Settings::get_option( 'order_state_trigger' ) !== $status ) {
			return false;
		}

		// Check if invite was already sent to prevent duplicates.
		if ( self::was_invite_sent( $order_id ) ) {
			return false;
		}

		return self::send_invite( $order_id, false );
	}

	/**
	 * Check if an invite was already sent for this order.
	 *
	 * @param int $order_id The order ID.
	 * @return bool True if invite was already sent.
	 */
	public static function was_invite_sent( int $order_id ): bool {
		$order = wc_get_order( $order_id );
		if ( ! $order instanceof \WC_Order ) {
			return false;
		}
		return ! empty( $order->get_meta( self::META_INVITE_SENT ) );
	}

	/**
	 * Send a review invitation for an order.
	 *
	 * @param int  $order_id The order ID.
	 * @param bool $manual   Whether this is a manual send (bypasses duplicate check).
	 * @return bool True if invitation was sent.
	 */
	public static function send_invite( int $order_id, bool $manual = false ): bool {
		$order = wc_get_order( $order_id );

		if ( ! $order instanceof \WC_Order ) {
			return false;
		}

		$lang = self::determine_language( $order );

		$invite_data = array(
			'location_id'  => Client::get_location_id(),
			'invite_email' => $order->get_billing_email(),
			'delay'        => (int) Settings::get_option( 'delay', 0 ),
			'first_name'   => $order->get_billing_first_name(),
			'last_name'    => $order->get_billing_last_name(),
			'ref_code'     => $order->get_id(),
			'language'     => $lang,
		);

		if ( 'yes' === Settings::get_option( 'review_products' ) ) {
			$invite_data['product_code'] = self::get_order_product_ids( $order );
		}

		$result = Client::send_invite( $invite_data );

		if ( ! is_wp_error( $result ) ) {
			// Mark order as having received an invite.
			$order->update_meta_data( self::META_INVITE_SENT, current_time( 'mysql' ) );
			$order->save();

			if ( $manual ) {
				$order->add_order_note( __( 'Kiyoh review invitation sent manually.', 'kiyoh-reviews' ) );
			} else {
				$order->add_order_note( __( 'Kiyoh review invitation sent.', 'kiyoh-reviews' ) );
			}
			if ( Client::debug_enabled() ) {
				Logger::info( sprintf( 'Invite sent for order %d (manual: %s)', $order_id, $manual ? 'yes' : 'no' ) );
			}
		} else {
			$error_message = $result->get_error_message();
			if ( $manual ) {
				/* translators: %s: error message */
				$order->add_order_note( sprintf( __( 'Failed to send Kiyoh review invitation manually. Error: %s', 'kiyoh-reviews' ), $error_message ) );
			} else {
				/* translators: %s: error message */
				$order->add_order_note( sprintf( __( 'Failed to send Kiyoh review invitation. Error: %s', 'kiyoh-reviews' ), $error_message ) );
			}
			if ( Client::debug_enabled() ) {
				Logger::error( sprintf( 'Invite failed for order %d (manual: %s)', $order_id, $manual ? 'yes' : 'no' ) );
			}
		}

		return ! is_wp_error( $result );
	}

	/**
	 * Determine the email language based on settings and order.
	 *
	 * @param \WC_Order $order The order.
	 * @return string The language code.
	 */
	private static function determine_language( \WC_Order $order ): string {
		$lang = Settings::get_option( 'email_lang', 'en' );

		if ( 'yes' === Settings::get_option( 'email_lang_billing_country' ) ) {
			$langs           = Settings::map_country_lang();
			$billing_country = $order->get_billing_country();

			if ( isset( $langs[ $billing_country ] ) ) {
				$lang = $langs[ $billing_country ];
			}
		}

		return $lang;
	}

	/**
	 * Get product IDs from an order (max 5).
	 *
	 * @param \WC_Order $order The order.
	 * @return array<int> The product IDs.
	 */
	private static function get_order_product_ids( \WC_Order $order ): array {
		$products     = array();
		$order_items  = $order->get_items();
		$max_products = 5;
		$i            = 0;

		foreach ( $order_items as $order_item ) {
			if ( $i >= $max_products ) {
				break;
			}
			$products[] = $order_item->get_product_id();
			++$i;
		}

		return $products;
	}
}
